using System;
using System.Collections.Generic;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    public class ServiceTime : IDisposable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly CompositeDisposable _disposables = new();
        [NotNull]
        private readonly IReactiveProperty<bool> _isNetworkTimeProtected = new ReactiveProperty<bool>(false);
        private readonly List<IServerTimeProvider> _serverTimeProviders = new()
        {
            new MagifyServerTimeProvider(),
            new MicrosoftServerTimeProvider(),
        };

        private long? _localTimeMillis;
        private long? _remoteTimeMillis;
        private bool _isLoading;

        [NotNull]
        public IReadOnlyReactiveProperty<bool> IsNetworkTimeProtected => _isNetworkTimeProtected;
        public DateTime Now => DateTime.Now + NetworkTimeDelta();
        public DateTime UtcNow => DateTime.UtcNow + NetworkTimeDelta();

        public ServiceTime()
        {
            ApplicationHelper
                .EveryApplicationFocus()
                .Where(hasFocus => hasFocus)
                .Subscribe(_ =>
                {
                    MarkAsUnprotected();
                    SyncTime().Forget();
                })
                .AddTo(_disposables);
            MarkAsUnprotected();
        }

        void IDisposable.Dispose()
        {
            _disposables?.Dispose();
        }

        public void MarkAsUnprotected()
        {
            _isNetworkTimeProtected.Value = false;
            SyncTime().Forget();
        }

        private async UniTaskVoid SyncTime()
        {
            if (_isLoading)
            {
                return;
            }

            try
            {
                _isLoading = true;
                await UniTask.SwitchToMainThread(_disposables.GetOrCreateToken());
                var remoteTime = await _serverTimeProviders.TryGetServerTime(_disposables.GetOrCreateToken());
                if (remoteTime.HasValue)
                {
                    _localTimeMillis = DateTime.UtcNow.ToUnixMilliseconds();
                    _remoteTimeMillis = remoteTime.Value.ToUnixMilliseconds();
                    _logger.Log("Remote time updated: " +
                                $"UTCLocalTimeMS: {_localTimeMillis} ({DateTime.UtcNow}) | " +
                                $"UTCRemoteTimeMS: {_remoteTimeMillis} ({remoteTime.Value})");
                }
                else
                {
                    _remoteTimeMillis = _localTimeMillis = null;
                }

                _isNetworkTimeProtected.Value = IsMagifyHasTimeData();
            }
            finally
            {
                _isLoading = false;
            }
        }

        private TimeSpan NetworkTimeDelta()
        {
            return IsMagifyHasTimeData()
                ? TimeSpan.FromMilliseconds(_remoteTimeMillis!.Value - _localTimeMillis!.Value)
                : TimeSpan.Zero;
        }

        private bool IsMagifyHasTimeData()
        {
            return _localTimeMillis is not null && _localTimeMillis != 0
                && _remoteTimeMillis is not null && _remoteTimeMillis != 0;
        }
    }
}