using System;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    public interface IServicePrefs
    {
        IReadOnlyReactiveProperty<int> PurchasesCounter { get; }
        IReadOnlyReactiveProperty<int> RewardedVideoCounter { get; }
        IReadOnlyReactiveProperty<int> InterstitialVideoCounter { get; }
        IReadOnlyReactiveProperty<bool> NativeRateUsShown { get; }
        IReadOnlyReactiveProperty<bool> ConsentGivenEventSent { get; }
        IReadOnlyReactiveProperty<string> BoughtSubscriptionProduct { get; }
        IReadOnlyReactiveProperty<SubscriptionStatus> SubscriptionStatus { get; }
        IReadOnlyReactiveProperty<DateTime> SubscriptionExpireDate { get; }
        IReadOnlyReactiveProperty<DateTime> SubscriptionBoughtDate { get; }
        void Reset();
    }

    internal class ServicePrefs : IDisposable, IServicePrefs
    {
        private const string KeyPurchasesCounter = "purchases_counter";
        private const string KeyRewardedVideoCounter = "rewarded_video_counter";
        private const string KeyInterstitialVideoCounter = "interstitial_video_counter";
        private const string KeyNativeRateUsShown = "native_rate_us_shown";
        private const string KeyConsentGivenSent = "concent_given_sent";
        private const string KeyPauseTime = "pause_time";
        private const string KeySubscriptionProduct = "subscription_product";
        private const string KeySubscriptionStatus = "subscription_status";
        private const string KeySubscriptionExpireDate = "subscription_expire_date";
        private const string KeySubscriptionBoughtDate = "subscription_bought_date";

        [NotNull]
        private readonly CompositeDisposable _disposables = new CompositeDisposable();

        [NotNull]
        private BinaryStorage _storage;

        #region Prefs storage data

        [NotNull]
        public IReactiveProperty<int> PurchasesCounter { get; }
        IReadOnlyReactiveProperty<int> IServicePrefs.PurchasesCounter => PurchasesCounter;

        [NotNull]
        public IReactiveProperty<int> RewardedVideoCounter { get; }
        IReadOnlyReactiveProperty<int> IServicePrefs.RewardedVideoCounter => RewardedVideoCounter;

        [NotNull]
        public IReactiveProperty<int> InterstitialVideoCounter { get; }
        IReadOnlyReactiveProperty<int> IServicePrefs.InterstitialVideoCounter => InterstitialVideoCounter;

        [NotNull]
        public IReactiveProperty<bool> NativeRateUsShown { get; }
        IReadOnlyReactiveProperty<bool> IServicePrefs.NativeRateUsShown => NativeRateUsShown;

        [NotNull]
        public IReactiveProperty<bool> ConsentGivenEventSent { get; }
        IReadOnlyReactiveProperty<bool> IServicePrefs.ConsentGivenEventSent => ConsentGivenEventSent;

        [NotNull]
        public IReactiveProperty<SubscriptionStatus> SubscriptionStatus { get; }
        IReadOnlyReactiveProperty<SubscriptionStatus> IServicePrefs.SubscriptionStatus => SubscriptionStatus;

        [NotNull]
        public IReactiveProperty<string> BoughtSubscriptionProduct { get; }
        IReadOnlyReactiveProperty<string> IServicePrefs.BoughtSubscriptionProduct => BoughtSubscriptionProduct;

        [NotNull]
        public IReactiveProperty<DateTime> SubscriptionExpireDate { get; }
        IReadOnlyReactiveProperty<DateTime> IServicePrefs.SubscriptionExpireDate => SubscriptionExpireDate;

        [NotNull]
        public IReactiveProperty<DateTime> SubscriptionBoughtDate { get; }
        IReadOnlyReactiveProperty<DateTime> IServicePrefs.SubscriptionBoughtDate => SubscriptionBoughtDate;

        [NotNull]
        public IReactiveProperty<DateTime> PauseTime { get; }

        #endregion

        internal static ServicePrefs Create(string storagePath)
        {
            var storage = BinaryStorage
                .Construct(storagePath)
                .AddPrimitiveTypes()
                .ThrowOnLoadError()
                .SupportEnum<SubscriptionStatus>()
                .Build();

            var prefs = new ServicePrefs(storage);
            return prefs;
        }

        public void Reset()
        {
            PurchasesCounter.Value = default;
            RewardedVideoCounter.Value = default;
            InterstitialVideoCounter.Value = default;;
            NativeRateUsShown.Value = default;
            ConsentGivenEventSent.Value = default;
            PauseTime.Value = default;
            BoughtSubscriptionProduct.Value = default;
            SubscriptionStatus.Value = default;
            SubscriptionExpireDate.Value = default;
            SubscriptionBoughtDate.Value = default;
            _storage.ResetAll();
        }

        private ServicePrefs([NotNull] BinaryStorage storage)
        {
            _storage = storage;
            _storage.AddTo(_disposables);
            PurchasesCounter = _storage.GetReactiveProperty<int>(KeyPurchasesCounter);
            RewardedVideoCounter = _storage.GetReactiveProperty<int>(KeyRewardedVideoCounter);
            InterstitialVideoCounter = _storage.GetReactiveProperty<int>(KeyInterstitialVideoCounter);
            NativeRateUsShown = _storage.GetReactiveProperty<bool>(KeyNativeRateUsShown);
            ConsentGivenEventSent = _storage.GetReactiveProperty<bool>(KeyConsentGivenSent);
            PauseTime = _storage.GetReactiveProperty<DateTime>(KeyPauseTime);
            BoughtSubscriptionProduct = _storage.GetReactiveProperty<string>(KeySubscriptionProduct);
            SubscriptionStatus = _storage.GetReactiveProperty(KeySubscriptionStatus, Magify.SubscriptionStatus.Inactive);
            SubscriptionExpireDate = _storage.GetReactiveProperty<DateTime>(KeySubscriptionExpireDate);
            SubscriptionBoughtDate = _storage.GetReactiveProperty<DateTime>(KeySubscriptionBoughtDate);
        }

        void IDisposable.Dispose()
        {
            _disposables.Dispose();
        }
    }
}