﻿#if UNITY_PURCHASES
using System.Collections.Generic;
using System.Linq;
using Cysharp.Threading.Tasks;
using UnityEngine;
using UnityEngine.Purchasing;

namespace Magify
{
    internal class ProductsFetcher
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly IStoreController _controller;
        private readonly Dictionary<string, ProductDef> _currentProducts = new();
        private readonly HashSet<ProductDef> _upcomingProducts = new();

        private UniTaskCompletionSource<ProductObtainFailReason?> _currentOperation;

        public bool FetchInProcess => _currentOperation != null;

        public ProductsFetcher(IStoreController controller)
        {
            _controller = controller;
        }

        public async UniTask<ProductObtainFailReason?> FetchProductsAsync(uint operationId, IReadOnlyCollection<ProductDef> products)
        {
            try
            {
                return await PrepareWaitTask(operationId, products);
            }
            finally
            {
                _logger.Log($"[FETCH {operationId}] Products fetching finished");
            }
        }

        private async UniTask<ProductObtainFailReason?> PrepareWaitTask(uint operationId, IReadOnlyCollection<ProductDef> products)
        {
            _logger.Log($"[FETCH {operationId}] Preparing fetching operation for products: {string.Join(", ", products)}");

            if (_currentOperation != null && products.All(c => _currentProducts.ContainsKey(c.Id)))
            {
                _logger.Log($"[FETCH {operationId}] There is an active operation that fetching all products that needed for this operation - just wait it");
                return await _currentOperation.Task;
            }

            if (_currentOperation != null)
            {
                _logger.Log($"[FETCH {operationId}] There is active operation in progress - collect products for next fetch");
                _upcomingProducts.AddRange(products);
                await _currentOperation.Task;
            }

            if (_currentOperation != null)
            {
                return await _currentOperation.Task;
            }

            _logger.Log($"[FETCH {operationId}] There is no active operation - create new");
            try
            {
                _currentOperation = BeginNewFetchingOperation(operationId, _upcomingProducts.Count > 0 ? _upcomingProducts.ToArray() : products);
                _upcomingProducts.Clear();
                return await _currentOperation.Task;
            }
            finally
            {
                _currentProducts.Clear();
                _currentOperation = null;
            }
        }

        private UniTaskCompletionSource<ProductObtainFailReason?> BeginNewFetchingOperation(uint operationId, IReadOnlyCollection<ProductDef> products)
        {
            if (_currentProducts.Count != 0)
            {
                _logger.LogError($"[FETCH {operationId}] Collection {nameof(_currentProducts)} ({_currentProducts.Count}) should be empty at this point");
                _currentProducts.Clear();
            }

            products.ForEach(c => _currentProducts[c.Id] = c);

            var promise = new UniTaskCompletionSource<ProductObtainFailReason?>();
            var productsDefinitions = new HashSet<ProductDefinition>(products.Select(product =>
            {
                var payouts = product.Payout.Select(item => new PayoutDefinition(item.Type, item.Quantity));
                return new ProductDefinition(product.Id, product.Id, product.ToProductType(), true, payouts);
            }));

            _logger.Log($"[FETCH {operationId}] Call {nameof(IStoreController)}.{nameof(IStoreController.FetchAdditionalProducts)} to get info about products");
            _controller.FetchAdditionalProducts(productsDefinitions, success, failed);

            return promise;

            void success()
            {
                _logger.Log($"[FETCH {operationId}] Successfully got information about products");
                promise.TrySetResult(null);
            }

            void failed(InitializationFailureReason reason, string message)
            {
                _logger.LogWarning($"[FETCH {operationId}] Failed to get information about new products. Reason: {reason}; Message: {message}");
                promise.TrySetResult(reason.ToPurchaseFailReason());
            }
        }
    }
}

#endif