using System;
using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    public class SubscriptionService : BaseSubscriptionService, IDisposable
    {
        [NotNull]
        private readonly IReadOnlyDictionary<CampaignType, CampaignRequest> _currentCampaigns;

        [CanBeNull]
        private new IInAppStore InAppStore
        {
            get => base.InAppStore as IInAppStore;
            set => base.InAppStore = value;
        }

        internal SubscriptionService([NotNull] ServicePrefs prefs, [NotNull] ServiceTime time, [NotNull] INetworkStatusProvider network, [NotNull] IReadOnlyDictionary<CampaignType, CampaignRequest> currentCampaigns)
            : base(prefs, time, network)
        {
            _currentCampaigns = currentCampaigns;
        }

        internal virtual void SetPurchasingProvider([CanBeNull] IInAppStore inAppStore)
        {
            if (InAppStore != null)
            {
                InAppStore.OnPurchaseFinished -= PurchaseFinishedHandler;
                InAppStore.OnProductFetched -= ProductFetchedHandlerInternal;
            }

            InAppStore = inAppStore;
            if (InAppStore != null)
            {
                InAppStore.OnPurchaseFinished += PurchaseFinishedHandler;
                InAppStore.OnProductFetched += ProductFetchedHandlerInternal;
            }
        }

        void IDisposable.Dispose()
        {
            if (InAppStore != null)
            {
                InAppStore.OnPurchaseFinished -= PurchaseFinishedHandler;
                InAppStore.OnProductFetched -= ProductFetchedHandlerInternal;
            }
        }

        /// <remarks> <b>
        /// [Obsolete] You don't need to call this method in the full version of Magify. It's called automatically.
        /// </b> </remarks>
        public override void ProductFetchedHandler([NotNull] string productId)
        {
            Logger.LogWarning($"[Obsolete] You don't need to call {nameof(ProductFetchedHandler)} method in the full version of Magify. It's called automatically.");
            ProductFetchedHandlerInternal(productId);
        }

        private void ProductFetchedHandlerInternal([NotNull] string productId)
        {
            if (InAppStore != null)
            {
                InAppStore.OnProductFetched -= ProductFetchedHandlerInternal;
            }
            base.ProductFetchedHandler(productId);
        }

        internal override void RefreshSubscription([CanBeNull] string productId)
        {
            var product = MagifyManager.SubscriptionProducts.FirstOrDefault(c => c.Id == productId);
            if (product == null)
            {
                Logger.LogError($"Unexpected subscription product! {Prefs.BoughtSubscriptionProduct.Value} bought but is not in {nameof(MagifyManager.SubscriptionProducts)}");
                Prefs.BoughtSubscriptionProduct.Value = null;
                Prefs.SubscriptionExpireDate.Value = DateTime.UnixEpoch;
                Prefs.SubscriptionStatus.Value = SubscriptionStatus.Inactive;
                return;
            }

            base.RefreshSubscription(productId);
        }

        private void PurchaseFinishedHandler(string productId, PurchaseInfo purchaseInfo)
        {
            if (productId == null)
            {
                Logger.LogError($"{nameof(PurchaseFinishedHandler)} cannot handle null {nameof(productId)}.");
                return;
            }
            if (purchaseInfo == null)
            {
                Logger.LogError($"{nameof(PurchaseFinishedHandler)} cannot handle null {nameof(purchaseInfo)}");
                return;
            }
            var product = MagifyManager.SubscriptionProducts.FirstOrDefault(c => c.Id == productId);
            if (product == null) return;
            if (purchaseInfo.Receipt == null || purchaseInfo.SubscriptionInfo == null)
            {
                Logger.LogError($"Saving new subscription failed. The requested product ({productId}) not purchased yet (receipt is empty).");
                return;
            }
            Logger.Log($"Saving new subscription: \nProduct id - {product.Id}, \nReceipt: {purchaseInfo.Receipt}");
            var info = purchaseInfo.SubscriptionInfo;
            if (_currentCampaigns.Values.Any(request => request.Campaign.IsCampaignWithCreativeAndProduct(productId)))
            {
                Logger.Log($"Campaign for {product.Id} is active. Tracking subscription activation.");
                if (purchaseInfo.TrustedPurchaseRecord != null)
                {
                    MagifyManager.TrackTrustedSubscriptionActivation(purchaseInfo.TrustedPurchaseRecord);
                }
                else if (purchaseInfo.SkipVerification)
                {
                    MagifyManager.TrackSubscriptionActivationWithoutVerification(
                        info.IsFreeTrial,
                        product.Id,
                        purchaseInfo.Price,
                        purchaseInfo.Currency,
                        purchaseInfo.TransactionId,
                        purchaseInfo.OriginalTransactionId,
                        purchaseInfo.Store);
                }
                else
                {
                    MagifyManager.TrackSubscriptionActivation(
                        info.IsFreeTrial,
                        product.Id,
                        purchaseInfo.Price,
                        purchaseInfo.Currency,
                        purchaseInfo.TransactionId,
                        purchaseInfo.PurchaseToken,
                        purchaseInfo.OriginalTransactionId,
                        purchaseInfo.Receipt.Payload,
                        purchaseInfo.Store);
                }
            }
            else
            {
                Logger.Log($"Campaign for {product.Id} is not active. Tracking external subscription activation.");
                if (purchaseInfo.TrustedPurchaseRecord != null)
                {
                    MagifyManager.TrackExternalTrustedSubscriptionActivation(purchaseInfo.TrustedPurchaseRecord);
                }
                else if (purchaseInfo.SkipVerification)
                {
                    MagifyManager.TrackExternalSubscriptionActivationWithoutVerification(
                        info.IsFreeTrial,
                        productId,
                        purchaseInfo.Price,
                        purchaseInfo.Currency,
                        purchaseInfo.TransactionId,
                        purchaseInfo.OriginalTransactionId,
                        purchaseInfo.Store);
                }
                else
                {
                    MagifyManager.TrackExternalSubscriptionActivation(
                        info.IsFreeTrial,
                        product.Id,
                        purchaseInfo.Price,
                        purchaseInfo.Currency,
                        purchaseInfo.TransactionId,
                        purchaseInfo.PurchaseToken,
                        purchaseInfo.OriginalTransactionId,
                        purchaseInfo.Receipt.Payload,
                        purchaseInfo.Store);
                }
            }
            Prefs.BoughtSubscriptionProduct.Value = product.Id;
            Prefs.SubscriptionExpireDate.Value = info.ExpireDate;
            Prefs.SubscriptionBoughtDate.Value = DateTime.UtcNow;
            Prefs.SubscriptionStatus.Value = GetSubscriptionInfoStatus(info.IsFreeTrial, info.IsExpired(Time.UtcNow));
            FirstLoadPassed = true;
            SubscriptionRefreshedSubject.OnNext(Unit.Default);
            UpdateCancellationToken?.Cancel();
            UpdateCancellationToken?.Dispose();
            UpdateCancellationToken = null;
        }
    }
}