using System.Globalization;
using JetBrains.Annotations;

namespace Magify
{
    public class PurchaseInfo
    {
        public readonly string Price;
        public readonly string Currency;

        /// <summary>
        /// For Android it is orderId from product's receipt
        /// For iOS it is transactionID
        /// </summary>
        public readonly string TransactionId;

        /// <summary>
        /// For Android it is orderId from product's receipt
        /// For iOS it is appleOriginalTransactionID
        /// </summary>
        public readonly string OriginalTransactionId;

        /// <summary>
        /// Represents Android purchase token.
        /// Null for all other platforms.
        /// </summary>
        public readonly string PurchaseToken;

        public readonly PurchaseReceiptModel Receipt;

        public readonly SubscriptionInfo SubscriptionInfo;

        public readonly bool IsRestored;

        public readonly PurchaseStore Store;

        [CanBeNull]
        public TrustedPurchaseRecord TrustedPurchaseRecord { get; set; }

        /// <summary>
        /// Marks this purchase as 'does not require verification through Magify services'. In that case Magify isn't going to send verification requests to the server.
        /// </summary>
        public bool SkipVerification { get; set; }

        public PurchaseInfo(
                string price,
                string currency,
                string transactionId,
                string originalTransactionId,
                string purchaseToken,
                PurchaseReceiptModel receipt,
                PurchaseStore store,
                SubscriptionInfo subscriptionInfo = default,
                TrustedPurchaseRecord trustedPurchaseRecord = default,
                bool skipVerification = false,
                bool isRestored = false)
        {
            Price = price;
            Currency = currency;
            TransactionId = transactionId;
            OriginalTransactionId = originalTransactionId;
            PurchaseToken = purchaseToken;
            Receipt = receipt;
            SubscriptionInfo = subscriptionInfo;
            IsRestored = isRestored;
            Store = store;
            TrustedPurchaseRecord = trustedPurchaseRecord;
            SkipVerification = skipVerification;
        }

#if UNITY_PURCHASES
        public PurchaseInfo(UnityEngine.Purchasing.Product product, SubscriptionInfo subscriptionInfo = default, bool skipVerification = false)
        {
            Price = product.metadata.localizedPrice.ToString(CultureInfo.InvariantCulture);
            Currency = product.metadata.isoCurrencyCode;
            TransactionId = product.GetTransactionID();
            OriginalTransactionId = product.GetOriginalTransactionID();
            PurchaseToken = product.GetPurchaseToken();
            Receipt = new PurchaseReceiptModel(product.receipt);
            SubscriptionInfo = subscriptionInfo;
            SkipVerification = skipVerification;
            Store = PurchaseStore.Native;
        }
#endif

        public static PurchaseInfo ForRestored(PurchaseStore store)
        {
            return new PurchaseInfo(
                price: default,
                currency: default,
                transactionId: default,
                originalTransactionId: default,
                purchaseToken: default,
                receipt: default,
                store,
                isRestored: true);
        }
    }
}