#if MAGIFY_ADVANCED
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using Magify.Rx;
using JetBrains.Annotations;
using UnityEngine;
using UnityEngine.Pool;

namespace Magify
{
    public partial class MagifyService
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LogScope);

        public const string LogScope = "Service";

        private readonly ServicePrefs _prefs;
        private readonly MagifySettings _settings;
        private readonly Dictionary<CampaignType, CampaignRequest> _currentCampaigns = new();
#if MAGIFY_3_OR_NEWER
        private readonly IReadOnlyReactiveProperty<int> _sessionCounter;
#else
        private readonly IReactiveProperty<int> _sessionCounter;
#endif
        private readonly IReactiveProperty<bool> _sessionStarted;
        [NotNull]
        private readonly MagifyWorkersManager _workersManager;
        [NotNull]
        private readonly Subject<Unit> _onStoredAppFeaturesLoadingRequested = new();
        [NotNull]
        private readonly IReactiveProperty<bool> _loadStoredAppFeaturesOnNetworkRestore = new ReactiveProperty<bool>();

        private CompositeDisposable _disposables = new();
        private CancellationTokenSource _terminateCancellationToken = new();
        private int _configSyncsHandled = 0;

        public IObservable<Unit> OnConfigSynced { get; }
        [NotNull]
        internal IObservable<Unit> OnStoredAppFeaturesLoadingRequested => _onStoredAppFeaturesLoadingRequested;

        [NotNull]
        public MagifySettings Settings => _settings;

        [NotNull]
        public IServicePrefs Prefs => _prefs;

        [NotNull]
        public INetworkStatusProvider Network { get; private set; }

        [NotNull]
        public ServiceTime Time { get; }

        [NotNull]
        public CampaignsProvider Campaigns { get; }

        [NotNull]
        public PopupsHandler Popups { get; }

        [NotNull]
        public LimitedTimeOfferProvider Offers { get; }

        [NotNull]
        public AnalyticsService Analytics { get; }

        [NotNull]
        public ProductsObtainer Obtainer { get; }

        [NotNull]
        public FeaturesProvider Features { get; }

        [NotNull]
        public MagifyPresenter Presenter => MagifyManager.Presenter;

        [NotNull]
        public AppStateService AppState { get; }

        [NotNull]
        public AdvertiserService Advertiser { get; }

        [CanBeNull]
        public IAppNavigator AppNavigator { get; private set; }

        [NotNull]
        public PurchaserService Purchaser { get; private set; }

        [NotNull]
        public SubscriptionService Subscription { get; private set; }

        [NotNull]
        public UserAssignment UserAssignment { get; } = new();

        [NotNull]
        internal AdPreloader AdPreloader { get; }

        [NotNull]
        internal ILocalizer Localizer { get; }

        [NotNull]
        internal DelayedCampaignRegistry DelayedCampaignRegistry { get; } = new();

        [NotNull]
        public string ClientId => MagifyManager.ClientId;

        /// <summary>
        /// Indicates whether the <see cref="RequestSessionStartAsync"/> method has already been called.
        /// </summary>
        public IReadOnlyReactiveProperty<bool> SessionStarted => _sessionStarted;

        /// <summary>
        /// Stores the current session number
        /// </summary>
        public IReadOnlyReactiveProperty<int> Session => _sessionCounter;

        /// <summary>
        /// Represents the current interval between sessions, that will be used to determine
        /// the new session start when player returns to the game after a long pause (when app is in background).
        /// </summary>
        public IReadOnlyReactiveProperty<TimeSpan> SessionsInterval => MagifyManager.SessionsInterval;

        /// <inheritdoc cref="MagifyManager.FirstInstalledVersion"/>
        public string FirstInstalledVersion => MagifyManager.FirstInstalledVersion;

        [Obsolete("You should no longer use this property. " +
                  "It is outdated and will be removed around the second half of 2026. " +
                  "Consider using " + nameof(MagifyServiceArgs.IsSandbox))]
        public IReactiveProperty<Environment> Environment { get; }

        [NotNull]
        public IReactiveProperty<ConfigScope> RemoteConfigScopes => MagifyManager.RemoteConfigScopes;

        public IReactiveProperty<AuthorizationStatus> AuthorizationStatus { get; }

        /// <summary>
        /// Switches the preloading of stored app features. SDK might load them in background on different triggers:
        /// <ul>
        ///     <li>start of new session</li>
        ///     <li>changing of network reachability</li>
        ///     <li><i>some other triggers...</i></li>
        /// </ul>
        /// </summary>
        public bool StoredAppFeaturesPreloadingEnabled { get; set; }

        /// <summary>
        /// This property determines whether logging is turned on or off in the SKD
        /// </summary>
        public IReactiveProperty<bool> IsLoggingEnabled { get; }

        /// <summary>
        /// Responsible for storing the referrer identifier, which can be obtained from various sources like Facebook Deferred Deep Links
        /// </summary>
        public IReactiveProperty<string> ReferrerId { get; }

        /// <inheritdoc cref="MagifyManager.ExternalPurchaseVerificationHandler"/>
        [NotNull]
        public IReactiveProperty<IPurchaseVerificationHandler> ExternalPurchaseVerificationHandler => Purchaser.ExternalPurchaseVerificationHandler;

        /// <inheritdoc cref="MagifyManager.VerificationRetryInterval"/>
        [NotNull]
        public IReactiveProperty<float> VerificationRetryInterval => Purchaser.VerificationRetryInterval;

        internal MagifyService([NotNull] MagifySettings settings, [CanBeNull] MagifyServiceArgs args, [NotNull] ServicePrefs prefs)
        {
            _settings = settings;
            _prefs = prefs.AddTo(_disposables);

            // Create internal services
            Time = new ServiceTime().AddTo(_disposables);
            Campaigns = new CampaignsProvider();
            Network = new NetworkStatus().AddTo(_disposables);
            Offers = new LimitedTimeOfferProvider().AddTo(_disposables);
            Localizer = new EmbeddedLocalizer();
            Analytics = new AnalyticsService();
            Features = new FeaturesProvider(_settings).AddTo(_disposables);
            Popups = new PopupsHandler(_settings, Localizer, Network, new ResourcesPopupsProvider(PackageInfo.ResourcesFolder), args?.PopupsProvider);
            Obtainer = new ProductsObtainer(Popups);
            Subscription = new SubscriptionService(_prefs, Time, Network, _currentCampaigns).AddTo(_disposables);
            Purchaser = new PurchaserService(settings, _prefs, Obtainer).AddTo(_disposables);
            AdPreloader = new AdPreloader().AddTo(_disposables);
            Advertiser = new AdvertiserService(Settings, _prefs, Campaigns, Obtainer, AdPreloader, Network).AddTo(_disposables);
            AppState = new AppStateService().AddTo(_disposables);

            // Setup internal services
            Campaigns.RegisterEmbeddedCampaignHandler(new ReadyInterstitialCampaignHandler(Advertiser, Popups));
            Campaigns.RegisterEmbeddedCampaignHandler(new NoSplashInterstitialCampaignHandler(Advertiser, Network, DelayedCampaignRegistry));
            Campaigns.RegisterEmbeddedCampaignHandler(new PopupSplashInterstitialCampaignHandler(Advertiser, Network, DelayedCampaignRegistry, Popups));
            Campaigns.RegisterEmbeddedCampaignHandler(new NoUiCreativeCampaignHandler(Obtainer));
            Campaigns.RegisterEmbeddedCampaignHandler(new ImageCreativeCampaignHandler(Obtainer, Popups));
            Campaigns.RegisterEmbeddedCampaignHandler(new CustomCreativeCampaignHandler(Obtainer, Popups, Offers));
            Campaigns.RegisterEmbeddedCampaignHandler(new RateReviewCampaignHandler(Settings, _prefs, Popups));

            Obtainer.RegisterEmbeddedProductObtainer(new InAppProductObtainer(Purchaser));
            Obtainer.RegisterEmbeddedProductObtainer(new BonusProductObtainer());
            Obtainer.RegisterEmbeddedProductObtainer(new InfoProductObtainer());
            Obtainer.RegisterEmbeddedProductObtainer(new CrossPromoProductObtainer());
            Obtainer.RegisterEmbeddedProductObtainer(new ExternalLinkProductObtainer());
            Obtainer.RegisterEmbeddedProductObtainer(new RewardProductObtainer(Advertiser));

            StoredAppFeaturesPreloadingEnabled = args?.StoredAppFeaturesPreloadingEnabled ?? false;
            Purchaser.ExternalPurchaseVerificationHandler.Value = args?.PurchaseVerificationHandler;

            // Setup internal events
            OnConfigSynced = Observable.FromEvent(t => MagifyManager.OnConfigLoaded += t, t => MagifyManager.OnConfigLoaded -= t);

            // Create public properties
#if MAGIFY_3_OR_NEWER
            _sessionCounter = MagifyManager.SessionNumber;
#else
            _sessionCounter = new ReactiveProperty<int>(MagifyManager.SessionNumber);
            Observable.FromEvent<int>(t => MagifyManager.OnSessionChanged += t, t => MagifyManager.OnSessionChanged -= t)
                .Subscribe(session => _sessionCounter.Value = session)
                .AddTo(_disposables);
#endif
            _sessionStarted = new ReactiveProperty<bool>(false);
            Environment = new ReactiveProperty<Environment>(MagifyManager.Environment);
            AuthorizationStatus = new ReactiveProperty<AuthorizationStatus>(MagifyManager.AuthorizationStatus);
            IsLoggingEnabled = new ReactiveProperty<bool>(MagifyManager.Logging.IsLoggingEnabled);
            ReferrerId = new ReactiveProperty<string>(MagifyManager.ReferrerId);

            // Subscribe on internal events
            Network.Reachability
                .SkipLatestValueOnSubscribe()
                .Where(c => c is NetworkState.Reachable)
                .Subscribe(NetworkReachabilityChangedHandler)
                .AddTo(_disposables);
            _prefs.SubscriptionStatus
                .SkipLatestValueOnSubscribe()
                .Where(c => MagifyManager.SubscriptionStatus != c)
                .Subscribe(SubscriptionStatusChangedHandler)
                .AddTo(_disposables);
            Session
                .Where(_ => StoredAppFeaturesPreloadingEnabled)
                .Subscribe(_ => _workersManager.DoJob<StoredAppFeatureLoadingWorkerJob>())
                .AddTo(_disposables);
            OnConfigSynced
                .Subscribe(ConfigSyncedHandler)
                .AddTo(_disposables);
            Environment
                .SkipLatestValueOnSubscribe()
                .Where(c => MagifyManager.Environment != c)
                .Subscribe(EnvironmentChangedHandler)
                .AddTo(_disposables);
            AuthorizationStatus
                .SkipLatestValueOnSubscribe()
                .Where(c => MagifyManager.AuthorizationStatus != c)
                .Subscribe(AuthorizationStatusChangedHandler)
                .AddTo(_disposables);
            IsLoggingEnabled
                .SkipLatestValueOnSubscribe()
                .Subscribe(c => MagifyManager.Logging.IsLoggingEnabled = c)
                .AddTo(_disposables);
            ReferrerId
                .SkipLatestValueOnSubscribe()
                .Subscribe(c => MagifyManager.ReferrerId = c)
                .AddTo(_disposables);
            DelayedCampaignRegistry
                .OnDelayedCampaignFinished
                .Subscribe(c => CampaignPostprocessor(c.Request.Event, c.Request.Campaign, c.Result, CancellationToken.None).Forget())
                .AddTo(_disposables);
            Offers.OnFinished
                .Subscribe(OnLtoFinished)
                .AddTo(_disposables);
            ApplicationHelper.EveryApplicationFocus()
                .Where(hasFocus => !hasFocus)
                .Subscribe(_ => _prefs.PauseTime.Value = DateTime.UtcNow)
                .AddTo(_disposables);
            ApplicationHelper.EveryApplicationFocus()
                .Where(hasFocus => hasFocus)
                .ObserveOnMainThread(_disposables.GetOrCreateToken())
                .Subscribe(_ => FocusGainedHandler())
                .AddTo(_disposables);
            ApplicationHelper.EveryApplicationFocus()
                .ObserveOnMainThread(_disposables.GetOrCreateToken())
                .Subscribe(hasFocus => Subscription.HandleAppFocus(hasFocus))
                .AddTo(_disposables);

            var workers = new IMagifyWorker[]
            {
                new HandlePauseTimeOnFocusGainedWorker(_prefs, _settings, Time),
                new StoredAppFeatureLoadingWorker(Features, Network, _loadStoredAppFeaturesOnNetworkRestore, _onStoredAppFeaturesLoadingRequested),
            };
            _workersManager = new MagifyWorkersManager(workers).AddTo(_disposables);
        }

        #region External handlers

        /// <inheritdoc cref="AdvertiserService.SetAdsMediator(Magify.IAdsMediator)"/>
        [NotNull]
        public MagifyService SetAdsMediator([CanBeNull] IAdsMediator mediator)
        {
            Advertiser.SetAdsMediator(mediator);
            return this;
        }

        /// <summary>
        /// Set the purchasing provider for the service. This method should be called right after the service initialization.
        /// </summary>
        /// <param name="inAppStore">Implementation that will be used to track in-app products and subscriptions purchases</param>
        [NotNull]
        public MagifyService SetPurchasingProvider([CanBeNull] IInAppStore inAppStore)
        {
            Subscription.SetPurchasingProvider(inAppStore);
            Purchaser.SetPurchasingProvider(inAppStore);
            return this;
        }

        /// <summary>
        /// Set the app navigator for the service. It allows to jump to the internal links and enables embedded product obtainer for internal link products.
        /// </summary>
        [NotNull]
        public MagifyService SetAppNavigator([CanBeNull] IAppNavigator appNavigator)
        {
            if (AppNavigator != null)
            {
                Obtainer.RemoveEmbeddedObtainer<InternalLinkProductObtainer>();
            }
            AppNavigator = appNavigator;
            if (AppNavigator != null)
            {
                Obtainer.RegisterEmbeddedProductObtainer(new InternalLinkProductObtainer(AppNavigator));
            }
            return this;
        }

#if UNITY_PURCHASES
        /// <summary>
        /// Use embedded Unity Purchasing implementation. See: <see cref="UnityPurchases"/>
        /// </summary>
        [NotNull]
        public MagifyService UseEmbeddedUnityPurchasing()
        {
            return SetPurchasingProvider(new UnityPurchases(Network, Localizer));
        }
#endif

#if MAX_ADS_MEDIATOR
        /// <summary>
        /// Use embedded Max Mediator implementation. <br/>
        /// See: <see cref="MaxMediator"/>  (it's already implements <see cref="IAdsMediator"/>, so you can use it to show ads)
        /// </summary>
        [NotNull]
        public MagifyService UseEmbeddedMaxMediator()
        {
            _logger.Log("Use embedded MaxMediator");
            return SetAdsMediator(new MaxMediator(Settings));
        }

        /// <summary>
        /// It will configure the embedded <see cref="MaxMediator"/> implementation before tracking and showing of ads.
        /// </summary>
        [NotNull]
        public MagifyService InitializeEmbeddedMaxMediator()
        {
            _logger.Log($"Initialize embedded {nameof(MaxMediator)}");
            ((MaxMediator)Advertiser.Mediator!).Initialize();
            return this;
        }
#endif

#if LEVELPLAY_MEDIATOR
        /// <summary>
        /// Use embedded Level Play Mediator implementation. <br/>
        /// See: <see cref="LevelPlayMediator"/> (it's already implements <see cref="IAdsMediator"/>, so you can use it to show ads)
        /// </summary>
        [NotNull]
        public MagifyService UseEmbeddedLevelPlayMediator()
        {
            _logger.Log("Use embedded LevelPlay");
            return SetAdsMediator(new LevelPlayMediator(Settings));
        }

        /// <summary>
        /// It will configure the embedded <see cref="LevelPlayMediator"/> implementation before tracking and showing of ads.
        /// </summary>
        /// <param name="testSuite">Allows enabling test suite (debug panel) of Level play mediator</param>
        [NotNull]
        public MagifyService InitializeEmbeddedLevelPlayMediator(bool testSuite = false)
        {
            _logger.Log($"Initialize embedded {nameof(LevelPlayMediator)}");
            ((LevelPlayMediator)Advertiser.Mediator!).Initialize(testSuite);
            return this;
        }
#endif

        #endregion

        #region Public API

        public void SetAttStatus(bool authorized)
        {
            MagifyManager.SetAttStatus(authorized);
        }

        public void SetMediaSource(string networkName = null, string campaignName = null, string adGroup = null)
        {
            MagifyManager.SetMediaSource(networkName, campaignName, adGroup);
        }

        public void SetUserEmail(string email)
        {
            MagifyManager.SetUserEmail(email);
        }

        public void Sync()
        {
            MagifyManager.Sync();
        }

        /// <summary>
        /// Tweak environment and isSandbox flag.
        /// The SDK would be reset if the environment and isSandbox flag values were not saved.
        /// </summary>
        /// <inheritdoc cref="MagifyManager.ThrowIfMagifyIsNotReady"/>
        /// <remarks><inheritdoc cref="MagifyDocs.TestOnly" path="/remarks"/></remarks>
        public void TweakEnvironment(Environment? environment, bool? isSandbox)
        {
            MagifyManager.TweakEnvironment(environment, isSandbox);
        }

        public void Reset(bool clearNativeStorage = true, bool clearCloudStorage = true)
        {
            MagifyManager.Reset(clearNativeStorage, clearCloudStorage);
        }

        private void ShutDown()
        {
            _terminateCancellationToken?.Cancel();
            _terminateCancellationToken?.Dispose();
            _disposables?.Dispose();
            _terminateCancellationToken = null;
            _disposables = null;
            MagifyManager.ShutDown();
        }

        #endregion

        #region RequestCampaignAsync

        public async UniTask<CampaignResult> RequestSessionStartAsync(CancellationToken cancellationToken = default)
        {
            _logger.Log($"{nameof(RequestSessionStartAsync)} called. SessionStarted: {SessionStarted.Value}");
            if (SessionStarted.Value)
            {
                _logger.LogWarning($"You don't need to call {nameof(RequestSessionStartAsync)} more than ones. Skip it");
                return CampaignResult.None;
            }

            string @event;
            if (!_prefs.ConsentGivenEventSent.Value)
            {
                _logger.Log($"This is first session - request campaign for '{_settings.ConsentGivenEvent}' instead of '{_settings.SessionStartedEvent}'");
                _prefs.ConsentGivenEventSent.Value = true;
                @event = _settings.ConsentGivenEvent;
            }
            else
            {
                _logger.Log($"This isn't first session - request campaign for '{_settings.SessionStartedEvent}'");
                @event = _settings.SessionStartedEvent;
                var seconds = 0;
                if (_prefs.PauseTime.Value != default)
                {
                    seconds = (int)(DateTime.UtcNow - _prefs.PauseTime.Value).TotalSeconds;
                    if (seconds < 0)
                    {
                        _logger.LogError($"Trying to send {_settings.SessionStartedEvent} but away time less than 0. Probably time change detected. Set it to 0");
                        seconds = 0;
                    }
                    _prefs.PauseTime.Value = default;
                }
                Campaigns.SetParam(@event, _settings.AwayTimeParam, seconds);
            }
            _sessionStarted.Value = true;
            if (cancellationToken == default)
            {
                cancellationToken = CancellationToken.None;
            }
            var result = await RequestCampaignAsync(@event, "", null, cancellationToken);
            Campaigns.RemoveParam(@event, _settings.AwayTimeParam);
            return result;
        }

        public UniTask<CampaignResult> RequestCampaignAsync(string @event, [CanBeNull] Dictionary<string, object> @params = null)
        {
            return RequestCampaignAsync(@event, "", @params, CancellationToken.None);
        }

        public UniTask<CampaignResult> RequestCampaignAsync(string @event, CancellationToken cancellationToken)
        {
            return RequestCampaignAsync(@event, "", null, cancellationToken);
        }

        public UniTask<CampaignResult> RequestCampaignAsync(string @event, Dictionary<string, object> @params, CancellationToken cancellationToken)
        {
            return RequestCampaignAsync(@event, "", @params, cancellationToken);
        }

        public UniTask<CampaignResult> RequestCampaignAsync(string @event, string payoutType, Dictionary<string, object> @params = null)
        {
            return RequestCampaignAsync(@event, payoutType, @params, CancellationToken.None);
        }

        public async UniTask<CampaignResult> RequestCampaignAsync(string @event, string payoutType, [CanBeNull] Dictionary<string, object> @params, CancellationToken cancellationToken)
        {
            ThrowIfServiceTerminated();

            if (string.IsNullOrEmpty(@event))
            {
                _logger.Log("Skip event because it is empty");
                return CampaignResult.None;
            }

            _logger.Log(string.IsNullOrEmpty(payoutType), $"{nameof(RequestCampaignAsync)}. {nameof(@event)}={@event}");
            _logger.Log(!string.IsNullOrEmpty(payoutType), $"{nameof(RequestCampaignAsync)}. {nameof(@event)}={@event}; {nameof(payoutType)}={payoutType}");
            if (!SessionStarted.Value)
            {
                _logger.LogError($"Running campaign before session start is not allowed! Make sure to call {nameof(RequestSessionStartAsync)} before any other campaigns");
                return CampaignResult.None;
            }

            if (DelayedCampaignRegistry.Any)
            {
                var silentCampaign = Campaigns.GetCampaign(@event, @params);
                if (silentCampaign == null)
                {
                    _logger.Log("Campaign not found");
                    return CampaignResult.None;
                }
                if (DelayedCampaignRegistry.GetFirstCampaignByType(silentCampaign.Type) != null)
                {
                    _logger.Log("Campaign with same type was delayed, skipping current request");
                    return CampaignResult.None;
                }
            }

            var campaign = Campaigns.GetCampaignToHandle(@event, @params);
            if (campaign == null)
            {
                _logger.Log("Campaign not found");
                return CampaignResult.None;
            }
            if (Advertiser.Mediator == null && (campaign is InterstitialCampaign || campaign is ICampaignWithProducts cwp1 && cwp1.Products.Any(c => c is RewardProduct)))
            {
                _logger.LogError("Ads Mediator not found");
#if MAGIFY_3_OR_NEWER
                MagifyManager.TrackImpressionFailFor(campaign.Name, "Ads Mediator not found");
#else
                MagifyManager.TrackImpressionFailFor(campaign.Type, "Ads Mediator not found");
#endif
                return CampaignResult.None;
            }
            if (Purchaser.InAppStore == null && campaign is ICampaignWithProducts cwp2 && cwp2.Products.Any(c => c is InAppProduct or SubscriptionProduct))
            {
                _logger.LogError("InAps purchaser not found");
#if MAGIFY_3_OR_NEWER
                MagifyManager.TrackImpressionFailFor(campaign.Name, "InAps purchaser not found");
#else
                MagifyManager.TrackImpressionFailFor(campaign.Type, "InAps purchaser not found");
#endif
                return CampaignResult.None;
            }
            if (campaign is ICampaignWithCreative { Creative: null })
            {
                _logger.LogError("Unknown creative");
#if MAGIFY_3_OR_NEWER
                MagifyManager.TrackImpressionFailFor(campaign.Name, "Unknown creative");
#else
                MagifyManager.TrackImpressionFailFor(campaign.Type, "Unknown creative");
#endif
                return CampaignResult.None;
            }
            if (campaign is ICampaignWithProducts { Products: { Count: 0 } })
            {
                _logger.LogError("No products in campaign");
#if MAGIFY_3_OR_NEWER
                MagifyManager.TrackImpressionFailFor(campaign.Name, "No products in campaign");
#else
                MagifyManager.TrackImpressionFailFor(campaign.Type, "No products in campaign");
#endif
                return CampaignResult.None;
            }
            if (!string.IsNullOrEmpty(payoutType) && campaign is ICampaignWithProducts campaignWithProducts && campaignWithProducts.Products.FirstOrDefault(c => c.Payout.Any(k => k.Type == payoutType)) == null)
            {
                _logger.LogError($"Product with item {payoutType} not found");
#if MAGIFY_3_OR_NEWER
                MagifyManager.TrackImpressionFailFor(campaign.Name, $"Product with item {payoutType} not found");
#else
                MagifyManager.TrackImpressionFailFor(campaign.Type, $"Product with item {payoutType} not found");
#endif
                return CampaignResult.None;
            }
            if (campaign is InterstitialCampaign { Screen: null })
            {
                _logger.LogError("Unknown interstitial screen");
#if MAGIFY_3_OR_NEWER
                MagifyManager.TrackImpressionFailFor(campaign.Name, "Unknown interstitial screen");
#else
                MagifyManager.TrackImpressionFailFor(campaign.Type, "Unknown interstitial screen");
#endif
                return CampaignResult.None;
            }

            var result = await RequestCampaignAsync(campaign, @event, payoutType, @params, cancellationToken);
            return await CampaignPostprocessor(@event, campaign, result, cancellationToken);
        }

        private async UniTask<CampaignResult> RequestCampaignAsync(ICampaign campaign, string @event, string payoutType, [CanBeNull] Dictionary<string, object> @params, CancellationToken cancellationToken)
        {
            // use it to test cancellations
            // cancellationToken = new CancellationTokenSource(TimeSpan.FromSeconds(1.5)).Token;

            var disposables = GenericPool<CompositeDisposable>.Get();
            var paramsDictionary = DictionaryPool<string, object>.Get();
            paramsDictionary!.Copy(@params);

            var request = new CampaignRequest(@event, campaign, disposables)
            {
                Params = paramsDictionary,
                PayoutType = payoutType
            };

            var result = CampaignResult.None;
            try
            {
                var handler = Campaigns.GetHandlerFor(request);
                if (handler == null)
                {
                    _logger.LogError($"Handler for campaign not found: {request.Campaign.Name}");
#if MAGIFY_3_OR_NEWER
                    MagifyManager.TrackImpressionFailFor(campaign.Name, "Handler for campaign not found");
#else
                    MagifyManager.TrackImpressionFailFor(campaign.Type, "Handler for campaign not found");
#endif
                    return result = CampaignResult.None;
                }
                if (cancellationToken == CancellationToken.None)
                {
                    cancellationToken = _terminateCancellationToken.Token;
                }
                else
                {
                    var linkedCancellationTokenSource = CancellationTokenSource
                        .CreateLinkedTokenSource(cancellationToken, _terminateCancellationToken.Token)
                        .AddTo(disposables);
                    cancellationToken = linkedCancellationTokenSource.Token;
                }
                _logger.Log($"'{handler.GetType().Name}' was chosen as handler for this campaign");
                _currentCampaigns[campaign.Type] = request;
                result = await handler.HandleCampaignAsync(request, cancellationToken);
                _logger.Log($"'{handler.GetType().Name}' finished his work with result {result} ");
                return result;
            }
            catch (OperationCanceledException)
            {
                return result = CampaignResult.Aborted;
            }
            catch (Exception e)
            {
                Debug.LogException(e);
                request.TrackShowFailed($"Failed with exception {e.GetType().Name}: {e.Message}");
                return result = CampaignResult.Failed;
            }
            finally
            {
                _currentCampaigns.Remove(campaign.Type);
                if (result != CampaignResult.Delayed && !request.Disposables.IsDisposed)
                {
                    request.Disposables.Clear();
                    GenericPool<CompositeDisposable>.Release(request.Disposables);
                }
            }
        }

        private async UniTask<CampaignResult> CampaignPostprocessor(string @event, ICampaign campaign, CampaignResult result, CancellationToken token)
        {
            if (campaign == null || token.IsCancellationRequested) return result;

            _logger.Log($"Postprocessor for campaign {campaign.Name} ({campaign.Type}). {nameof(@event)}={@event}; {nameof(result)}={result}");
            Dictionary<string, object> @params = null;

            Dictionary<string, object> getParamsLazy()
            {
                @params ??= new Dictionary<string, object>
                {
                    { _settings.SourceParam, @event }
                };
                return @params;
            }

            switch (campaign)
            {
                case InterstitialCampaign when result is CampaignResult.Applied or CampaignResult.Declined:
                    // _analytics.LogEvent(new InterstitialAdEvent(@event));
                    await RequestCampaignAsync(_settings.InterstitialShowedEvent, getParamsLazy(), token);
                    break;
                case InterstitialCampaign when result == CampaignResult.Failed:
                    // _analytics.LogEvent(new InterstitialFailedEvent(@event));
                    await RequestCampaignAsync(_settings.InterstitialFailedEvent, getParamsLazy(), token);
                    break;
                case RewardedVideoCampaign when result is CampaignResult.Applied:
                    await RequestCampaignAsync(_settings.RewardedVideoShowedEvent, getParamsLazy(), token);
                    break;
                case RewardedVideoCampaign when result == CampaignResult.Failed:
                    await RequestCampaignAsync(_settings.RewardedVideoFailedEvent, getParamsLazy(), token);
                    break;
                case ICampaignWithCreative creative when result == CampaignResult.Declined && creative.Creative is not NoUiCreative:
                    // _analytics.LogEvent(new CampaignClosedEvent(@event, campaign.Name));
                    getParamsLazy().Add(_settings.CampaignParam, campaign.Name);
                    getParamsLazy().Add(_settings.CampaignTypeParam, campaign.Type);
                    await RequestCampaignAsync(_settings.CampaignClosedEvent, getParamsLazy(), token);
                    break;
            }

            return result;
        }

        #endregion

        #region Internal logic

        private void OnLtoFinished(LtoInfo offer)
        {
            _logger.Log($"{nameof(OnLtoFinished)} has been called for {offer.Spot} ({offer.CampaignName})");
            var @params = new Dictionary<string, object>
            {
                { _settings.CampaignParam, offer.CampaignName }
            };

            _logger.Log($"Ask campaign for event {_settings.OfferFinishedEvent} to continue LTO chain");
            var campaign = Campaigns.GetCampaignToHandle(_settings.OfferFinishedEvent, @params);
            if (campaign != null)
            {
                _logger.LogError($"Campaign {campaign.Name} will be ignored because it was triggered by {_settings.OfferFinishedEvent}");
            }
        }

        private void ThrowIfServiceTerminated()
        {
            if (_disposables.IsDisposed)
            {
                throw new MagifyTerminatedException();
            }
        }

        private void EnvironmentChangedHandler(Environment environment)
        {
            _logger.Log($"Change Environment to {environment}");
            TweakEnvironment(environment, null);
            MagifyManager.Sync();
        }

        private void SubscriptionStatusChangedHandler(SubscriptionStatus subscription)
        {
            _logger.Log($"Change SubscriptionStatus to {subscription}");
            MagifyManager.SubscriptionStatus = subscription;
            MagifyManager.Sync();
        }

        private void AuthorizationStatusChangedHandler(AuthorizationStatus authorizationStatus)
        {
            _logger.Log($"Change AuthorizationStatus to {authorizationStatus}");
            MagifyManager.AuthorizationStatus = authorizationStatus;
            MagifyManager.Sync();
        }

        private void NetworkReachabilityChangedHandler(NetworkState state)
        {
            _logger.Log($"Network reachability changed to {state} - call {nameof(MagifyManager)}.{nameof(MagifyManager.Sync)}");
            MagifyManager.Sync();
            if (_loadStoredAppFeaturesOnNetworkRestore.Value)
            {
                _workersManager.DoJob<StoredAppFeatureLoadingWorkerJob>();
            }
        }

        private void ConfigSyncedHandler<T>(T _)
        {
            _configSyncsHandled++;
            _logger.Log($"{nameof(ConfigSyncedHandler)} called ({_configSyncsHandled} call)");
            if (StoredAppFeaturesPreloadingEnabled && _configSyncsHandled == 1)
            {
                _workersManager.DoJob<StoredAppFeatureLoadingWorkerJob>();
            }
        }

        private void FocusGainedHandler()
        {
            if (!_currentCampaigns.Any() && !DelayedCampaignRegistry.Any)
            {
                Time.MarkAsUnprotected();

                if (string.IsNullOrEmpty(_settings.HotSessionStartedEvent))
                {
                    _logger.Log($"Skip {nameof(_settings.HotSessionStartedEvent)} event because it is empty");
                    return;
                }

                if (!SessionStarted.Value)
                {
                    return;
                }

                if (_prefs.PauseTime.Value == default)
                {
                    _logger.LogError($"{nameof(_prefs.PauseTime)} is default. It is impossible by design but it happened. Check last changes in class {nameof(MagifyService)}");
                    return;
                }
                var seconds = (int)(DateTime.UtcNow - _prefs.PauseTime.Value).TotalSeconds;
                if (DateTime.UtcNow < _prefs.PauseTime.Value)
                {
                    _logger.LogError($"Trying to send {_settings.HotSessionStartedEvent} but away time less than 0. Probably time change detected. Set it to 0");
                    seconds = 0;
                }

                requestHotSessionStart().Forget();

                async UniTaskVoid requestHotSessionStart()
                {
                    Campaigns.SetParam(_settings.HotSessionStartedEvent, _settings.AwayTimeParam, seconds);
                    await RequestCampaignAsync(_settings.HotSessionStartedEvent, "", null, CancellationToken.None);
                    Campaigns.RemoveParam(_settings.HotSessionStartedEvent, _settings.AwayTimeParam);
                }
            }
            _workersManager.DoJob<HandlePauseTimeOnFocusGainedWorkerJob>();
        }

#if UNITY_EDITOR
        internal void StartNewSession()
        {
            _sessionCounter.Value++;
        }

        internal void SetINetworkStatusProvider(INetworkStatusProvider networkStatusProvider)
        {
            (Network as IDisposable)?.Dispose();
            Network = networkStatusProvider;
            Network.Reachability
                .SkipLatestValueOnSubscribe()
                .Where(c => c is NetworkState.Reachable)
                .Subscribe(NetworkReachabilityChangedHandler)
                .AddTo(_disposables);
        }
#endif

        #endregion
    }
}
#endif