﻿using System;
using JetBrains.Annotations;

namespace Magify
{
    public interface ILocalizer
    {
        public string Localize(LocalizationKey key, params object[] args);

        public string Localize(ErrorPopup.Type errorType, params object[] args)
        {
            var key = errorType switch
            {
                ErrorPopup.Type.Unknown => LocalizationKey.UnknownError,
                ErrorPopup.Type.NoInternet => LocalizationKey.NoInternetError,
                _ => throw new ArgumentOutOfRangeException(nameof(errorType), errorType, $"Unexpected errorType: {errorType}")
            };
            return Localize(key, args);
        }
    }

    public enum LocalizationKey
    {
        BuyLabel,
        UnknownError,
        NoInternetError,
    }

    internal class EmbeddedLocalizer : ILocalizer
    {
        private readonly ILocalizer _externalLocalizer;

        public EmbeddedLocalizer([CanBeNull] ILocalizer externalLocalizer = null)
        {
            _externalLocalizer = externalLocalizer;
        }

        public string Localize(LocalizationKey key, params object[] args)
        {
            if (_externalLocalizer != null)
            {
                try
                {
                    var translated = _externalLocalizer.Localize(key);
                    if (!string.IsNullOrEmpty(translated))
                    {
                        return translated;
                    }
                }
                catch
                {
                    // ignored
                }
            }

            return key switch
            {
                LocalizationKey.BuyLabel => "Buy",
                LocalizationKey.UnknownError => "Something went wrong...",
                LocalizationKey.NoInternetError => "No internet connection",
                _ => throw new ArgumentOutOfRangeException(nameof(key), key, $"Unexpected key: {key}")
            };
        }
    }
}