﻿using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using UnityEngine;

namespace Magify
{
    public class NoUiCreativeCampaignHandler : ICampaignHandler
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly ProductsObtainer _productsObtainer;

        public NoUiCreativeCampaignHandler(ProductsObtainer productsObtainer)
        {
            _productsObtainer = productsObtainer;
        }

        public bool CanHandleCampaign(CampaignRequest request)
        {
            return request.Campaign is ICampaignWithProducts and ICampaignWithCreative { Creative: NoUiCreative };
        }

        public async UniTask<CampaignResult> HandleCampaignAsync(CampaignRequest request, CancellationToken cancellationToken)
        {
            var campaignWithProducts = (ICampaignWithProducts)request.Campaign;
            var products = campaignWithProducts.Products;

            var product = string.IsNullOrEmpty(request.PayoutType)
                ? products.First()
                : products.First(c => c.Payout.Any(k => k.Type == request.PayoutType));

            if (!_productsObtainer.CanObtainProduct(product))
            {
                _logger.LogError($"No product obtainer for {product.GetType().Name}");
                request.TrackShowFailed($"No product obtainer for {product.GetType().Name}");
                return CampaignResult.Failed;
            }

            var obtainResult = await _productsObtainer.ObtainProductAsync(product, request, cancellationToken);
            cancellationToken.ThrowIfCancellationRequested();
            _logger.Log($"{nameof(NoUiCreativeCampaignHandler)}: Obtain product {product.Id} result: {obtainResult.FailReason}");

            switch (obtainResult.FailReason)
            {
                case null:
                    return CampaignResult.Applied;
                case ProductObtainFailReason.NoInternet:
                    return CampaignResult.NoInternet;
                case ProductObtainFailReason.UserCancelled:
                    return CampaignResult.Declined;
                default:
                    return CampaignResult.Failed;
            }
        }
    }
}