using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal class AdPreloader : IDisposable
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private IAdsMediator _mediator;

        private readonly CompositeDisposable _disposables = new();
        private readonly ExponentialBackoff _interstitialDelay = new(maxDelayMilliseconds: ulong.MaxValue);
        private readonly ExponentialBackoff _rewardedDelay = new(maxDelayMilliseconds: ulong.MaxValue);

        private CancellationTokenSource _interstitialToken;
        private CancellationTokenSource _rewardedToken;

        [NotNull]
        public ReactiveProperty<bool> IsEnabled { get; } = new ReactiveProperty<bool>();

        public AdPreloader()
        {
            IsEnabled.Subscribe(enabled =>
            {
                if (enabled) Enable();
                else Disable();
            }).AddTo(_disposables);
        }

        void IDisposable.Dispose()
        {
            DisposeCurrentMediator();
            _disposables?.Dispose();
        }

        public void SetAdsMediator([CanBeNull] IAdsMediator mediator)
        {
            DisposeCurrentMediator();
            _mediator = mediator;
            if (IsEnabled.Value)
            {
                Enable();
            }
        }

        private void DisposeCurrentMediator()
        {
            if (IsEnabled.Value)
            {
                Disable();
            }
            _mediator = null;
        }

        private void Enable()
        {
            if (_mediator != null)
            {
                _logger.Log($"The preload of ads for mediator {_mediator.GetType().Name} has been enabled.");
                _mediator.OnInterHidden += PreloadNextInterstitial;
                _mediator.OnRewardHidden += PreloadNextRewarded;
                PreloadNextInterstitial(default);
                PreloadNextRewarded(default);
            }
        }

        private void Disable()
        {
            if (_mediator != null)
            {
                _logger.Log($"The preload of ads for mediator {_mediator.GetType().Name} has been disabled.");
                _mediator.OnInterHidden -= PreloadNextInterstitial;
                _mediator.OnRewardHidden -= PreloadNextRewarded;
            }
            CancelInterstitialPreload();
            CancelRewardedPreload();
        }

        private async void PreloadNextInterstitial(IAdsImpression prevAdsImpression)
        {
            CancelInterstitialPreload();
            if (_mediator.IsInitialized && _mediator.IsInterVideoReady) return;

            _interstitialToken = new();
            var loadPromise = new UniTaskCompletionSource<bool>();
            try
            {
                await UniTask.WaitUntil(() => _mediator.IsInitialized, cancellationToken: _interstitialToken.Token);
                _mediator.OnInterLoaded += handleLoaded;
                _mediator.OnInterLoadFailed += handleLoadFailed;
                while (true)
                {
                    _logger.Log($"Try to preload interstitial ad [attempt {_rewardedDelay.Retries}].");
                    _mediator.LoadInterVideo();
                    if (await loadPromise.Task)
                    {
                        _logger.Log("Interstitial ad has been successfully preloaded.");
                        return;
                    }
                    await UniTask.Delay(TimeSpan.FromMilliseconds(_interstitialDelay.NextDelay()), cancellationToken: _interstitialToken.Token);
                }
            }
            catch (OperationCanceledException)
            {
            }
            finally
            {
                if (_mediator != null)
                {
                    _mediator.OnInterLoaded -= handleLoaded;
                    _mediator.OnInterLoadFailed -= handleLoadFailed;
                }
            }

            void handleLoaded() => loadPromise.TrySetResult(true);
            void handleLoadFailed(string _) => loadPromise.TrySetResult(false);
        }

        private async void PreloadNextRewarded(IAdsImpression prevAdsImpression)
        {
            CancelRewardedPreload();
            if (_mediator.IsInitialized && _mediator.IsRewardedVideoReady) return;

            _rewardedToken = new();
            var loadPromise = new UniTaskCompletionSource<bool>();
            try
            {
                await UniTask.WaitUntil(() => _mediator.IsInitialized, cancellationToken: _rewardedToken.Token);
                _mediator.OnRewardLoaded += handleLoaded;
                _mediator.OnRewardLoadFailed += handleLoadFailed;
                while (true)
                {
                    _logger.Log($"Try to preload rewarded ad [attempt {_rewardedDelay.Retries}].");
                    _mediator.LoadRewardVideo();
                    if (await loadPromise.Task)
                    {
                        _logger.Log("Rewarded ad has been successfully preloaded.");
                        return;
                    }
                    await UniTask.Delay(TimeSpan.FromMilliseconds(_rewardedDelay.NextDelay()), cancellationToken: _rewardedToken.Token);
                }
            }
            catch (OperationCanceledException)
            {
            }
            finally
            {
                if (_mediator != null)
                {
                    _mediator.OnRewardLoaded -= handleLoaded;
                    _mediator.OnRewardLoadFailed -= handleLoadFailed;
                }
            }

            void handleLoaded() => loadPromise.TrySetResult(true);
            void handleLoadFailed(string _) => loadPromise.TrySetResult(false);
        }

        private void CancelInterstitialPreload()
        {
            _interstitialDelay.Reset();
            _interstitialToken?.Cancel();
            _interstitialToken?.Dispose();
            _interstitialToken = default;
        }

        private void CancelRewardedPreload()
        {
            _rewardedDelay.Reset();
            _rewardedToken?.Cancel();
            _rewardedToken?.Dispose();
            _rewardedToken = default;
        }
    }
}