﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using UnityEditor;
using UnityEditor.PackageManager;
using UnityEditor.PackageManager.Requests;
using UnityEngine;

namespace Magify.DependenciesInstaller
{
    internal static class DependenciesInstaller
    {
        internal static async Task Install(IReadOnlyCollection<DependencyElementConfiguration> elements)
        {
            EditorApplication.LockReloadAssemblies();
            await AddScopedRegistries(elements);
            EditorApplication.UnlockReloadAssemblies();
            AssetDatabase.Refresh();

            EditorApplication.LockReloadAssemblies();
            foreach (var element in elements)
            {
                switch (element.InstallationMode)
                {
                    case InstallationMode.ByPackageName:
                        await Install(element.PackageNameId, element.DisplayName);
                        break;
                    case InstallationMode.ByGitUrl:
                        await Install(element.GitUrlId, element.DisplayName);
                        break;
                    case InstallationMode.ManualDownload:
                        await PackageDownloader.TryDownload(element.ManualDownload.Url, element.ManualDownload.NameWithExtension);
                        break;
                }
            }
            EditorApplication.UnlockReloadAssemblies();
        }

        internal static Task<bool> Install(string installationIdentifier, string packageName)
        {
            if (string.IsNullOrWhiteSpace(installationIdentifier)) return Task.FromResult(false);
            return WaitPackageInstallingRequest(Client.Add(installationIdentifier), packageName);
        }

        private static async Task AddScopedRegistries(IReadOnlyCollection<DependencyElementConfiguration> elements)
        {
            try
            {
                var registeredScopes = await ScopedRegistryInstaller.GetScopes();
                var parsed = new Dictionary<(string, string), HashSet<string>>();
                foreach (var element in elements)
                {
                    if (!isValid(element)) continue;
                    var registry = element.Registry;
                    parsed.TryAdd((registry.RegistryName, registry.URL), new());
                }

                foreach (var element in elements)
                {
                    if (!isValid(element)) continue;
                    var registry = element.Registry;
                    var set = parsed[(registry.RegistryName, registry.URL)];
                    foreach (var scope in registry.Scopes) set.Add(scope);
                }

                foreach (var ((name, url), set) in parsed)
                {
                    if (set.Count == 0) continue;
                    await InstallRegistry(new(name, url, set.Where(item => !registeredScopes.Contains(item)).ToArray()));
                }
            }
            catch (Exception e)
            {
                Debug.LogException(e);
            }
            return;

            bool isValid(DependencyElementConfiguration element)
            {
                return element.InstallationMode != InstallationMode.None
                    && !element.IsInstalled
                    && !string.IsNullOrWhiteSpace(element.Registry.RegistryName)
                    && !string.IsNullOrWhiteSpace(element.Registry.URL)
                    && element.Registry.Scopes is { Length: > 0 };
            }
        }

        private static async Task InstallRegistry(ScopedRegistry scopedRegistry)
        {
            var finishedRequest = await ScopedRegistryInstaller.Install(scopedRegistry);
            var builder = new StringBuilder();
            builder.Append(finishedRequest.Status == StatusCode.Failure
                    ? "Failed to add Scoped Registry:\n"
                    : "Scoped Registry successfully added:\n")
                .Append(scopedRegistry.RegistryName)
                .Append(",\n")
                .Append(scopedRegistry.URL)
                .Append("\n{\n");
            foreach (var scope in scopedRegistry.Scopes) builder.Append("\t").Append(scope).Append(",\n");
            builder.Append("}");
            if (finishedRequest.Status == StatusCode.Failure) Debug.LogError(builder.ToString());
            else Debug.Log(builder.ToString());
        }

        private static async Task<bool> WaitPackageInstallingRequest(Request request, string packageName)
        {
            var title = $"{packageName} installing...";
            var progress = 0f;
            while (request.Status == StatusCode.InProgress)
            {
                EditorUtility.DisplayProgressBar(title, title, Mathf.Clamp01(progress));
                await Task.Yield();
                progress += 0.01f;
            }

            EditorUtility.ClearProgressBar();
            if (request.Status == StatusCode.Failure)
            {
                Debug.LogError($"A request to add {packageName} was failed ({request.Error.message}). You can try to add this package yourself." +
                               "\n-- You can also try to use " + DependenciesInstallerWindow.MenuItemPath + " to try again.");
                return false;
            }
            Debug.Log($"The {packageName} package has been successfully installed!");
            AssetDatabase.Refresh();
            return true;
        }
    }
}