﻿#if UNITY_LOCALIZATION
using System.Collections.Generic;
using System.Linq;
using TMPro;
using UnityEditor.Localization;
using UnityEngine;
using UnityEngine.Localization.Components;
using UnityEngine.Localization.Settings;
using UnityEngine.Localization.Tables;

namespace Magify
{
    public class UnityLocalizationEditorHelper : ILocalizationEditorHelper
    {
        public string GetCurrentLocale() => LocalizationSettings.SelectedLocale.LocaleName;

        public void LocalizeText(TMP_Text tmpText)
        {
            var localization = tmpText.GetComponent<LocalizeStringEvent>();
            if (localization != null)
            {
                if (localization.StringReference.Count != 0)
                {
                    tmpText.text = localization.StringReference.GetLocalizedString();
                }
                else
                {
                    var currentLocale = LocalizationSettings.SelectedLocale.Identifier.Code;
                    tmpText.text = GetLocalizedString(currentLocale, localization.StringReference.TableEntryReference).LocalizedValue;
                }
            }
        }

        public string GetLocalizedCacheCharacters(TMP_Text tmpText, bool isUpperOrLowerCase, List<string> availableLocales, bool includeAllLanguages)
        {
            var characters = string.Empty;
            var bundleLocalization = tmpText.GetComponent<BundleLocalizationCache>();

            if (bundleLocalization != null)
            {
                foreach (var key in bundleLocalization.LocalizeReferences)
                {
                    if (includeAllLanguages)
                    {
                        foreach (var localeCode in availableLocales)
                        {
                            characters += getLocalizedCharacters(key.TableEntryReference, localeCode);
                        }
                    }
                    else
                    {
                        characters += getLocalizedCharacters(key.TableEntryReference, LocalizationSettings.SelectedLocale.Identifier.Code);
                    }
                }

                Object.DestroyImmediate(bundleLocalization, true);
            }

            return characters;

            string getLocalizedCharacters(TableEntryReference tableEntryReference, string localeCode)
            {
                var localizedText = GetUsedLocalizedCharacters(localeCode, tableEntryReference);
                if (isUpperOrLowerCase)
                {
                    localizedText = localizedText.ToLower() + localizedText.ToUpper();
                }
                return localizedText;
            }
        }

        public Dictionary<string, string> GetLocalizationStringForPreparing(TMP_Text tmp, List<string> availableLocales, bool includeAllLanguages)
        {
            var collection = new Dictionary<string, string>
            {
                [CreativeBuilderConstants.DefaultTextKey] = tmp.text
            };

            var localization = tmp.GetComponent<LocalizeStringEvent>();
            if (localization != null)
            {
                if (includeAllLanguages)
                {
                    foreach (var localeCode in availableLocales)
                    {
                        if (localization.StringReference.Count != 0)
                        {
                            SetCurrentLocale(localeCode);
                            collection.Add(localeCode, localization.StringReference.GetLocalizedString());
                        }
                        else
                        {
                            collection.Add(localeCode, GetLocalizedString(localeCode, localization.StringReference.TableEntryReference).LocalizedValue);
                        }
                    }
                }
                else
                {
                    var currentLocale = LocalizationSettings.SelectedLocale.Identifier.Code;
                    collection.Add(currentLocale, GetLocalizedString(currentLocale, localization.StringReference.TableEntryReference).LocalizedValue);
                }
            }

            return collection;
        }

        public string GetCharactersForPreparing(TMP_Text tmp, List<string> availableLocales, bool includeAllLanguages)
        {
            var symbols = tmp.text;

            var localization = tmp.GetComponent<LocalizeStringEvent>();
            if (localization != null)
            {
                if (includeAllLanguages)
                {
                    foreach (var localeCode in availableLocales)
                    {
                        symbols += GetUsedLocalizedCharacters(localeCode, localization.StringReference.TableEntryReference);
                    }
                }
                else
                {
                    symbols += GetUsedLocalizedCharacters(LocalizationSettings.SelectedLocale.Identifier.Code, localization.StringReference.TableEntryReference);
                }
            }

            return symbols;
        }

        public void RemoveLocalizationScripts(GameObject editingScopePrefabContentsRoot)
        {
            var localizeStringEvents = editingScopePrefabContentsRoot.GetComponentsInChildren<LocalizeStringEvent>(true);
            foreach (var localizeStringEvent in localizeStringEvents)
            {
                Object.DestroyImmediate(localizeStringEvent, true);
            }
        }

        public void SetCurrentLocale(string localeCode)
        {
            var locale = LocalizationSettings.AvailableLocales.GetLocale(localeCode);
            LocalizationSettings.SelectedLocale = locale;
        }

        private string GetUsedLocalizedCharacters(string cultureCode, TableEntryReference tableEntryReference)
        {
            var localizeEntry = GetLocalizedString(cultureCode, tableEntryReference);
            var smart = new BundleLiteralCharacterExtractor(LocalizationSettings.StringDatabase?.SmartFormatter);
            if (localizeEntry.IsSmart)
            {
                var literals = smart.ExtractLiteralsCharacters(localizeEntry.LocalizedValue);
                var sorted = literals.Distinct().OrderBy(c => c);
                return string.Concat(sorted);
            }

            return localizeEntry.LocalizedValue;
        }

        private StringTableEntry GetLocalizedString(string cultureCode, TableEntryReference tableEntryReference)
        {
            var localizationCollection = LocalizationEditorSettings.GetStringTableCollection(UnityLocalizerTables.GlobalLocalizationTableName);
            var currentCultureTable = (StringTable)localizationCollection.GetTable(cultureCode);
            return currentCultureTable.GetEntryFromReference(tableEntryReference);
        }
    }
}
#endif