using System.IO;
using Magify.Model;
using FluentAssertions;
using NUnit.Framework;
using UnityEngine;

namespace Magify.Tests
{
    public class JsonModelParseTests
    {
        private const string TestConfig = "JsonModelParse/test_config";
        private readonly string _defaultConfigPath = Path.Combine(Application.streamingAssetsPath, "Magify-GP.json");

        [Test]
        public void WhenDeserializeContext_ThenShouldDoneWithoutErrorAndCorrectFillModel()
        {
            // Arrange
            var configAsset = Resources.Load<TextAsset>(TestConfig);

            // Act
            var campaignsContext = JsonFacade.DeserializeObject<CampaignsContext>(configAsset.text);

            // Assert
            campaignsContext.Should().NotBeNull();
        }

        [Test]
        public void WhenSerializeContext_ThenShouldDoneWithoutErrorAndCorrectFillModel()
        {
            // Arrange
            var configAsset = Resources.Load<TextAsset>(TestConfig);
            var campaignsContext = JsonFacade.DeserializeObject<CampaignsContext>(configAsset.text);

            // Act
            var serializeContext = JsonFacade.SerializeObject(campaignsContext);

            // Assert
            serializeContext.Should().NotBeNullOrEmpty();
        }

        [Test]
        public void WhenDeserializeDefaultContext_ThenShouldDoneWithoutErrorAndCorrectFillModel()
        {
            // Arrange
            var configAsset = File.ReadAllText(_defaultConfigPath);

            // Act
            var campaignsContext = JsonFacade.DeserializeObject<CampaignsContext>(configAsset);

            // Assert
            campaignsContext.Should().NotBeNull();
        }

        [Test]
        public void WhenSerializeDefaultContext_ThenShouldDoneWithoutErrorAndCorrectFillModel()
        {
            // Arrange
            var configAsset = File.ReadAllText(_defaultConfigPath);
            var campaignsContext = JsonFacade.DeserializeObject<CampaignsContext>(configAsset);

            // Act
            var serializeContext = JsonFacade.SerializeObject(campaignsContext);

            // Assert
            serializeContext.Should().NotBeNullOrEmpty();
        }

        [Test]
        public void WhenDeserializeErrorResponse_ThenShouldReturnCorrectObject()
        {
            // Arrange
            const string jsonString = @"{""jsonrpc"": ""2.0"", ""error"": {""code"": 3, ""message"": ""Invalid JWT.""}, ""id"": ""1""}";

            // Act
            var response = JsonFacade.DeserializeObject<ErrorResponse>(jsonString);

            // Assert
            response.Should().NotBeNull();
            response.Error.Should().NotBeNull();
            response.Error.Code.Should().Be(ErrorCode.Expired);
            response.Error.Message.Should().Be("Invalid JWT.");
            response.Id.Should().Be("1");
        }
    }
}