using System;
using FluentAssertions;
using Newtonsoft.Json;
using NUnit.Framework;

namespace Magify.Tests.Json
{
    public static class DateTimeExtensions
    {
        public static long AsUnixTime(this DateTime dt, TimeUnit unit)
        {
            var span = dt - DateTime.UnixEpoch;
            var result = unit switch
            {
                TimeUnit.Ticks => span.Ticks,
                TimeUnit.Millisecond => span.TotalMilliseconds,
                TimeUnit.Second => span.TotalSeconds,
                TimeUnit.Minute => span.TotalMinutes,
                TimeUnit.Hour => span.TotalHours,
                TimeUnit.Day => span.TotalDays,
                _ => throw new ArgumentOutOfRangeException(nameof(unit), unit, null)
            };
            return (long)result;
        }

        public static long AsUnixTime(this DateTime? dt, TimeUnit unit)
        {
            return dt?.AsUnixTime(unit) ?? 0;
        }
    }

    public class UnitDateTimeConverterTests
    {
        class NullableDateTimeContainer
        {
            [JsonProperty("date_ticks")]
            [JsonConverter(typeof(UnitDateTimeConverter))]
            public DateTime? DateInTicks { get; set; }

            [JsonProperty("date_milliseconds")]
            [JsonConverter(typeof(UnitDateTimeConverter), TimeUnit.Millisecond)]
            public DateTime? DateInMillisecond { get; set; }

            [JsonProperty("date_seconds")]
            [JsonConverter(typeof(UnitDateTimeConverter), TimeUnit.Second)]
            public DateTime? DateInSeconds { get; set; }

            [JsonProperty("date_minutes")]
            [JsonConverter(typeof(UnitDateTimeConverter), TimeUnit.Minute)]
            public DateTime? DateInMinutes { get; set; }

            [JsonProperty("date_hours")]
            [JsonConverter(typeof(UnitDateTimeConverter), TimeUnit.Hour)]
            public DateTime? DateInHours { get; set; }

            [JsonProperty("date_days")]
            [JsonConverter(typeof(UnitDateTimeConverter), TimeUnit.Day)]
            public DateTime? DateInDays { get; set; }
        }

        class DateTimeContainer
        {
            [JsonProperty("date_ticks")]
            [JsonConverter(typeof(UnitDateTimeConverter))]
            public DateTime DateInTicks { get; set; }

            [JsonProperty("date_milliseconds")]
            [JsonConverter(typeof(UnitDateTimeConverter), TimeUnit.Millisecond)]
            public DateTime DateInMillisecond { get; set; }

            [JsonProperty("date_seconds")]
            [JsonConverter(typeof(UnitDateTimeConverter), TimeUnit.Second)]
            public DateTime DateInSeconds { get; set; }

            [JsonProperty("date_minutes")]
            [JsonConverter(typeof(UnitDateTimeConverter), TimeUnit.Minute)]
            public DateTime DateInMinutes { get; set; }

            [JsonProperty("date_hours")]
            [JsonConverter(typeof(UnitDateTimeConverter), TimeUnit.Hour)]
            public DateTime DateInHours { get; set; }

            [JsonProperty("date_days")]
            [JsonConverter(typeof(UnitDateTimeConverter), TimeUnit.Day)]
            public DateTime DateInDays { get; set; }
        }

        [Test]
        public void WhenValidDateTimeSerialized_AndDeserialized_ThenDateTimeIsCorrect()
        {
            // Arrange
            var now = DateTime.UtcNow;
            var data = new DateTimeContainer
            {
                DateInTicks = now,
                DateInMillisecond = now,
                DateInSeconds = now,
                DateInMinutes = now,
                DateInHours = now,
                DateInDays = now,
            };
            var json = JsonFacade.SerializeObject(data);

            // Act
            data = JsonFacade.DeserializeObject<DateTimeContainer>(json);

            // Assert
            data.DateInTicks.Should().Be(now);
            data.DateInMillisecond.AsUnixTime(TimeUnit.Millisecond).Should().Be(now.AsUnixTime(TimeUnit.Millisecond));
            data.DateInSeconds.AsUnixTime(TimeUnit.Second).Should().Be(now.AsUnixTime(TimeUnit.Second));
            data.DateInMinutes.AsUnixTime(TimeUnit.Minute).Should().Be(now.AsUnixTime(TimeUnit.Minute));
            data.DateInHours.AsUnixTime(TimeUnit.Hour).Should().Be(now.AsUnixTime(TimeUnit.Hour));
            data.DateInDays.AsUnixTime(TimeUnit.Day).Should().Be(now.AsUnixTime(TimeUnit.Day));
        }

        [Test]
        public void WhenNullableDateTimeSerialized_AndDeserialized_ThenDateTimeNull()
        {
            // Arrange
            var data = new NullableDateTimeContainer();
            var json = JsonFacade.SerializeObject(data);

            // Act
            data = JsonFacade.DeserializeObject<NullableDateTimeContainer>(json);

            // Assert
            data.DateInTicks.Should().BeNull();
            data.DateInMillisecond.Should().BeNull();
            data.DateInSeconds.Should().BeNull();
            data.DateInMinutes.Should().BeNull();
            data.DateInHours.Should().BeNull();
            data.DateInDays.Should().BeNull();
        }

        [Test]
        public void WhenDefaultDateTimeSerialized_AndDeserialized_ThenDateIsDefault()
        {
            // Arrange
            var data = new DateTimeContainer();
            var json = JsonFacade.SerializeObject(data);

            // Act
            data = JsonFacade.DeserializeObject<DateTimeContainer>(json);

            // Assert
            data.DateInTicks.Should().Be(default);
            data.DateInMillisecond.Should().Be(default);
            data.DateInSeconds.Should().Be(default);
            data.DateInMinutes.Should().Be(default);
            data.DateInHours.Should().Be(default);
            data.DateInDays.Should().Be(default);
        }

        [Test]
        public void WhenValidNullableDateTimeSerialized_AndDeserialized_ThenDateTimeIsCorrect()
        {
            // Arrange
            var now = DateTime.UtcNow;
            var data = new NullableDateTimeContainer
            {
                DateInTicks = now,
                DateInMillisecond = now,
                DateInSeconds = now,
                DateInMinutes = now,
                DateInHours = now,
                DateInDays = now,
            };
            var json = JsonFacade.SerializeObject(data);

            // Act
            data = JsonFacade.DeserializeObject<NullableDateTimeContainer>(json);

            // Assert
            data.DateInTicks.Should().NotBeNull();
            data.DateInMillisecond.Should().NotBeNull();
            data.DateInSeconds.Should().NotBeNull();
            data.DateInMinutes.Should().NotBeNull();
            data.DateInHours.Should().NotBeNull();
            data.DateInDays.Should().NotBeNull();

            data.DateInTicks.Should().Be(now);
            data.DateInMillisecond.AsUnixTime(TimeUnit.Millisecond).Should().Be(now.AsUnixTime(TimeUnit.Millisecond));
            data.DateInSeconds.AsUnixTime(TimeUnit.Second).Should().Be(now.AsUnixTime(TimeUnit.Second));
            data.DateInMinutes.AsUnixTime(TimeUnit.Minute).Should().Be(now.AsUnixTime(TimeUnit.Minute));
            data.DateInHours.AsUnixTime(TimeUnit.Hour).Should().Be(now.AsUnixTime(TimeUnit.Hour));
            data.DateInDays.AsUnixTime(TimeUnit.Day).Should().Be(now.AsUnixTime(TimeUnit.Day));
        }
    }
}