using System.Collections.Generic;
using System.Linq;
using FluentAssertions;
using JetBrains.Annotations;
using Magify.Features;
using Magify.Model;
using NUnit.Framework;
using UnityEngine;

namespace Magify.Tests
{
    [Ignore("Need to be updated")]
    internal class StoredAppFeaturesProviderTests
    {
        public const string Key = "json_app_feature_test";
        public const string Value = "https://d1h7ix05hmhp5.cloudfront.net/media/stored-application-features/magify-sample-gp/ed872e0e-cb05-4fc0-9390-83d928e6b730.json";
        public const string AnotherKey = "other_json_app_feature_test";
        public const string AnotherValue = "https://d1h7ix05hmhp5.cloudfront.net/media/stored-application-features/magify-sample-gp/ed872e0e-cb05-4fc0-9390-83d928e6b730.json";

        [NotNull]
        private static object[] Context
        {
            get
            {
                var ctx = new CampaignsContext { StoredAppFeatures = new Dictionary<string, FeatureConfig> { { Key, new FeatureConfig(){DefaultValue = Value} }, }, };
                return new object[] { JsonFacade.SerializeObject(ctx) };
            }
        }
        [NotNull]
        private static object[] AnotherContext
        {
            get
            {
                var ctx = new CampaignsContext { StoredAppFeatures = new Dictionary<string, FeatureConfig> { { AnotherKey, new FeatureConfig(){DefaultValue = AnotherValue} }, }, };
                return new object[] { JsonFacade.SerializeObject(ctx) };
            }
        }

        [NotNull]
        private static object[] SavedAndDownloadedContexts
        {
            get
            {
                var ctx = new CampaignsContext { StoredAppFeatures = new Dictionary<string, FeatureConfig> { { Key, new FeatureConfig(){DefaultValue = Value} }, }, };
                return new object[]
                {
                    new object[] { JsonFacade.SerializeObject(ctx), ContextKind.Saved },
                    new object[] { JsonFacade.SerializeObject(ctx), ContextKind.Downloaded },
                };
            }
        }

        [NotNull]
        private static object[] LongContext
        {
            get
            {
                var ctx = new CampaignsContext { StoredAppFeatures = new Dictionary<string, FeatureConfig> {
                    { Key, new FeatureConfig(){DefaultValue = Value} },
                    { Key + "1", new FeatureConfig(){DefaultValue = Value} },
                    { Key + "2", new FeatureConfig(){DefaultValue = Value} },
                    { Key + "3", new FeatureConfig(){DefaultValue = Value} },
                    { Key + "4", new FeatureConfig(){DefaultValue = Value} },
                    { Key + "5", new FeatureConfig(){DefaultValue = Value} },
                    { Key + "6", new FeatureConfig(){DefaultValue = Value} },
                }};
                return new object[] { new object[] { JsonFacade.SerializeObject(ctx), ctx.StoredAppFeatures.Count } };
            }
        }

        [Test]
        [TestCaseSource(nameof(Context))]
        public void ParseContext_ThenContainsFeatures([NotNull] string contextJson)
        {
            // Arrange
            // Act
            var context = CampaignsContext.Deserialize(contextJson);

            // Assert
            context.StoredAppFeatures.Should().NotBeNull();
            context.StoredAppFeatures.Should().ContainKey(Key);
            context.StoredAppFeatures.Values.Any(v => (string)v.DefaultValue == Value).Should().BeTrue();
        }

        [Test]
        [TestCaseSource(nameof(Context))]
        public void ParseDefaultContext_HandleIt_HasDefaultFeatures([NotNull] string contextJson)
        {
            // Arrange
            var context = CampaignsContext.Deserialize(contextJson);

            // Act
            using var subsystems = Create(out var features);
            subsystems.UpdateContextForAll(context, ContextKind.Default);

            // Assert
            features.DefaultValues.Should().ContainKey(Key);
            features.DefaultValues.Should().ContainValue(Value);
        }

        [Test]
        [TestCaseSource(nameof(SavedAndDownloadedContexts))]
        public void ParseCurrentContext_HandleIt_HasCurrentFeatures([NotNull] string contextJson, ContextKind kind)
        {
            // Arrange
            var context = CampaignsContext.Deserialize(contextJson);

            // Act
            using var subsystems = Create(out var features);
            subsystems.UpdateContextForAll(context, kind);

            // Assert
            features.CurrentValues.Should().ContainKey(Key);
            features.CurrentValues.Should().ContainValue(Value);
        }

        [Test]
        [TestCaseSource(nameof(LongContext))]
        public void ParseCurrentContext_HandleIt_HasCount([NotNull] string contextJson, int count)
        {
            // Arrange
            var context = CampaignsContext.Deserialize(contextJson);

            // Act
            using var subsystems = Create(out var features);
            subsystems.UpdateContextForAll(context, ContextKind.Downloaded);

            // Assert
            features.CurrentValues.Should().HaveCount(count);
        }

        [Test]
        public void ParseDefaultAndDownloadedContexts_HandleIt_HasBothValues()
        {
            // Arrange
            var defaultContext = CampaignsContext.Deserialize(Context.First() as string);
            var currentContext = CampaignsContext.Deserialize(AnotherContext.First() as string);

            // Act
            using var subsystems = Create(out var features);
            subsystems.UpdateContextForAll(defaultContext, ContextKind.Default);
            subsystems.UpdateContextForAll(currentContext, ContextKind.Downloaded);

            // Assert
            features.Should().Contain(feature => feature.Name == Key);
            features.Should().Contain(feature => feature.Name == AnotherKey);
            features.Should().Contain(feature => feature.Url == Value);
            features.Should().Contain(feature => feature.Url == AnotherValue);
        }

        [Test]
        public void ParseSavedAndDefaultContexts_HandleIt_HasBothValues()
        {
            // Arrange
            var defaultContext = CampaignsContext.Deserialize(Context.First() as string);
            var currentContext = CampaignsContext.Deserialize(AnotherContext.First() as string);

            // Act
            using var subsystems = Create(out var features);
            subsystems.UpdateContextForAll(currentContext, ContextKind.Saved);
            subsystems.UpdateContextForAll(defaultContext, ContextKind.Default);

            // Assert
            features.Should().Contain(feature => feature.Name == Key);
            features.Should().Contain(feature => feature.Name == AnotherKey);
            features.Should().Contain(feature => feature.Url == Value);
            features.Should().Contain(feature => feature.Url == AnotherValue);
        }

        [Test]
        [TestCase(ContextKind.Default, ContextKind.Downloaded)]
        [TestCase(ContextKind.Saved, ContextKind.Downloaded)]
        [TestCase(ContextKind.Downloaded, ContextKind.Downloaded)]
        public void ParseContexts_HandleIt_StartIteration_ThenIteratedAllItems(ContextKind context1, ContextKind context2)
        {
            // Arrange
            var contextAndCount = LongContext.First() as object[];
            var context = CampaignsContext.Deserialize(contextAndCount.First() as string);
            var count = (int)contextAndCount.Skip(1).First();

            // Act
            using var subsystems = Create(out var features);
            subsystems.UpdateContextForAll(context, context1);
            var iterated = 0;
            var map = new Dictionary<string, string>();
            foreach (var feature in features)
            {
                iterated++;
                map.Add(feature.Name, feature.Url);
            }

            // Assert
            iterated.Should().Be(count);
            context.StoredAppFeatures.ForEach(pair =>
            {
                map.Should().ContainKey(pair.Key);
                map.Should().ContainValue(pair.Value.DefaultValue as string);
            });
            Debug.Log($"Total count: {count}, iterated(half): {iterated}");
        }

        [Test]
        [TestCase(ContextKind.Default, ContextKind.Downloaded)]
        [TestCase(ContextKind.Saved, ContextKind.Downloaded)]
        [TestCase(ContextKind.Downloaded, ContextKind.Downloaded)]
        public void ParseContexts_HandleIt_StartIteration_ThenUpdate_ThenIterationMustStop(ContextKind context1, ContextKind context2)
        {
            // Arrange
            var contextAndCount = LongContext.First() as object[];
            var context = CampaignsContext.Deserialize(contextAndCount.First() as string);
            var count = (int)contextAndCount.Skip(1).First();
            var half = count / 2;

            // Act
            using var subsystems = Create(out var features);
            subsystems.UpdateContextForAll(context, context1);
            var iterated = 0;
            foreach (var _ in features)
            {
                iterated++;
                if (iterated == half)
                    subsystems.UpdateContextForAll(context, context2);
            }

            // Assert
            iterated.Should().Be(half);
            Debug.Log($"Total count: {count}, iterated(half): {iterated}");
        }

        [NotNull]
        private SubsystemsCollection Create([NotNull] out StoredAppFeaturesProvider storedAppFeaturesProvider)
        {
            var subsystems = new SubsystemsCollection();
            var internalConfigPrefs = InternalConfigPrefs.Create(EditorModeTestsEnvironment.InternalConfigPrefsPath).AddTo(subsystems);
            var platform = new PlatformDefault(internalConfigPrefs).AddTo(subsystems);
            var appVersionProvider = new AppVersionProvider().AddTo(subsystems);
            var prefs = GeneralPrefs.Create(EditorModeTestsEnvironment.GeneralPrefsPath, EditorModeTestsEnvironment.LocalGeneralPrefsPath, new AppVersionProvider()).AddTo(subsystems);
            var mockServerApi = NSubstitute.Substitute.For<IServerApi>();
            var appStatePrefs = AppStatePrefs.Create(EditorModeTestsEnvironment.AppStatePrefsPath).AddTo(subsystems);
            var analyticsTracker = new AnalyticsTracker(mockServerApi, prefs, platform, appStatePrefs, EditorModeTestsEnvironment.RootStoragePath, appVersionProvider).AddTo(subsystems);
            var featuresProvider = new FeaturesProvider(prefs, analyticsTracker).AddTo(subsystems);
            storedAppFeaturesProvider = new StoredAppFeaturesProvider(prefs).AddTo(subsystems);
            var updater = new FeaturesUpdater(featuresProvider, storedAppFeaturesProvider, prefs).AddTo(subsystems);
            ((IInitializable)updater).Initialize();
            return subsystems;
        }
    }
}