﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using UnityEngine;
using UnityEngine.Networking;

namespace Magify.Tests
{
    internal class NetworkMoq : INetworkClient
    {
        internal delegate WebResponseMessage Handler(string url, WebRequestMessage message, CancellationToken cancellationToken);
        internal delegate UniTask<WebResponseMessage> AsyncHandler(string url, WebRequestMessage message, CancellationToken cancellationToken);
        private readonly Handler _handler;
        private readonly AsyncHandler _asyncHandler;

        public NetworkMoq([NotNull] Handler handler)
        {
            _handler = handler;
        }

        public NetworkMoq([NotNull] AsyncHandler asyncHandler)
        {
            _asyncHandler = asyncHandler;
        }

        public async UniTask<WebResponseMessage> SendAsync(string url, WebRequestMessage message, CancellationToken cancellationToken)
        {
            var result = _asyncHandler == null
                ? _handler!(url, message, cancellationToken)
                : await _asyncHandler(url, message, cancellationToken);
            cancellationToken.ThrowIfCancellationRequested();
            return result;
        }

        public static WebResponseMessage OkResult(WebRequestMessage message, RuntimePlatform? runtimePlatform = null)
        {
            object text = string.Empty;
            runtimePlatform ??= RuntimePlatform.Android;
            if (message.Method == WebRequestMethods.GetToken)
                text = new TokenResponse { Id = "id", Token = new string(Enumerable.Range(0, 100).Select(_ => 'a').ToArray()), };
            if (message.Method == WebRequestMethods.GetContext)
                text = new BaseResponse<CampaignsContext>() { Result = GetCampaignsContext((message.Payload as ContextPayload).Scopes.FromStringArray())};
            if (message.Method == WebRequestMethods.VerifyPurchase(runtimePlatform.Value))
                text = string.Empty;
            if (message.Method == ApplicationStateApiMethods.GetProgress(runtimePlatform.Value))
                text = new GaladrielResponse<AppStateStateResponse>(){ Id = "id", Result = new AppStateStateResponse(){Progress = "progress"}};
            if (message.Method == ApplicationStateApiMethods.Authorize(runtimePlatform.Value))
                text = new GaladrielResponse<GaladrielUser>(){Id = "id", Result = new GaladrielUser(){UserId = message.AuthToken}};

            return new WebResponseMessage
            {
                Result = UnityWebRequest.Result.Success,
                ResponseCode = 200,
                Error = string.Empty,
                Text = JsonFacade.SerializeObject(text),
                RequestMessage = message,
            };
        }

        public static WebResponseMessage ProtocolErrorResult(WebRequestMessage message, string text)
        {
            return new WebResponseMessage
            {
                Result = UnityWebRequest.Result.ProtocolError,
                ResponseCode = 403,
                Error = string.Empty,
                Text = text,
                RequestMessage = message,
            };
        }

        public static WebResponseMessage ProtocolErrorResult(WebRequestMessage message, [NotNull] ErrorResponse text)
        {
            return new WebResponseMessage
            {
                Result = UnityWebRequest.Result.ProtocolError,
                ResponseCode = 403,
                Error = string.Empty,
                Text = JsonFacade.SerializeObject(text),
                RequestMessage = message,
            };
        }

        public static bool TryExtractEvent<T>(WebRequestMessage message, Model.EventType type, out T @event)
            where T: IAnalyticsEvent
        {
            if (message.Payload is StorePayload storePayload && storePayload.Model == type)
            {
                @event = (T)storePayload.Objects!.FirstOrDefault();
                return @event != null;
            }
            @event = default;
            return false;
        }

        public static bool TryExtractEvents<T>(WebRequestMessage message, Model.EventType type, out List<T> events)
            where T: IAnalyticsEvent
        {
            if (message.Payload is StorePayload storePayload && storePayload.Model == type)
            {
                events = (List<T>)storePayload.Objects;
                return events != null;
            }
            events = default;
            return false;
        }

        public static bool TryExtractToken(WebRequestMessage message, out AuthTokenPayload authTokenPayload)
        {
            authTokenPayload = message.Payload as AuthTokenPayload;
            return authTokenPayload != null;
        }

        public static bool TryExtractVerification(WebRequestMessage message, out VerifyPurchasePayload verifyPurchasePayload, RuntimePlatform platform = RuntimePlatform.Android)
        {
            if (message.Payload is VerifyPurchasePayload payload && message.Method == WebRequestMethods.VerifyPurchase(platform))
            {
                verifyPurchasePayload = payload;
                return true;
            }
            verifyPurchasePayload = default;
            return false;
        }

        [NotNull]
        private static CampaignsContext GetCampaignsContext(ConfigScope configScope)
        {
            var result = new CampaignsContext
            {
                ContextBuildTime = DateTime.UtcNow,
            };
            if (configScope.HasFlag(ConfigScope.Limits))
            {
                result.Limits = new();
            }
            if (configScope.HasFlag(ConfigScope.ProductIds))
            {
                result.DefaultProducts = new();
            }
            if (configScope.HasFlag(ConfigScope.Content))
            {
                result.Content = new();
            }
            if (configScope.HasFlag(ConfigScope.Campaigns))
            {
                result.CampaignModels = new();
            }
            if (configScope.HasFlag(ConfigScope.AppFeatures))
            {
                result.Features = new();
            }
            if (configScope.HasFlag(ConfigScope.Segmentations))
            {
                result.Segmentations = Array.Empty<string>();
            }
            if (configScope.HasFlag(ConfigScope.AbTests))
            {
                result.AssignedAbTests = new();
            }
            if (configScope.HasFlag(ConfigScope.StoredAppFeatures))
            {
                result.StoredAppFeatures = new();
            }

            return result;
        }
    }
}