using System.Threading;
using JetBrains.Annotations;
using NUnit.Framework;

namespace Magify.Tests
{
    internal partial class MagifyClientTests
    {
        private const string AppName = "TestApp";
        private const string ConfigPath = "";
        private const Environment OfflineEnvironment = Environment.Offline;

        public enum ClientKind
        {
            Minimal,
            Full,
        }

        internal class MagifyClientTestsPart
        {
            [SetUp]
            [TearDown]
            public void ClearAllData()
            {
                EditorModeTestsEnvironment.Clear();
            }

            protected CancellationToken GetCancellationToken(int cancelDelay = 1000)
            {
                return new CancellationTokenSource(cancelDelay).Token;
            }
        }

        [NotNull]
        private static IMagifyClient CreateDefault(ClientKind kind = ClientKind.Full, INetworkClient networkClient = default, EditorDevice editorDevice = default, Environment? environment = null)
        {
            if (kind is ClientKind.Full)
            {
                if (networkClient == null)
                {
                    return new MagifyClient(AppName, ConfigPath, EditorModeTestsEnvironment.RootStoragePath, new ClientStateConfig(), environment ?? OfflineEnvironment, editorDevice);
                }
                return new MagifyClient(AppName, ConfigPath, EditorModeTestsEnvironment.RootStoragePath, new ClientStateConfig(), environment ?? OfflineEnvironment, editorDevice, networkClient);
            }
            if (kind is ClientKind.Minimal)
            {
                editorDevice ??= EditorDevice.GetDefaultAndroidDevice();
                return networkClient == null
                    ? new MinimalMagifyClient(AppName, ConfigPath, EditorModeTestsEnvironment.RootStoragePath, environment ?? OfflineEnvironment)
                    : new MinimalMagifyClient(AppName, ConfigPath, EditorModeTestsEnvironment.RootStoragePath, environment ?? OfflineEnvironment, editorDevice, networkClient);
            }
            throw new System.ArgumentOutOfRangeException();
        }

        [NotNull]
        private static MagifyClient CreateDefaultAndInit()
        {
            var client = new MagifyClient(AppName, ConfigPath, EditorModeTestsEnvironment.RootStoragePath, EditorModeTestsEnvironment.DefaultClientStateConfig, OfflineEnvironment, EditorDevice.GetDefaultAndroidDevice());
            client.InitializeSdk();
            client.Setup();
            return client;
        }


        [NotNull]
        private static IMagifyClient CreateDefaultAndInit(ClientKind kind, Environment? environment = Environment.Staging, INetworkClient networkClient = default, EditorDevice editorDevice = default)
        {
            if (kind is ClientKind.Full)
            {
                var client = new MagifyClient(AppName, ConfigPath, EditorModeTestsEnvironment.RootStoragePath, EditorModeTestsEnvironment.DefaultClientStateConfig, OfflineEnvironment, editorDevice);
                client.InitializeSdk();
                client.Setup();
                return client;
            }
            if (kind is ClientKind.Minimal)
            {
                editorDevice ??= EditorDevice.GetDefaultAndroidDevice();
                var client = networkClient == null
                    ? new MinimalMagifyClient(AppName, ConfigPath, EditorModeTestsEnvironment.RootStoragePath, OfflineEnvironment, editorDevice)
                    : new MinimalMagifyClient(AppName, ConfigPath, EditorModeTestsEnvironment.RootStoragePath, OfflineEnvironment, editorDevice, networkClient);
                client.InitializeSdk();
                client.Setup();
                return client;
            }
            throw new System.ArgumentOutOfRangeException();
        }

        private static MagifyClient CreateWithDeviceAndInit(EditorDevice editorDevice)
        {
            var client = new MagifyClient(AppName, ConfigPath, EditorModeTestsEnvironment.RootStoragePath, EditorModeTestsEnvironment.DefaultClientStateConfig, OfflineEnvironment, editorDevice);
            client.InitializeSdk();
            client.Setup();
            return client;
        }

        [NotNull]
        private static CampaignsTracker GetCampaignsTracker([NotNull] IMagifyClient magifyClient)
        {
            return GetPrivateField<CampaignsTracker>(magifyClient, "_campaignsTracker");
        }

        [NotNull]
        private static IAnalyticsTracker GetAnalyticsTracker([NotNull] IMagifyClient magifyClient)
        {
            return GetPrivateField<IAnalyticsTracker>(magifyClient, "_analyticsTracker");
        }

        [NotNull]
        private static PlatformAPI GetPlatformAPI([NotNull] IMagifyClient magifyClient)
        {
            return GetPrivateField<PlatformAPI>(magifyClient, "_platform");
        }

        [NotNull]
        private static GeneralPrefs GetGeneralPrefs([NotNull] IMagifyClient magifyClient)
        {
            return GetPrivateField<GeneralPrefs>(magifyClient, "_generalPrefs");
        }

        [NotNull]
        private static SessionCounter GetSessionCounter([NotNull] IMagifyClient magifyClient)
        {
            return GetPrivateField<SessionCounter>(magifyClient, "_sessionCounter");
        }

		[NotNull]
        private static ServerApi GetServerApi([NotNull] IMagifyClient magifyClient)
        {
            return GetPrivateField<ServerApi>(magifyClient, "_serverApi");
        }

        [NotNull]
        private static ContextApplicator GetContextApplicator([NotNull] IMagifyClient magifyClient)
        {
            return GetPrivateField<ContextApplicator>(magifyClient, "_contextApplicator");
        }

		[NotNull]
        private static T GetPrivateField<T>([NotNull] IMagifyClient client, [NotNull] string fieldName)
            where T : class
        {
            return (T)client
                    .GetType()
                    .GetField(fieldName, System.Reflection.BindingFlags.NonPublic | System.Reflection.BindingFlags.Instance)
                    !.GetValue(client)!;
        }
    }
}