using System.Collections.Generic;
using FluentAssertions;
using NUnit.Framework;

namespace Magify.Tests
{
    public class IsCorrespondTests
    {
        [Test]
        public void IsCorrespondedPurchaseLimits_AndAllLimitsEmpty_ThenShouldReturnTrue()
        {
            // Arrange
            var purchaseLimits = new PurchaseLimits
            {
                IncludeIfAll = new HashSet<string>(),
                IncludeIfAny = new HashSet<string>(),
                ExcludeIfAll = new HashSet<string>(),
                ExcludeIfAny = new HashSet<string>()
            };
            var purchase = new HashSet<string> { "1", "2" };

            // Act.
            // Assert
            purchaseLimits.IsCorresponded(purchase).Should().BeTrue();
        }

        [Test]
        public void IsCorrespondedPurchaseLimits_AndIncludeIfAllSameWithPurchase_ThenShouldReturnTrue()
        {
            // Arrange
            var purchaseLimits = new PurchaseLimits
            {
                IncludeIfAll = new HashSet<string> { "1", "2" },
                IncludeIfAny = new HashSet<string>(),
                ExcludeIfAll = null,
                ExcludeIfAny = new HashSet<string>()
            };
            var purchase = new HashSet<string> { "1", "2" };

            // Act.
            // Assert
            purchaseLimits.IsCorresponded(purchase).Should().BeTrue();
        }

        [Test]
        public void IsCorrespondedPurchaseLimits_AndIncludeIfAnyHasSomeInPurchase_ThenShouldReturnTrue()
        {
            // Arrange
            var purchaseLimits = new PurchaseLimits
            {
                IncludeIfAll = new HashSet<string>(),
                IncludeIfAny = new HashSet<string> { "1", "2" },
                ExcludeIfAll = new HashSet<string>(),
                ExcludeIfAny = null
            };
            var purchase = new HashSet<string> { "1" };

            // Act.
            // Assert
            purchaseLimits.IsCorresponded(purchase).Should().BeTrue();
        }

        [Test]
        public void IsCorrespondedPurchaseLimits_AndExcludeIfAllSameInPurchase_ThenShouldReturnFalse()
        {
            // Arrange
            var purchaseLimits = new PurchaseLimits
            {
                IncludeIfAll = null,
                IncludeIfAny = new HashSet<string>(),
                ExcludeIfAll = new HashSet<string> { "1", "2" },
                ExcludeIfAny = new HashSet<string>()
            };
            var purchase = new HashSet<string> { "1", "2" };

            // Act.
            // Assert
            purchaseLimits.IsCorresponded(purchase).Should().BeFalse();
        }

        [Test]
        public void IsCorrespondedPurchaseLimits_AndExcludeIfAnyHasSomeInPurchase_ThenShouldReturnFalse()
        {
            // Arrange
            var purchaseLimits = new PurchaseLimits
            {
                IncludeIfAll = new HashSet<string>(),
                IncludeIfAny = null,
                ExcludeIfAll = new HashSet<string>(),
                ExcludeIfAny = new HashSet<string> { "1", "2" }
            };
            var purchase = new HashSet<string> { "1" };

            // Act.
            // Assert
            purchaseLimits.IsCorresponded(purchase).Should().BeFalse();
        }
    }
}