﻿using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;

namespace Magify.Tests
{
    internal partial class ContextListenerTests
    {
        internal class ContentProviderTests
        {
            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContext_ThenContentShouldBeSameContentCount(ContextKind kind)
            {
                //Arrange
                var contentProvider = new ContentProvider();
                var context = CreateCampaignsContext();

                //Act
                ((IContextListener)contentProvider).UpdateContext(context, kind);

                //Assert
                contentProvider.Content!.Count.Should()!.Be(context.Content?.Count);
                contentProvider.Content!.Select(c => c.Key).ToArray()
                    .ShouldBeSameInAnyOrder(context.Content?.Select(c => c.Key).ToArray());
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenResetContent_ThenContentShouldBeNull(ContextKind kind)
            {
                //Arrange
                var contentProvider = new ContentProvider();
                var context = CreateCampaignsContext();

                //Act
                ((IContextListener)contentProvider).UpdateContext(context, kind);
                contentProvider.ResetContentMap();

                //Assert
                contentProvider.Content!.Should()!.BeNull();
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContextWithEmptyContent_ThenContentShouldBeNull(ContextKind kind)
            {
                //Arrange
                var contentProvider = new ContentProvider();
                var context = new CampaignsContext();

                //Act
                ((IContextListener)contentProvider).UpdateContext(context, kind);

                //Assert
                contentProvider.Content!.Should()!.BeNull();
            }

            [Test]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContextWhenContentAlreadyExist_ThenContentShouldBeNew(ContextKind kind)
            {
                //Arrange
                var contentProvider = new ContentProvider();
                var firstContext = CreateCampaignsContext();
                var secondContext = CreateCampaignsContext();

                //Act
                ((IContextListener)contentProvider).UpdateContext(firstContext, kind);
                var firstContent = contentProvider.Content;
                ((IContextListener)contentProvider).UpdateContext(secondContext, kind);
                var secondContent = contentProvider.Content;

                //Assert
                firstContent.Should()!.NotEqual(secondContent);
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContextMultiplyTimes_ThenUpdatedContentShouldBeSameAsContext(ContextKind kind)
            {
                //Arrange
                var contentProvider = new ContentProvider();
                var firstContext = CreateCampaignsContext();
                var secondContext = CreateCampaignsContext();

                //Act
                ((IContextListener)contentProvider).UpdateContext(firstContext, kind);
                var firstContent = contentProvider.Content;
                ((IContextListener)contentProvider).UpdateContext(secondContext, kind);
                var secondContent = contentProvider.Content;

                //Assert
                firstContent!.Select(c => c.Key).ToArray()
                    .ShouldBeSameInAnyOrder(firstContext.Content?.Select(c => c.Key).ToArray());

                secondContent!.Select(c => c.Key).ToArray()
                    .ShouldBeSameInAnyOrder(secondContext.Content?.Select(c => c.Key).ToArray());
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public async Task WhenUpdateContextOnThreadPool_ThenUpdateContextShouldBeThreadSafety(ContextKind kind)
            {
                //Arrange
                var listener = new ContentProvider();
                var firstContext = CreateCampaignsContext();
                var secondContext = CreateCampaignsContext();
                Dictionary<string, ContentGroup> firstUpdateContent = default;
                Dictionary<string, ContentGroup> secondUpdateContent = default;
                var barrier = new Barrier(2);

                //Act
                var firstTask = UniTask.RunOnThreadPool(() =>
                {
                    barrier.SignalAndWait();
                    ((IContextListener)listener).UpdateContext(firstContext, kind);
                    firstUpdateContent = firstContext.Content;
                });

                var secondTask = UniTask.RunOnThreadPool(() =>
                {
                    barrier.SignalAndWait();
                    ((IContextListener)listener).UpdateContext(secondContext, kind);
                    secondUpdateContent = secondContext.Content;
                });

                await UniTask.WhenAll(firstTask, secondTask);

                //Assert
                firstUpdateContent.Count.Should()!.Be(firstContext.Content?.Count);
                firstUpdateContent.Select(c => c.Key).ToArray()
                    .ShouldBeSameInAnyOrder(firstContext.Content?.Select(c => c.Key).ToArray());

                secondUpdateContent.Count.Should()!.Be(secondContext.Content?.Count);
                secondUpdateContent.Select(c => c.Key).ToArray()
                    .ShouldBeSameInAnyOrder(secondContext.Content?.Select(c => c.Key).ToArray());
            }
        }
    }
}