﻿using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using NUnit.Framework;

namespace Magify.Tests
{
    internal class CompositeCancellationTokenSourceTests
    {
        [Test]
        public void WhenCancellationTokenSourceCanceled_ThenCompositeCancellationTokenSourceShouldBeCancelled()
        {
            //Arrange
            using var cts = new CancellationTokenSource();
            using var compositeCancellationTokenSource = new CompositeCancellationTokenSource(cts.Token);

            //Act
            cts.Cancel();

            //Assert
            compositeCancellationTokenSource.IsCancellationRequested.Should()!.BeTrue();
        }

        [Test]
        public void WhenEmptyCompositeCancellationTokenSourceCreated_ThenCompositeCancellationTokenSourceShouldNotBeCancelled()
        {
            //Arrange
            using var compositeCancellationTokenSource = new CompositeCancellationTokenSource();

            //Act

            //Assert
            compositeCancellationTokenSource.IsCancellationRequested.Should()!.BeFalse();
        }

        [Test]
        [TestCase(2)]
        [TestCase(5)]
        [TestCase(10)]
        public void WhenManyCancellationTokenSourceCanceled_ThenCompositeCancellationTokenSourceShouldBeCancelled(int tokenCounts)
        {
            //Arrange
            using var compositeCancellationTokenSource = new CompositeCancellationTokenSource();
            var list = new List<CancellationTokenSource>();
            tokenCounts.EnumerateEach().Select(_ => new CancellationTokenSource()).ForEach(cts =>
            {
                list.Add(cts);
                // ReSharper disable once AccessToDisposedClosure
                compositeCancellationTokenSource.Register(cts!.Token);
            });

            //Act
            list.ForEach(cts => cts!.Cancel());

            //Assert
            compositeCancellationTokenSource.IsCancellationRequested.Should()!.BeTrue();
            list.ForEach(cts => cts!.Dispose());
        }

        [Test]
        [TestCase(2)]
        [TestCase(100)]
        [TestCase(5000)]
        [SuppressMessage("ReSharper", "AccessToDisposedClosure")]
        public async Task WhenManyCancellationTokenSourceCanceledOnDifferentThreads_ThenCompositeCancellationTokenSourceShouldBeCancelled(int tokenCounts)
        {
            //Arrange
            using var mainCts = new CancellationTokenSource(1000);
            using var compositeCancellationTokenSource = new CompositeCancellationTokenSource();
            var list = new List<CancellationTokenSource>();
            tokenCounts.EnumerateEach().Select(_ => new CancellationTokenSource()).ForEach(list.Add);
            await UniTask.WhenAll(list.Select(cts =>
            {
                return UniTask.RunOnThreadPool(() => compositeCancellationTokenSource.Register(cts!.Token), cancellationToken: mainCts.Token);
            }));

            //Act
            await UniTask.WhenAll(list.Select(cts => UniTask.RunOnThreadPool(() => cts!.Cancel(), cancellationToken: mainCts.Token)));

            //Assert
            compositeCancellationTokenSource.IsCancellationRequested.Should()!.BeTrue();
            list.ForEach(cts => cts!.Dispose());
        }

        [Test]
        [TestCase(2)]
        [TestCase(10)]
        [TestCase(100)]
        public void WhenCompositeCancellationTokenSourceCanceled_ThenCompositeCancellationTokenSourceShouldBeCancelled(int tokenCounts)
        {
            //Arrange
            using var compositeCancellationTokenSource = new CompositeCancellationTokenSource();
            var list = new List<CancellationTokenSource>();
            tokenCounts.EnumerateEach().Select(_ => new CancellationTokenSource()).ForEach(cts =>
            {
                list.Add(cts);
                // ReSharper disable once AccessToDisposedClosure
                compositeCancellationTokenSource.Register(cts!.Token);
            });

            //Act
            compositeCancellationTokenSource.Cancel();

            //Assert
            compositeCancellationTokenSource.IsCancellationRequested.Should()!.BeTrue();
            list.ForEach(cts => cts!.Dispose());
        }

        [Test]
        [TestCase(10, false)]
        [TestCase(50, false)]
        [TestCase(100, false)]
        [TestCase(10, true)]
        [TestCase(50, true)]
        [TestCase(100, true)]
        public async Task WhenCompositeCancellationTokenSourceCanceledWithDelay_ThenCompositeCancellationTokenSourceShouldBeCancelled(int delay, bool useTimeSpan)
        {
            //Arrange
            using var cts = new CancellationTokenSource();
            using var compositeCancellationTokenSource = new CompositeCancellationTokenSource(cts.Token);

            //Act
            if (useTimeSpan)
            {
                compositeCancellationTokenSource.CancelAfter(TimeSpan.FromMilliseconds(delay));
            }
            else
            {
                compositeCancellationTokenSource.CancelAfter(delay);
            }

            await UniTask.Delay(delay * 2, cancellationToken: new CancellationTokenSource().Token);
            await UniTask.Yield();

            //Assert
            compositeCancellationTokenSource.IsCancellationRequested.Should()!.BeTrue();
        }

        [Test]
        public void WhenCompositeCancellationTokenSourceDispose_ThenCompositeCancellationTokenSourceShouldBeCancelled()
        {
            //Arrange
            var exceptionCount = 0;
            using var cts = new CancellationTokenSource();
            var compositeCancellationTokenSource = new CompositeCancellationTokenSource(cts.Token);

            //Act
            compositeCancellationTokenSource.Dispose();

            try
            {
                compositeCancellationTokenSource.Register(new CancellationTokenSource().Token);
            }
            catch (Exception)
            {
                exceptionCount++;
            }

            try
            {
                compositeCancellationTokenSource.Cancel();
            }
            catch (Exception)
            {
                exceptionCount++;
            }

            try
            {
                compositeCancellationTokenSource.CancelAfter(10);
            }
            catch (Exception)
            {
                exceptionCount++;
            }

            try
            {
                compositeCancellationTokenSource.CancelAfter(TimeSpan.FromMilliseconds(10));
            }
            catch (Exception)
            {
                exceptionCount++;
            }

            //Assert
            exceptionCount.Should()!.Be(4);
        }
    }
}