﻿using System.Collections.Generic;
using System.Linq;
using FluentAssertions;
using JetBrains.Annotations;
using Magify.Model;
using NUnit.Framework;
using UnityEngine;

namespace Magify.Tests
{
    internal class AppStateEncoderTests
    {
        [NotNull]
        private static object[] AppStateTestCases()
        {
            return new object[]
            {
                new ApplicationState
                {
                    ClientState = "client state, hope that everything is ok",
                    SdkByPlatform = new Dictionary<RuntimePlatform, ApplicationState.PlatformSdk>()
                    {
                        { RuntimePlatform.Android, new ApplicationState.PlatformSdk(
                            "general prefs, hope that everything is ok",
                            "counters prefs, hope that everything is ok") },
                    },
                },
                new ApplicationState
                {
                    ClientState = null,
                    SdkByPlatform = new Dictionary<RuntimePlatform, ApplicationState.PlatformSdk>()
                    {
                        { RuntimePlatform.Android, new ApplicationState.PlatformSdk(null, null) },
                    },
                },
                new ApplicationState
                {
                    ClientState = string.Empty,
                    SdkByPlatform = new Dictionary<RuntimePlatform, ApplicationState.PlatformSdk>()
                    {
                        { RuntimePlatform.Android, new ApplicationState.PlatformSdk(string.Empty, string.Empty) },
                    },
                },
            }.SelectMany(s =>
            {
                return new object[]
                {
                    new[] { s, new ToJsonToBase64MagifyEncoder(), EncodingMode.ToJsonToBase64 },
                    new[] { s, new ToJsonToZipToBase64MagifyEncoder(), EncodingMode.ToJsonToZipToBase64 },
                };
            }).ToArray();
        }

        [Test]
        [TestCaseSource(nameof(AppStateTestCases))]
        public void Encode_ThenNotNull([NotNull] ApplicationState state, [NotNull] IMagifyEncoder encoder, EncodingMode encodingMode)
        {
            // Arrange
            var prefix = encodingMode.ToPrefixString();

            // Act
            var result = encoder.Encode(state, prefix);

            // Assert
            result.Should()!.NotBeNullOrEmpty();
        }

        [Test]
        [TestCaseSource(nameof(AppStateTestCases))]
        public void Encode_ThenDecodeNotNull([NotNull] ApplicationState state, [NotNull] IMagifyEncoder encoder, EncodingMode encodingMode)
        {
            // Arrange
            var prefix = encodingMode.ToPrefixString();
            var encoded = encoder.Encode(state, prefix);

            // Act
            var result = encoder.Decode<ApplicationState>(encoded, EncodingModeUtils.EncodingModePrefixLength);

            // Assert
            result.Should()!.NotBeNull();
        }

        [Test]
        [TestCaseSource(nameof(AppStateTestCases))]
        public void Encode_ThenDecode_ThenFieldsEquals([NotNull] ApplicationState state, [NotNull] IMagifyEncoder encoder, EncodingMode encodingMode)
        {
            // Arrange
            var prefix = encodingMode.ToPrefixString();
            var encoded = encoder.Encode(state, prefix);

            // Act
            var result = encoder.Decode<ApplicationState>(encoded, EncodingModeUtils.EncodingModePrefixLength);

            // Assert
            result.Should()!.NotBeNull();
            result!.ClientState!.Should()!.Be(state.ClientState);
            result!.SdkByPlatform![RuntimePlatform.Android]!.GeneralPrefs!.Should()!.Be(state.SdkByPlatform![RuntimePlatform.Android]!.GeneralPrefs);
            result!.SdkByPlatform![RuntimePlatform.Android]!.CountersPrefs!.Should()!.Be(state.SdkByPlatform![RuntimePlatform.Android]!.CountersPrefs);
        }
    }
}