using System;
using System.Linq;
using FluentAssertions;
using NUnit.Framework;
using UnityEngine;

namespace Magify.Tests
{
    public class BinaryStorageTests
    {
        public static ByteEnum[] ByteEnumInputs => Enum.GetValues(typeof(ByteEnum)).Cast<ByteEnum>().ToArray();
        public static SByteEnum[] SByteEnumInputs => Enum.GetValues(typeof(SByteEnum)).Cast<SByteEnum>().ToArray();
        public static ShortEnum[] ShortEnumInputs => Enum.GetValues(typeof(ShortEnum)).Cast<ShortEnum>().ToArray();
        public static UShortEnum[] UShortEnumInputs => Enum.GetValues(typeof(UShortEnum)).Cast<UShortEnum>().ToArray();
        public static IntEnum[] IntEnumInputs => Enum.GetValues(typeof(IntEnum)).Cast<IntEnum>().ToArray();
        public static UIntEnum[] UIntEnumInputs => Enum.GetValues(typeof(UIntEnum)).Cast<UIntEnum>().ToArray();
        public static LongEnum[] LongEnumInputs => Enum.GetValues(typeof(LongEnum)).Cast<LongEnum>().ToArray();
        public static ULongEnum[] ULongEnumInputs => Enum.GetValues(typeof(ULongEnum)).Cast<ULongEnum>().ToArray();

        public static DateTime[] DateTimeInputs => new[]
        {
            DateTime.Now, DateTime.UnixEpoch, new DateTime(1, 4, 1, 1, 1, 1), new DateTime(31, 1, 12)
        };

        public static Vector2[] Vector2Inputs => new[]
        {
            Vector2.left, Vector2.right, Vector2.down, Vector2.up, Vector2.zero, Vector2.negativeInfinity, Vector2.one,
        };

        public static Vector3[] Vector3Inputs => new[]
        {
            Vector3.left, Vector3.right, Vector3.down, Vector3.up, Vector3.zero, Vector3.negativeInfinity, Vector3.one,
        };

        [SetUp, TearDown]
        public void CleanStorageBetweenTests()
        {
            EditorModeTestsEnvironment.Clear();
        }

        [Test]
        public void WhenStorageCreated_AndNothingElse_ThenAllStandardTypesSupported()
        {
            // Arrange
            using var storage = BinaryStorage.Get(EditorModeTestsEnvironment.TempStoragePath);

            // Assert
            storage.Supports<bool>().Should().Be(true);
            storage.Supports<char>().Should().Be(true);
            storage.Supports<byte>().Should().Be(true);
            storage.Supports<sbyte>().Should().Be(true);
            storage.Supports<short>().Should().Be(true);
            storage.Supports<ushort>().Should().Be(true);
            storage.Supports<uint>().Should().Be(true);
            storage.Supports<int>().Should().Be(true);
            storage.Supports<long>().Should().Be(true);
            storage.Supports<ulong>().Should().Be(true);
            storage.Supports<float>().Should().Be(true);
            storage.Supports<double>().Should().Be(true);
            storage.Supports<decimal>().Should().Be(true);
            storage.Supports<string>().Should().Be(true);
            storage.Supports<DateTime>().Should().Be(true);
            storage.Supports<Vector2>().Should().Be(true);
            storage.Supports<Vector3>().Should().Be(true);
        }

        [TestCase(true)]
        [TestCase(false)]
        public void WhenBinaryBoolean_AndValueChanged_ThenDoGeneralSectionCheck(bool value)
        {
            DoGeneralStorageSectionCheck(value);
        }

        [TestCase('c')]
        [TestCase('$')]
        [TestCase('\0')]
        [TestCase('里')]
        public void WhenCharSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(char value)
        {
            DoGeneralStorageSectionCheck('c');
        }

        [TestCase(null)] // null
        [TestCase("")] // empty
        [TestCase("Hello world!")] // latin
        [TestCase("Прывітанне сусвет!")] // cyrillic
        [TestCase("你好世界")] // chinese
        [TestCase("مرحبا بالعالم!")] // arabic
        public void WhenStringSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(string value)
        {
            DoGeneralStorageSectionCheck(value);
        }

        [TestCase(byte.MinValue)]
        [TestCase(byte.MaxValue)]
        public void WhenByteSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(byte value)
        {
            DoGeneralStorageSectionCheck(value);
            DoGeneralStorageNullableSectionCheck<byte>(null);
            DoGeneralStorageNullableSectionCheck<byte>(value);
        }

        [TestCase(sbyte.MinValue)]
        [TestCase(0)]
        [TestCase(sbyte.MaxValue)]
        public void WhenSByteSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(sbyte value)
        {
            DoGeneralStorageSectionCheck(value);
            DoGeneralStorageNullableSectionCheck<sbyte>(null);
            DoGeneralStorageNullableSectionCheck<sbyte>(value);
        }

        [TestCase(short.MinValue)]
        [TestCase(0)]
        [TestCase(short.MaxValue)]
        public void WhenShortSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(short value)
        {
            DoGeneralStorageSectionCheck(value);
            DoGeneralStorageNullableSectionCheck<short>(null);
            DoGeneralStorageNullableSectionCheck<short>(value);
        }

        [TestCase(ushort.MinValue)]
        [TestCase(ushort.MaxValue)]
        public void WhenUShortSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(ushort value)
        {
            DoGeneralStorageSectionCheck(value);
            DoGeneralStorageNullableSectionCheck<ushort>(null);
            DoGeneralStorageNullableSectionCheck<ushort>(value);
        }

        [TestCase(int.MinValue)]
        [TestCase(0)]
        [TestCase(int.MaxValue)]
        public void WhenInt32SectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(int value)
        {
            DoGeneralStorageSectionCheck(value);
            DoGeneralStorageNullableSectionCheck<int>(null);
            DoGeneralStorageNullableSectionCheck<int>(value);
        }

        [TestCase(uint.MinValue)]
        [TestCase(uint.MaxValue / 2)]
        [TestCase(uint.MaxValue)]
        public void WhenUInt32SectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(uint value)
        {
            DoGeneralStorageSectionCheck(value);
            DoGeneralStorageNullableSectionCheck<uint>(null);
            DoGeneralStorageNullableSectionCheck<uint>(value);
        }

        [TestCase(long.MinValue)]
        [TestCase(0)]
        [TestCase(long.MaxValue)]
        public void WhenInt64SectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(long value)
        {
            DoGeneralStorageSectionCheck(value);
            DoGeneralStorageNullableSectionCheck<long>(null);
            DoGeneralStorageNullableSectionCheck<long>(value);
        }

        [TestCase(ulong.MinValue)]
        [TestCase(ulong.MaxValue)]
        public void WhenUInt64SectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(ulong value)
        {
            DoGeneralStorageSectionCheck(value);
            DoGeneralStorageNullableSectionCheck<ulong>(null);
            DoGeneralStorageNullableSectionCheck<ulong>(value);
        }

        [TestCase(float.MinValue)]
        [TestCase(0f)]
        [TestCase(float.MaxValue)]
        public void WhenSingleSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(float value)
        {
            DoGeneralStorageSectionCheck(value);
            DoGeneralStorageNullableSectionCheck<float>(null);
            DoGeneralStorageNullableSectionCheck<float>(value);
        }

        [TestCase(double.MinValue)]
        [TestCase(0.0)]
        [TestCase(double.MaxValue)]
        public void WhenDoubleSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(double value)
        {
            DoGeneralStorageSectionCheck(value);
            DoGeneralStorageNullableSectionCheck<double>(null);
            DoGeneralStorageNullableSectionCheck<double>(value);
        }

        [TestCase(int.MinValue)]
        [TestCase(0.0)]
        [TestCase(int.MaxValue)]
        public void WhenDecimalSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(decimal value)
        {
            DoGeneralStorageSectionCheck(value);
            DoGeneralStorageNullableSectionCheck<decimal>(null);
            DoGeneralStorageNullableSectionCheck<decimal>(value);
        }

        [TestCaseSource(nameof(DateTimeInputs))]
        public void WhenDateTimeSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(DateTime value)
        {
            DoGeneralStorageSectionCheck(value);
            DoGeneralStorageNullableSectionCheck<DateTime>(null);
            DoGeneralStorageNullableSectionCheck<DateTime>(value);
        }

        [TestCaseSource(nameof(Vector2Inputs))]
        public void WhenVector2SectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(Vector2 value)
        {
            DoGeneralStorageSectionCheck(value);
            DoGeneralStorageNullableSectionCheck<Vector2>(null);
            DoGeneralStorageNullableSectionCheck<Vector2>(value);
        }

        [TestCaseSource(nameof(Vector3Inputs))]
        public void WhenVector3SectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(Vector3 value)
        {
            DoGeneralStorageSectionCheck(value);
            DoGeneralStorageNullableSectionCheck<Vector3>(null);
            DoGeneralStorageNullableSectionCheck<Vector3>(value);
        }

        [TestCaseSource(nameof(ByteEnumInputs))]
        public void WhenByteEnumInputsSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(ByteEnum value)
        {
            DoGeneralStorageSectionCheckForEnum(value);
        }

        [TestCaseSource(nameof(SByteEnumInputs))]
        public void WhenSByteEnumSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(SByteEnum value)
        {
            DoGeneralStorageSectionCheckForEnum(value);
        }

        [TestCaseSource(nameof(ShortEnumInputs))]
        public void WhenShortEnumInputsSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(ShortEnum value)
        {
            DoGeneralStorageSectionCheckForEnum(value);
        }

        [TestCaseSource(nameof(UShortEnumInputs))]
        public void WhenUShortEnumSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(UShortEnum value)
        {
            DoGeneralStorageSectionCheckForEnum(value);
        }

        [TestCaseSource(nameof(IntEnumInputs))]
        public void WhenIntEnumInputsSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(IntEnum value)
        {
            DoGeneralStorageSectionCheckForEnum(value);
        }

        [TestCaseSource(nameof(UIntEnumInputs))]
        public void WhenUIntEnumSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(UIntEnum value)
        {
            DoGeneralStorageSectionCheckForEnum(value);
        }

        [TestCaseSource(nameof(LongEnumInputs))]
        public void WhenLongEnumInputsSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(LongEnum value)
        {
            DoGeneralStorageSectionCheckForEnum(value);
        }

        [TestCaseSource(nameof(ULongEnumInputs))]
        public void WhenULongEnumSectionUsed_AndValueChanged_ThenDoGeneralSectionCheck(ULongEnum value)
        {
            DoGeneralStorageSectionCheckForEnum(value);
        }

        [Test]
        public void WhenStorageSupportsTwoEnumsWithSameName_AndNamespacesAreDifferent_ThenDataIsNotCorrupting()
        {
            // Arrange
            using (var storage = BinaryStorage
                       .Construct(EditorModeTestsEnvironment.TempStoragePath)
                       .SupportEnum<ScreenOrientation>()
                       .SupportEnum<UnityEngine.ScreenOrientation>(true)
                       .Build())
            {
                storage.Set("key", UnityEngine.ScreenOrientation.LandscapeLeft);
                storage.Set("key", ScreenOrientation.Value2);
            }

            // Act
            // Assert
            using (var storage = BinaryStorage
                       .Construct(EditorModeTestsEnvironment.TempStoragePath)
                       .SupportEnum<ScreenOrientation>()
                       .SupportEnum<UnityEngine.ScreenOrientation>(true)
                       .Build())
            {
                storage.Get<UnityEngine.ScreenOrientation>("key").Should().Be(UnityEngine.ScreenOrientation.LandscapeLeft);
                storage.Get<ScreenOrientation>("key").Should().Be(ScreenOrientation.Value2);
            }
        }

        [Test]
        public void WhenStorageChanged_AndReloaded_ThenAllDataAreValid()
        {
            // Arrange
            using (var storage = BinaryStorage
                       .Construct(EditorModeTestsEnvironment.TempStoragePath)
                       .AddEntry(BinaryTypeString.Shared)
                       .AddEntry(BinaryTypeBoolean.Shared)
                       .AddEntry(BinaryTypeInt64.Shared)
                       .Build())
            {
                storage.Set("key", "value");
                storage.Set("key", true);
                storage.Set("key", 60L);
            }

            // Act
            // Assert
            using (var storage = BinaryStorage
                       .Construct(EditorModeTestsEnvironment.TempStoragePath)
                       .AddEntry(BinaryTypeString.Shared)
                       .AddEntry(BinaryTypeBoolean.Shared)
                       .AddEntry(BinaryTypeInt64.Shared)
                       .Build())
            {
                storage.Has<string>("key").Should().BeTrue();
                storage.Has<bool>("key").Should().BeTrue();
                storage.Has<long>("key").Should().BeTrue();

                storage.Get<string>("key").Should().Be("value");
                storage.Get<bool>("key").Should().Be(true);
                storage.Get<long>("key").Should().Be(60L);
            }
        }

        [Test]
        public void WhenStorageHasObsoleteSection_AndStorageLoaded_ThenObsoleteSectionIgnored()
        {
            // Arrange
            using (var storage = BinaryStorage
                       .Construct(EditorModeTestsEnvironment.TempStoragePath)
                       .AddEntry(BinaryTypeString.Shared)
                       .AddEntry(BinaryTypeBoolean.Shared)
                       .AddEntry(BinaryTypeInt64.Shared)
                       .Build())
            {
                storage.Set("key", "value");
                storage.Set("key", true);
                storage.Set("key", 60L);
            }

            // Act
            // Assert
            using (var storage = BinaryStorage
                       .Construct(EditorModeTestsEnvironment.TempStoragePath)
                       .AddEntry(BinaryTypeBoolean.Shared)
                       .AddEntry(BinaryTypeInt64.Shared)
                       .Build())
            {
                storage.Supports<string>().Should().BeFalse();
                storage.Has<bool>("key").Should().BeTrue();
                storage.Has<long>("key").Should().BeTrue();
            }
        }

        [Test]
        public void WhenStorageHasFewKeys_AndResetAllCalled_ThenAllDataHasBeenErased()
        {
            // Arrange
            using var storage = BinaryStorage
                .Construct(EditorModeTestsEnvironment.TempStoragePath)
                .AddEntry(BinaryTypeInt32.Shared)
                .AddEntry(BinaryTypeString.Shared)
                .Build();
            storage.Set("key", 10);
            storage.Set("key", "value");

            // Act
            storage.ResetAll();

            // Assert
            storage.Has<int>("key").Should().Be(false);
            storage.Has<string>("key").Should().Be(false);
        }

        [Test]
        public void WhenStorageHasFewKeys_AndResetOnlyStringsAllCalled_ThenOnlyStringErased()
        {
            // Arrange
            using var storage = BinaryStorage
                .Construct(EditorModeTestsEnvironment.TempStoragePath)
                .AddEntry(BinaryTypeInt32.Shared)
                .AddEntry(BinaryTypeString.Shared)
                .Build();
            storage.Set("key", 10);
            storage.Set("key", "value");

            // Act
            storage.ResetOnly<string>();

            // Assert
            storage.Has<int>("key").Should().Be(true);
            storage.Has<string>("key").Should().Be(false);
        }

        [Test]
        public void WhenStorageHasFewKeys_AndResetAllWithPredicateCalled_ThenRemoveOnlyPredictedKeys()
        {
            // Arrange
            using var storage = BinaryStorage
                .Construct(EditorModeTestsEnvironment.TempStoragePath)
                .AddEntry(BinaryTypeInt32.Shared)
                .Build();
            storage.Set("prefix1.key1", 11);
            storage.Set("prefix1.key2", 12);
            storage.Set("prefix2.key1", 21);
            storage.Set("prefix2.key2", 22);

            // Act
            storage.Remove<int>(key => key.StartsWith("prefix1"));

            // Assert
            storage.Has<int>("prefix1.key1").Should().Be(false);
            storage.Has<int>("prefix1.key2").Should().Be(false);
            storage.Has<int>("prefix2.key1").Should().Be(true);
            storage.Has<int>("prefix2.key2").Should().Be(true);
        }

        [Test]
        public void WhenStorageHasValue_AndGetReactiveProperty_ThenReactivePropertyValueEqualsToStorageValue()
        {
            // Arrange
            using var storage = BinaryStorage
                .Construct(EditorModeTestsEnvironment.TempStoragePath)
                .AddEntry(BinaryTypeInt32.Shared)
                .Build();

            storage.Set("key", 10);

            // Act
            var prop = storage.GetReactiveProperty<int>("key");

            // Assert
            prop.Value.Should().Be(10);
        }

        [Test]
        public void WhenGetReactivePropertyBeforeStorageSet_AndChangeStorageValue_ThenReactivePropertyValueEqualsToStorageValue()
        {
            // Arrange
            using var storage = BinaryStorage
                .Construct(EditorModeTestsEnvironment.TempStoragePath)
                .AddEntry(BinaryTypeInt32.Shared)
                .Build();

            var prop = storage.GetReactiveProperty<int>("key");

            // Act
            storage.Set("key", 10);

            // Assert
            prop.Value.Should().Be(10);
        }

        [Test]
        public void WhenGetReactiveProperty_AndChangeItsValue_ThenStorageValueEqualsToReactiveProperty()
        {
            // Arrange
            using var storage = BinaryStorage
                .Construct(EditorModeTestsEnvironment.TempStoragePath)
                .AddEntry(BinaryTypeInt32.Shared)
                .Build();

            var prop = storage.GetReactiveProperty<int>("key");

            // Act
            prop.Value = 10;

            // Assert
            storage.Get<int>("key").Should().Be(10);
        }

        [Test]
        public void WhenGetReactiveProperty_AndValueRemovedByKey_ThenReactivePropertyIsDefault()
        {
            // Arrange
            using var storage = BinaryStorage
                .Construct(EditorModeTestsEnvironment.TempStoragePath)
                .AddEntry(BinaryTypeInt32.Shared)
                .Build();

            storage.Set("key", 10);
            var prop = storage.GetReactiveProperty<int>("key");

            // Act
            storage.Remove<int>("key");

            // Assert
            prop.Value.Should().Be(default);
        }

        [Test]
        public void WhenReactivePropertyChanged_AndStorageReloaded_ThenPropertyValueIsCorrect()
        {
            // Arrange
            using (var storage = BinaryStorage
                       .Construct(EditorModeTestsEnvironment.TempStoragePath)
                       .AddEntry(BinaryTypeInt32.Shared)
                       .Build())
            {
                storage.GetReactiveProperty<int>("key1").Value = 10;
                storage.GetReactiveProperty<int>("key2", 10);
            }

            // Act
            // Assert
            using (var storage = BinaryStorage
                       .Construct(EditorModeTestsEnvironment.TempStoragePath)
                       .AddEntry(BinaryTypeInt32.Shared)
                       .Build())
            {
                storage.Has<int>("key1").Should().BeTrue();
                storage.Has<int>("key2").Should().BeTrue();
                storage.Get<int>("key1").Should().Be(10);
                storage.Get<int>("key2").Should().Be(10);
            }
        }

        private void DoGeneralStorageSectionCheck<T>(T defaultValue)
        {
            // Arrange
            using var storage = BinaryStorage.Get(EditorModeTestsEnvironment.TempStoragePath);

            // Act
            storage.Supports<T>().Should().Be(true);
            storage.Set<T>("key", defaultValue);
            storage.LoadDataFromDisk();

            // Assert
            storage.Has<T>("key").Should().Be(true);
            storage.Get<T>("key").Should().Be(defaultValue);

            storage.Remove<T>("key").Should().Be(true);
            storage.Has<T>("key").Should().Be(false);
            storage.Get<T>("key").Should().Be(default(T));
        }

        private void DoGeneralStorageNullableSectionCheck<T>(T? defaultValue)
            where T: struct
        {
            // Arrange
            using var storage = BinaryStorage
                .Construct(EditorModeTestsEnvironment.TempStoragePath)
                .SupportNullable<T>()
                .Build();

            // Act
            storage.Supports<T?>().Should().Be(true);
            storage.Set<T?>("key", defaultValue);
            storage.LoadDataFromDisk();

            // Assert
            storage.Has<T?>("key").Should().Be(true);
            storage.Get<T?>("key").Should().Be(defaultValue);

            storage.Remove<T?>("key").Should().Be(true);
            storage.Has<T?>("key").Should().Be(false);
            storage.Get<T?>("key").Should().Be(default(T?));
        }

        private void DoGeneralStorageSectionCheckForEnum<T>(T defaultValue)
            where T : unmanaged, Enum
        {
            // Arrange
            using var storage = BinaryStorage
                .Construct(EditorModeTestsEnvironment.TempStoragePath)
                .SupportEnum<T>()
                .Build();

            // Act
            storage.Supports<T>().Should().Be(true);
            storage.Set<T>("key", defaultValue);
            storage.LoadDataFromDisk();

            // Assert
            storage.Has<T>("key").Should().Be(true);
            storage.Get<T>("key").Should().Be(defaultValue);

            storage.Remove<T>("key").Should().Be(true);
            storage.Has<T>("key").Should().Be(false);
            storage.Get<T>("key").Should().Be(default(T));
        }
    }
}