using System;
using Newtonsoft.Json;

namespace Magify
{
    internal static class UnitTimeConverterUtilities
    {
        public static void WriteValue(this JsonWriter writer, TimeSpan timeSpan, TimeUnit timeUnit, bool supportFloatingNumbers)
        {
            if (timeUnit == TimeUnit.Ticks)
            {
                writer.WriteValue(timeSpan.Ticks);
                return;
            }

            var number = timeUnit switch
            {
                TimeUnit.Millisecond => timeSpan.TotalMilliseconds,
                TimeUnit.Second => timeSpan.TotalSeconds,
                TimeUnit.Minute => timeSpan.TotalMinutes,
                TimeUnit.Hour => timeSpan.TotalHours,
                TimeUnit.Day => timeSpan.TotalDays,
                _ => throw new JsonSerializationException($"Unexpected TimeUnit {timeUnit}.")
            };
            if (supportFloatingNumbers)
            {
                writer.WriteValue(number);
            }
            else
            {
                writer.WriteValue((long)number);
            }
        }

        public static TimeSpan? ReadTimeSpan(this JsonReader reader, TimeUnit timeUnit, bool supportFloatingNumbers)
        {
            if (reader.TokenType is JsonToken.Null or JsonToken.None)
            {
                return null;
            }

            if (timeUnit == TimeUnit.Ticks)
            {
                if (reader.TokenType != JsonToken.Integer)
                {
                    throw new JsonSerializationException($"Cannot convert {reader.Value} value to long (Ticks).");
                }

                var ticks = Convert.ToInt64(reader.Value);
                return TimeSpan.FromTicks(ticks);
            }

            switch (reader.TokenType)
            {
                case JsonToken.Integer:
                {
                    var number = Convert.ToInt64(reader.Value);
                    return resultFromInt64(number);
                }
                case JsonToken.Float:
                {
                    var number = Convert.ToDouble(reader.Value);
                    return supportFloatingNumbers ? resultFromDouble(number) : resultFromInt64((long)number);
                }
                case JsonToken.String when double.TryParse((string)reader.Value, out var number):
                {
                    return supportFloatingNumbers ? resultFromDouble(number) : resultFromInt64((long)number);
                }
            }

            throw new JsonSerializationException($"Invalid date value. {reader.TokenType}={reader.Value}");

            TimeSpan resultFromInt64(long number)
            {
                return timeUnit switch
                {
                    TimeUnit.Millisecond => TimeSpan.FromMilliseconds(number),
                    TimeUnit.Second => TimeSpan.FromSeconds(number),
                    TimeUnit.Minute => TimeSpan.FromMinutes(number),
                    TimeUnit.Hour => TimeSpan.FromHours(number),
                    TimeUnit.Day => TimeSpan.FromDays(number),
                    _ => throw new JsonSerializationException($"Unexpected TimeUnit {timeUnit}.")
                };
            }

            TimeSpan resultFromDouble(double number)
            {
                return timeUnit switch
                {
                    TimeUnit.Millisecond => TimeSpan.FromMilliseconds(number),
                    TimeUnit.Second => TimeSpan.FromSeconds(number),
                    TimeUnit.Minute => TimeSpan.FromMinutes(number),
                    TimeUnit.Hour => TimeSpan.FromHours(number),
                    TimeUnit.Day => TimeSpan.FromDays(number),
                    _ => throw new JsonSerializationException($"Unexpected TimeUnit {timeUnit}.")
                };
            }
        }
    }
}