using System;
using JetBrains.Annotations;
using Newtonsoft.Json;
using Newtonsoft.Json.Converters;
using UnityEngine;
using UnityEngine.Scripting;

namespace Magify
{
    [Preserve]
    public class StringNullableEnumConverter : StringEnumConverter
    {
        public override object ReadJson(JsonReader reader, Type objectType, object existingValue, JsonSerializer serializer)
        {
            try
            {
                return base.ReadJson(reader, objectType, existingValue, serializer);
            }
            catch (Exception e) when (objectType.IsGenericType && objectType.GetGenericTypeDefinition() == typeof(Nullable<>))
            {
                Debug.LogException(e);
                return null;
            }
        }
    }

    [Preserve]
    public class ProductStoreConverter : JsonConverter
    {
        public override bool CanConvert(Type objectType)
        {
            return objectType == typeof(PurchaseStore) || objectType == typeof(PurchaseStore?);
        }

        public override void WriteJson(JsonWriter writer, object value, JsonSerializer serializer)
        {
            if (value is PurchaseStore productStore)
            {
                writer.WriteValue(productStore.Name);
            }
        }

        public override object ReadJson(JsonReader reader, Type objectType, object existingValue, JsonSerializer serializer)
        {
            var isNullable = objectType.IsGenericType && objectType.GetGenericTypeDefinition() == typeof(Nullable<>);
            objectType = isNullable ? Nullable.GetUnderlyingType(objectType)! : objectType;
            if (reader.TokenType == JsonToken.String)
            {
                var name = reader.Value?.ToString();
                if (string.IsNullOrEmpty(name))
                {
                    if (isNullable)
                    {
                        return null;
                    }
                    throw Exception($"Cannot convert null value to not nullable {objectType}.", reader);
                }
                return new PurchaseStore(name);
            }
            throw Exception($"Cannot convert {reader.TokenType} to {objectType}.", reader);
        }

        [NotNull]
        private static Exception Exception([NotNull] string message, [NotNull] JsonReader reader)
        {
            int lineNumber;
            int linePosition;
            if (reader is IJsonLineInfo lineInfo && lineInfo.HasLineInfo())
            {
                lineNumber = lineInfo.LineNumber;
                linePosition = lineInfo.LinePosition;
            }
            else
            {
                lineNumber = 0;
                linePosition = 0;
            }
            return new JsonSerializationException(message, reader.Path, lineNumber, linePosition, null);
        }
    }
}