using System;
using Newtonsoft.Json;
using Newtonsoft.Json.Converters;
using UnityEngine;
using UnityEngine.Scripting;

namespace Magify
{
    [Preserve]
    public class DayOfWeekConverter : StringEnumConverter
    {
        private Exception InvalidItemException { get; } = new JsonSerializationException("Invalid day of week value.");

        public override bool CanConvert(Type objectType)
        {
            return objectType == typeof(DayOfWeek) || objectType == typeof(DayOfWeek?);
        }

        public override void WriteJson(JsonWriter writer, object value, JsonSerializer serializer)
        {
            if (value is DayOfWeek dayOfWeek)
            {
                var dayString = dayOfWeek switch
                {
                    DayOfWeek.Monday => "mon",
                    DayOfWeek.Tuesday => "tue",
                    DayOfWeek.Wednesday => "wed",
                    DayOfWeek.Thursday => "thur",
                    DayOfWeek.Friday => "fri",
                    DayOfWeek.Saturday => "sat",
                    DayOfWeek.Sunday => "sun",
                    _ => throw new JsonSerializationException("Invalid day of week value.")
                };
                writer.WriteValue(dayString);
            }
            else
            {
                writer.WriteNull();
            }
        }

        public override object ReadJson(JsonReader reader, Type objectType, object existingValue, JsonSerializer serializer)
        {
            var nullable = objectType.IsGenericType && objectType.GetGenericTypeDefinition() == typeof(Nullable<>);
            if (nullable && reader.TokenType == JsonToken.Null)
            {
                return null;
            }

            DayOfWeek? dayOfWeek = null;
            if (reader.TokenType == JsonToken.String)
            {
                var value = reader.Value?.ToString()?.ToLower();
                dayOfWeek = value switch
                {
                    "mon" => DayOfWeek.Monday,
                    "tue" => DayOfWeek.Tuesday,
                    "wed" => DayOfWeek.Wednesday,
                    "thur" => DayOfWeek.Thursday,
                    "fri" => DayOfWeek.Friday,
                    "sat" => DayOfWeek.Saturday,
                    "sun" => DayOfWeek.Sunday,
                    _ => null
                };
            }

            if (nullable)
            {
                if (!dayOfWeek.HasValue)
                {
                    Debug.LogException(InvalidItemException);
                }
                return dayOfWeek;
            }
            if (!dayOfWeek.HasValue)
            {
                throw InvalidItemException;
            }
            return dayOfWeek.Value;
        }
    }
}