﻿using System;
using JetBrains.Annotations;

namespace Magify
{
    public sealed class CopyOnWriteList<T> where T: class // If you want to change it and use for value-types see IndexOf
    {
        [NotNull]
        public static readonly CopyOnWriteList<T> Empty = new();
        [NotNull]
        private readonly object _lock = new();

        [NotNull]
        public T[] Data { get; }

        private CopyOnWriteList()
        {
            Data = Array.Empty<T>();
        }

        public CopyOnWriteList([NotNull] T[] data)
        {
            Data = data;
        }

        [NotNull]
        public CopyOnWriteList<T> Add([CanBeNull] T value)
        {
            int length;
            T[] newData;
            lock (_lock)
            {
                length = Data.Length;
                newData = new T[length + 1];
                Array.Copy(Data, newData, length);
            }
            newData[length] = value;
            return new CopyOnWriteList<T>(newData);
        }

        [NotNull]
        public CopyOnWriteList<T> Remove([CanBeNull] T value)
        {
            T[] newData;
            lock (_lock)
            {
                var i = IndexOfThreadUnsafe(value);
                if (i < 0) return this;

                var length = Data.Length;
                if (length == 1) return Empty;

                newData = new T[length - 1];

                Array.Copy(Data, 0, newData, 0, i);
                Array.Copy(Data, i + 1, newData, i, length - i - 1);
            }

            return new CopyOnWriteList<T>(newData);
        }

        public int IndexOf([CanBeNull] T value)
        {
            lock (_lock)
            {
                return IndexOfThreadUnsafe(value);
            }
        }

        private int IndexOfThreadUnsafe([CanBeNull] T value)
        {
            for (var i = 0; i < Data.Length; ++i)
            {
                // For T: value-type here will be boxing
                if (Equals(Data[i], value))
                {
                    return i;
                }
            }
            return -1;
        }
    }
}