﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using JetBrains.Annotations;

namespace Magify
{
    internal interface ICollectionSection
    {
        // Marker-interface
    }

    internal class CollectionSection<T, TCollection> : Section<T, TCollection>, ICollectionSection
        where TCollection : class, IReactiveCollection, IRefillableCollection<T>, ICollection<T>, new()
    {
        public override event Action OnChanged;
        [NotNull]
        public override string Name { get; }

        public CollectionSection([NotNull] BinaryTypeHandler<T> binaryType) : base(binaryType)
        {
            Name = $"{typeof(TCollection).Name}<{BinaryType.TypeName}>";
        }

        public override bool HasData => Data.Any(c => c.Value!.Count > 0);

        [NotNull, ItemNotNull]
        public TCollection Get([NotNull] string key)
        {
            var res = TryGet(key);
            var value = res.Value;
            if (!res.Exists)
            {
                value = new TCollection();
                value.OnChanged += () => OnChanged?.Invoke();
                Data[key] = value;
            }
            return value;
        }

        public override bool Remove([NotNull] string key)
        {
            var res = TryGet(key);
            if (res.Exists)
            {
                res.Value.Clear();
            }
            return res.Exists;
        }

        public override void Reset()
        {
            foreach (var value in Data.Values)
            {
                value?.Clear();
            }
        }

        public override void WriteTo([NotNull] BinaryWriter writer)
        {
            writer.Write(Data.Count);
            foreach (var pair in Data)
            {
                writer.Write(pair.Key!); // it's impossible
                var size = pair.Value!.Sum(BinaryType.SizeOf) + sizeof(int);
                writer.Write(size);
                writer.Write(pair.Value.Count);
                foreach (var item in pair.Value)
                {
                    BinaryType.WriteTo(writer, item);
                }
            }
        }

        public override void ReadFrom([NotNull] BinaryReader reader)
        {
            var count = reader.ReadInt32();
            for (var i = 0; i < count; i++)
            {
                var key = reader.ReadString();
                reader.ReadInt32(); // skip reading size of value
                var itemsCount = reader.ReadInt32();
                var value = new TCollection();
                for (var j = 0; j < itemsCount; j++)
                {
                    value.Add(BinaryType.ReadFrom(reader));
                }
                if (Data.TryGetValue(key, out var existingCollection) && existingCollection != null)
                {
                    existingCollection.Refill(value);
                }
                else
                {
                    value.OnChanged += () => OnChanged?.Invoke();
                    Data[key] = value;
                }
            }
        }
    }
}