﻿using System;
using JetBrains.Annotations;
using Magify.Model;
using UnityEngine;

namespace Magify
{
    internal static class ServerApiUtils
    {
        [NotNull]
        internal const string UnexpectedExpiredTokenErrorMessage = "Something went wrong, request token can't return \"Expired\"";
        [NotNull]
        internal const string RevokedTokenErrorMessage = "Authorization Token revoked, trying to reset current token and request new one";
        [NotNull]
        internal const string ExpiredReissueTokenErrorMessage = "\"When requesting to reissue token, an \\\"Expired\\\" message is returned. Trying to reset current token and request new one.\"";

        [ContractAnnotation("=> true, result:notnull, exception:null; => false, result:canbenull, exception: notnull")]
        public static bool TryDeserializeFromBasicResponse<T>([NotNull] string text, [NotNull] MagifyLogger logger, [CanBeNull] out T result, [CanBeNull] out Exception exception)
        {
            var isDeserialized = TryDeserializeResponse<BaseResponse<T>>(text, logger, out var baseResponse, out exception);
            result = baseResponse == null ? default : baseResponse.Result;
            return isDeserialized;
        }

        [ContractAnnotation("=> true, result:notnull, exception:null; => false, result:canbenull, exception: notnull")]
        public static bool TryDeserializeResponse<T>([NotNull] string text, [NotNull] MagifyLogger logger, [CanBeNull] out T result, [CanBeNull] out Exception exception)
        {
            try
            {
                result = JsonFacade.DeserializeObject<T>(text);
                exception = null;
                return true;
            }
            catch (Exception e)
            {
                logger.LogError($"Failed to parse response text with exception: {e}");
                result = default;
                exception = e;
                return false;
            }
        }

        [CanBeNull]
        public static ErrorResponse TryExtractErrorResponse([NotNull] string text, [NotNull] MagifyLogger logger)
        {
            try
            {
                return JsonFacade.DeserializeObject<ErrorResponse>(text);
            }
            catch(Exception e)
            {
                logger.LogError($"Failed to recognize error response with exception: {e.Message}");
                return null;
            }
        }

        public static bool TryHandleDefaultErrorCodes([CanBeNull] ErrorResponse.ErrorInfo error,
            [NotNull] string method,
            [NotNull] AuthorizationToken authorizationToken,
            [NotNull] MagifyLogger logger,
            Action bannedStatusHandler,
            out RepeatState repeatState)
        {
            if (error == null)
            {
                repeatState = default;
                return false;
            }
            switch (error.Code)
            {
                case ErrorCode.Revoked:
                    logger.Log(RevokedTokenErrorMessage);
                    authorizationToken.AuthToken = null;
                    repeatState = RepeatState.Retry;
                    return true;

                case ErrorCode.Expired when method is WebRequestMethods.ReissueToken:
                    logger.Log(ExpiredReissueTokenErrorMessage);
                    authorizationToken.AuthToken = null;
                    repeatState = RepeatState.Retry;
                    return true;

                case ErrorCode.Expired when method is WebRequestMethods.GetToken:
                    logger.LogError(UnexpectedExpiredTokenErrorMessage);
                    repeatState = RepeatState.Retry;
                    return true;

                case ErrorCode.Expired when method is not (WebRequestMethods.GetToken or WebRequestMethods.ReissueToken):
                    authorizationToken.IsExpired = true;
                    repeatState = RepeatState.Retry;
                    return true;

                case ErrorCode.Banned:
                    bannedStatusHandler();
                    authorizationToken.AuthToken = null;
                    repeatState = RepeatState.Finish;
                    return true;
            }
            repeatState = default;
            return false;
        }
    }
}