using System;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal class ServerInteractionsProvider : ICancelable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Network);

        [NotNull]
        private readonly INetworkClient _networkClient;
        [NotNull]
        private readonly GetAuthTokenAsyncDelegate _asyncAuthTokenGetter;
        [NotNull]
        private readonly object _urlCancellationTokenLock = new();

        [NotNull]
        private CancellationTokenSource _urlCancellationToken = new();

        public bool IsDisposed { get; private set; }
        public EndpointUrl URL { get; private set; }

        public ServerInteractionsProvider(EndpointUrl url, [NotNull] INetworkClient networkClient, [NotNull] GetAuthTokenAsyncDelegate asyncAuthTokenGetter)
        {
            _networkClient = networkClient;
            _asyncAuthTokenGetter = asyncAuthTokenGetter;
            URL = url;
        }

        void IDisposable.Dispose()
        {
            if (IsDisposed)
            {
                return;
            }
            IsDisposed = true;
            lock (_urlCancellationTokenLock)
            {
                _urlCancellationToken.Cancel();
                _urlCancellationToken.Dispose();
            }
        }

        public void ChangeServerUrl(EndpointUrl url)
        {
            if (URL != url)
            {
                URL = url;
                lock (_urlCancellationTokenLock)
                {
                    ThreadingUtils.CancelAndReplace(ref _urlCancellationToken);
                }
            }
        }

        public ServerInteractionTask<TOut> MakeAsyncRequest<TOut>(
            [NotNull] in RequestContext<TOut> context,
            in RequestConfig config,
            CancellationToken cancellationToken)
        {
            _logger.Log($"{nameof(MakeAsyncRequest)} called for '{config.Method}' with config: {JsonFacade.SerializeObject(config)};");
            if (URL.IsOffline)
            {
                _logger.Log($"Offline mode is enabled. {nameof(MakeAsyncRequest)} is skipped.");
                return ServerInteractionTask<TOut>.OfflineModeIsEnabled();
            }

            cancellationToken.ThrowIfCancellationRequested();
            lock (_urlCancellationTokenLock)
            {
                cancellationToken = CancellationTokenSource.CreateLinkedTokenSource(
                    _urlCancellationToken.Token,
                    context.GetAllRequestsToken(),
                    cancellationToken
                ).Token;
            }

            if (context.SingleModeRequestTools != null)
            {
                return MakeRequestInSingleMode(
                    context.SingleModeRequestTools,
                    context.ResponseProcessor,
                    in config,
                    cancellationToken,
                    context.GetActiveRequestCancellationToken).ToServerInteractionTask();
            }
            if (context.IndependentModeRequestTools != null)
            {
                return MakeRequestInIndependentMode(
                    context.IndependentModeRequestTools,
                    context.ResponseProcessor,
                    in config,
                    in cancellationToken,
                    context.GetActiveRequestCancellationToken).ToServerInteractionTask();
            }
            return ServerInteractionTask<TOut>.NoCompatibleModeFound();
        }

        private UniTask<TOut> MakeRequestInSingleMode<TOut>(
            [NotNull] SingleModeRequestTools<TOut> tools,
            [NotNull] in ResponseProcessorDelegate<TOut> responseProcessor,
            in RequestConfig config,
            CancellationToken cancellationToken,
            [NotNull] in ActiveRequestCancellationTokenProviderDelegate activeRequestCancellationTokenProvider)
        {
            if (tools.TryCreateNewCompletionSource(in cancellationToken, out var completionTask, out var requestCancellationToken))
            {
                var method = config.Method;
                _logger.Log($"Start requesting to server for {typeof(TOut).Name}");
                CreateBaseRequestInSingleMode(in tools, in responseProcessor, in config, in requestCancellationToken, in activeRequestCancellationTokenProvider)
                    .ContinueWith(tools.TrySetResult)
                    .Forget(exception =>
                    {
                        if (exception is OperationCanceledException e)
                        {
                            _logger.Log($"Setting cancel status to the {method} tools with {e.Message}");
                            tools.TrySetCancelled(e.CancellationToken);
                        }
                        else
                        {
                            _logger.Log($"Setting exception status to the {method} tools with {exception.Message}");
                            tools.TrySetException(exception!);
                        }
                    });
            }
            else
            {
                _logger.Log($"Request to server for {typeof(TOut).Name} is waiting for existing request result");
                // We have to reset delay, because new request was triggered
                tools.ResetRequestsRepeatDelay();
            }

            _logger.Log("Wait till request result or cancel token");
            return completionTask
                .AttachExternalCancellation(cancellationToken)
                .AttachExternalCancellation(requestCancellationToken);
        }

        private UniTask<TOut> CreateBaseRequestInSingleMode<TOut>(
            [NotNull] in SingleModeRequestTools<TOut> tools,
            [NotNull] in ResponseProcessorDelegate<TOut> responseProcessor,
            in RequestConfig config,
            in CancellationToken cancellationToken,
            [NotNull] in ActiveRequestCancellationTokenProviderDelegate activeRequestCancellationTokenProvider)
        {
            return config.UseWebRequestWithRepeats
                ? PerformRequestWithRepeats(tools.Backoff, responseProcessor, config, cancellationToken, activeRequestCancellationTokenProvider, tools.RequestRepeatingDelayToken)
                : PerformOneRequest(responseProcessor, config, cancellationToken);
        }

        private UniTask<TOut> MakeRequestInIndependentMode<TOut>(
            [NotNull] in IndependentModeRequestTools tools,
            [NotNull] in ResponseProcessorDelegate<TOut> responseProcessor,
            in RequestConfig config,
            in CancellationToken cancellationToken,
            [NotNull] in ActiveRequestCancellationTokenProviderDelegate activeRequestCancellationTokenProvider)
        {
            return config.UseWebRequestWithRepeats
                ? PerformRequestWithRepeats(tools.Backoff, responseProcessor, config, cancellationToken, activeRequestCancellationTokenProvider, tools.RequestRepeatingDelayToken)
                : PerformOneRequest(responseProcessor, config, cancellationToken);
        }

        private async UniTask<TOut> PerformRequestWithRepeats<TOut>(
            [NotNull] ExponentialBackoff backoff,
            [NotNull] ResponseProcessorDelegate<TOut> responseProcessor,
            RequestConfig config,
            CancellationToken mainCancellationToken,
            [NotNull] ActiveRequestCancellationTokenProviderDelegate activeRequestCancellationTokenProvider,
            [NotNull] ReusableCancellationTokenSource repeatDelayCancellationToken)
        {
            backoff.Reset();
            while (true)
            {
                _logger.Log($"Run request cycle for {config.Method}, attempt {backoff.Retries}");
                CancellationToken currentToken;
                WebResponseMessage webResponse;
                try
                {
                    currentToken = CancellationTokenSource.CreateLinkedTokenSource(mainCancellationToken, activeRequestCancellationTokenProvider()).Token;
                    currentToken.ThrowIfCancellationRequested();
                    webResponse = await PerformRequestAsync(config.IsAuthTokenRequired, config.RequestMessageGetter(), currentToken);
                }
                catch (OperationCanceledException) when (!mainCancellationToken.IsCancellationRequested && currentToken.IsCancellationRequested && !IsDisposed)
                {
                    _logger.Log("Just active request token was cancelled");
                    continue;
                }
                catch (OperationCanceledException)
                {
                    throw;
                }
                catch (WaitForAuthTokenLoadingException)
                {
                    _logger.Log("Waiting for auth token failed with expected exception, retrying...");
                    continue;
                }
                mainCancellationToken.ThrowIfCancellationRequested();

                switch (responseProcessor(webResponse, out var result))
                {
                    case RepeatState.Finish:
                        return result;
                    case RepeatState.Retry:
                        continue;
                    case RepeatState.Wait:
                        await WaitForNextRepeat(backoff, mainCancellationToken, repeatDelayCancellationToken.GetOrCreate());
                        break;
                    default:
                        throw new MagifyArgumentOutOfRangeException(nameof(RepeatState));
                }
            }
        }

        private async UniTask<TOut> PerformOneRequest<TOut>(
            [NotNull] ResponseProcessorDelegate<TOut> responseProcessor,
            RequestConfig config,
            CancellationToken cancellationToken)
        {
            WebResponseMessage? webResponse = null;
            while (webResponse == null)
            {
                try
                {
                    cancellationToken.ThrowIfCancellationRequested();
                    webResponse = await PerformRequestAsync(config.IsAuthTokenRequired, config.RequestMessageGetter(), cancellationToken);
                }
                catch (WaitForAuthTokenLoadingException)
                {
                    _logger.Log("Waiting for auth token failed with expected exception, force retrying...");
                    webResponse = null;
                }
            }
            responseProcessor(webResponse!.Value, out var result);
            return result;
        }

        private async UniTask<WebResponseMessage> PerformRequestAsync(
            bool isAuthTokenRequired,
            WebRequestMessage message,
            CancellationToken mainCancellationToken)
        {
            if (isAuthTokenRequired)
            {
                _logger.Log($"Prepare token before call {message.Method}.");
                try
                {
                    message.AuthToken = await _asyncAuthTokenGetter(mainCancellationToken);
                }
                catch (OperationCanceledException) when (!mainCancellationToken.IsCancellationRequested && !IsDisposed)
                {
                    _logger.LogWarning($"While request {message.Method} was waiting for an auth token, the auth token request was cancelled. Request details: {message.Payload}");
                    throw new WaitForAuthTokenLoadingException();
                }
            }

            var response = await SendAsync(in message, in mainCancellationToken);
            if (mainCancellationToken.IsCancellationRequested)
            {
                _logger.Log($"Got response from performing request for {message.Method}, but token is cancelled");
                throw new OperationCanceledException(mainCancellationToken);
            }

            _logger.Log($"Get response from performing request for {message.Method}\nResponse: {response}");
            return response;
        }

        private UniTask<WebResponseMessage> SendAsync(in WebRequestMessage requestMessage, in CancellationToken cancellationToken)
        {
            _logger.Log($"{nameof(SendAsync)} called for {requestMessage.Method}");
            if (IsDisposed)
            {
                _logger.Log($"{nameof(ServerInteractionsProvider)} is disposed, cancellation of {requestMessage.Method}");
                throw new OperationCanceledException();
            }
            return _networkClient.SendAsync(URL.Value, requestMessage, cancellationToken);
        }

        [NotNull]
        private static async Task WaitForNextRepeat([NotNull] ExponentialBackoff backoff, CancellationToken mainCancellationToken, CancellationToken repeatDelayCancellationToken)
        {
            var delayTime = TimeSpan.FromMilliseconds(backoff.NextDelay());
            _logger.Log($"Retry perform request after {delayTime.TotalSeconds} sec.");
            try
            {
                var token = CancellationTokenSource.CreateLinkedTokenSource(mainCancellationToken, repeatDelayCancellationToken).Token;
                await Task.Delay(delayTime, cancellationToken: token);
            }
            catch (OperationCanceledException)
            {
                if (mainCancellationToken.IsCancellationRequested)
                {
                    throw;
                }
                if (repeatDelayCancellationToken.IsCancellationRequested)
                {
                    _logger.Log("Repeat delay waiting cancelled, it will be reset");
                    backoff.Reset();
                }
            }
        }

        private class WaitForAuthTokenLoadingException : MagifyException
        {
        }
    }
}