using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using UnityEngine.Networking;

namespace Magify
{
    internal partial class ServerApi
    {
        // ReSharper disable once ContainerAnnotationRedundancy
        [ItemCanBeNull]
        public UniTask<string> GetAuthorizationTokenAsync(CancellationToken cancellationToken)
        {
            ThrowIfDisposed();
            if (AuthorizationToken.HasValidToken())
            {
                _logger.Log($"Current valid token is {AuthorizationToken.AuthToken}");
                return UniTask.FromResult(AuthorizationToken.AuthToken);
            }

            var config = new RequestConfig(
                WebRequestMethods.GetToken,
                createNewTokenRequestPayload,
                useWebRequestWithRepeats: true,
                isAuthTokenRequired: false);
            var request = _interactionsProvider.MakeAsyncRequest(
                in _requestNewTokenContext,
                in config,
                cancellationToken
            );
            return request.HandleByDefault(config, rejectedResult: null);

            WebRequestMessage createNewTokenRequestPayload()
            {
                ThrowIfDisposed();
                var authConfig = _authConfigBuilder.AuthorizationConfig;
                return AuthorizationToken.IsExpired
                    ? CreateReissueTokenRequestPayload(authConfig, AuthorizationToken)
                    : CreateNewTokenRequestPayload(authConfig, _generalPrefs, _appStatePrefs, _platform, IsGeoIpEnabled.Value);
            }
        }

        public void ResetAuthorizationToken()
        {
            AuthorizationToken.AuthToken = string.Empty;
            CancelJustActiveTokenRequest();
        }

        private RepeatState NewTokenResponseHandler(WebResponseMessage response, [CanBeNull] out string result)
        {
            if (response.Result == UnityWebRequest.Result.Success && !string.IsNullOrEmpty(response.Text))
            {
                if (ServerApiUtils.TryDeserializeResponse<TokenResponse>(response.Text, _logger, out var tokenResponse, out var exception) && !string.IsNullOrEmpty(tokenResponse?.Token))
                {
                    _logger.Log("AuthToken successfully loaded");
                    result = AuthorizationToken.AuthToken = tokenResponse.Token;
                    return RepeatState.Finish;
                }
                else
                {
                    _logger.LogWarning($"Failed to parse token response: {exception?.Message}");
                    result = default;
                    return RepeatState.Wait;
                }
            }

            if (!string.IsNullOrEmpty(response.Text) && response.ResponseCode == 403)
            {
                var errorResponse = ServerApiUtils.TryExtractErrorResponse(response.Text, _logger);
                if (ServerApiUtils.TryHandleDefaultErrorCodes(errorResponse?.Error, response.RequestMessage.Method, AuthorizationToken, _logger, HandleBannedResponse, out var repeatState))
                {
                    result = AuthorizationToken.AuthToken;
                    return repeatState;
                }
            }

            result = null;
            return RepeatState.Wait;
        }

        public UniTask CancelAllTokenLoadings()
        {
            _logger.Log($"{nameof(CancelAllTokenLoadings)} has been called");
            _requestNewTokenContext.CancelAllRequests();
            var promise = _requestNewTokenContext.SingleModeRequestTools.CompletionSourceRaw;
            if (promise == null || promise.IsCompleted())
            {
                return UniTask.CompletedTask;
            }
            return promise.Task.SuppressCancellationThrow();
        }

        public void CancelJustActiveTokenRequest()
        {
            _logger.Log($"{nameof(CancelJustActiveTokenRequest)} has been called");
            _requestNewTokenContext.CancelActiveRequest();
        }

        /// <inheritdoc cref="MagifyDocs.TestOnly"/>
        internal WebRequestMessage CreateNewTokenRequestPayloadNonStatic()
        {
            ThrowIfDisposed();
            return CreateNewTokenRequestPayload(_authConfigBuilder.AuthorizationConfig, _generalPrefs, _appStatePrefs, _platform, IsGeoIpEnabled.Value);
        }

        private static WebRequestMessage CreateNewTokenRequestPayload(
            [NotNull] AuthorizationConfig authConfig,
            [NotNull] GeneralPrefs generalPrefs,
            [NotNull] AppStatePrefs appStatePrefs,
            [NotNull] PlatformAPI platform,
            bool isGeoIpEnabled)
        {
            var firstLaunchTime = new DateTimeOffset(authConfig.FirstLaunchDate);
            SocialAuth social = default;
            if (generalPrefs.SyncStateEnabled.Value
             && !string.IsNullOrEmpty(appStatePrefs.ExternalAuthProvider.Value)
             && !string.IsNullOrEmpty(appStatePrefs.ExternalAuthToken.Value))
            {
                social = new SocialAuth
                {
                    Provider = appStatePrefs.ExternalAuthProvider.Value,
                    Token = appStatePrefs.ExternalAuthToken.Value,
                };
            }
            var authToken = new AuthTokenPayload
            {
                ClientId = authConfig.UserClientId,
                Application = new ApplicationInfo
                {
                    Name = authConfig.ApplicationName,
                    Version = authConfig.ApplicationVersion
                },
                Device = new DeviceInfo
                {
                    Name = authConfig.DeviceName,
                    Version = authConfig.DeviceVersion
                },
                Segmentation = new Segmentation
                {
                    CountryCode = authConfig.CountryCode,
                    LanguageCode = authConfig.LanguageCode,
                    FirstLaunchSeconds = firstLaunchTime.ToUnixTimeSeconds(),
                    FirstLaunchDate = firstLaunchTime.ToString("yyyy-MM-dd"),
                    IsNewUser = authConfig.IsNewUser,
                    Orientation = authConfig.SupportedOrientation.GetDescription(),
                },
                IsSandbox = authConfig.IsSandbox,
                IsBasicIntegration = authConfig.IsBasicIntegration,
                SdkVersion = authConfig.SdkVersion,
                BuildNumber = platform.BuildNumber,
                Social = social,
                AdjustId = generalPrefs.AdjustId.Value,
            };

            platform.AddPlatformSpecificData(authToken, generalPrefs, isGeoIpEnabled, generalPrefs.SyncStateEnabled.Value);

            return new WebRequestMessage
            {
                AuthToken = null,
                Method = WebRequestMethods.GetToken,
                Payload = authToken,
            };
        }

        private static WebRequestMessage CreateReissueTokenRequestPayload([NotNull] AuthorizationConfig authConfig, [NotNull] AuthorizationToken expiredToken)
        {
            return new WebRequestMessage
            {
                Method = WebRequestMethods.ReissueToken,
                AuthToken = expiredToken.AuthToken,
                Payload = new ReissuePayloadParams
                {
                    SdkVersion = authConfig.SdkVersion,
                },
            };
        }
    }
}