using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using UnityEngine.Networking;

namespace Magify
{
    internal partial class ServerApi
    {
        public UniTask<string> AuthorizeUser([NotNull] string clientId, CancellationToken cancellationToken)
        {
            ThrowIfDisposed();
            var config = new RequestConfig(
                ApplicationStateApiMethods.Authorize(_platform.RuntimePlatform),
                createAuthorizeUserRequestPayload,
                useWebRequestWithRepeats: true,
                isAuthTokenRequired: true);
            var request = _interactionsProvider.MakeAsyncRequest(
                in _requestAuthorizeUserContext,
                in config,
                cancellationToken
            );
            return request.HandleByDefault(config, rejectedResult: null);

            WebRequestMessage createAuthorizeUserRequestPayload()
            {
                ThrowIfDisposed();
                return new WebRequestMessage
                {
                    AuthToken = default,
                    Method = ApplicationStateApiMethods.Authorize(_platform.RuntimePlatform),
                    Payload = new AuthorizeUserPayload
                    {
                        ClientId = clientId,
                    },
                };
            }
        }

        private RepeatState AuthorizeUserResponseHandler(WebResponseMessage response, out string result)
        {
            _logger.Log($"{nameof(AuthorizeUserResponseHandler)} called");
            if (response.Result == UnityWebRequest.Result.Success && !string.IsNullOrEmpty(response.Text))
            {
                if (ServerApiUtils.TryDeserializeResponse<GaladrielResponse<GaladrielUser>>(response.Text, _logger, out var authorizeUserResponse, out var exception))
                {
                    result = authorizeUserResponse.Result?.UserId;
                    if (result != null)
                    {
                        _logger.Log($"User successfully authorized {authorizeUserResponse.Result}");
                        return RepeatState.Finish;
                    }
                }
                else
                {
                    _logger.LogWarning($"Failed to parse context response: {exception.Message}");
                    result = default;
                }
                return RepeatState.Retry;
            }

            if (!string.IsNullOrEmpty(response.Text))
            {
                var errorResponse = ServerApiUtils.TryExtractErrorResponse(response.Text, _logger);
                if (ServerApiUtils.TryHandleDefaultErrorCodes(errorResponse?.Error, response.RequestMessage.Method, AuthorizationToken, _logger, HandleBannedResponse, out var repeatState))
                {
                    result = null;
                    return repeatState;
                }
            }

            result = null;
            return RepeatState.Retry;
        }

        public void CancelAllAuthorizeUserRequests()
        {
            _logger.Log($"{nameof(CancelAllAuthorizeUserRequests)} has been called");
            _requestAuthorizeUserContext.CancelAllRequests();
        }
    }
}