using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    internal interface IAuthorizationConfigBuilder : ICancelable
    {
        [NotNull]
        AuthorizationConfig AuthorizationConfig { get; }
    }

    internal class AuthorizationConfigBuilder : IInitializable, IAuthorizationConfigBuilder
    {
        [NotNull]
        private readonly string _applicationName;
        [NotNull]
        private readonly AppVersionProvider _appVersionProvider;
        [NotNull]
        private readonly PlatformAPI _platform;
        [NotNull]
        private readonly GeneralPrefs _prefs;
        [NotNull]
        private readonly InternalConfigPrefs _internalConfigPrefs;
        private readonly bool _isBasicIntegration;
        [NotNull]
        private readonly PooledCompositeDisposable _disposable = new();

        private AuthorizationConfig _authorizationConfig;

        public bool IsDisposed { get; private set; }

        [NotNull]
        public AuthorizationConfig AuthorizationConfig
        {
            get
            {
                if (_authorizationConfig == null)
                {
                    throw new CannotGetAuthConfigBeforeInitException();
                }

                return _authorizationConfig;
            }
        }

        public AuthorizationConfigBuilder(
            [NotNull] string applicationName,
            [NotNull] AppVersionProvider appVersionProvider,
            [NotNull] PlatformAPI platform,
            [NotNull] GeneralPrefs prefs,
            [NotNull] InternalConfigPrefs internalConfigPrefs,
            bool isBasicIntegration)
        {
            _applicationName = applicationName;
            _appVersionProvider = appVersionProvider;
            _platform = platform;
            _prefs = prefs;
            _internalConfigPrefs = internalConfigPrefs;
            _isBasicIntegration = isBasicIntegration;
        }

        void IInitializable.Initialize()
        {
            UpdateConfig();
            _prefs.ClientId
                .Subscribe(_ => UpdateConfig())
                .AddTo(_disposable);
        }

        public void UpdateConfig()
        {
            var locale = _platform.GetCurrentLocale();
            _authorizationConfig = new AuthorizationConfig
            {
                UserClientId = _prefs.ClientId.Value,
                IsNewUser = _prefs.IsNewUser,
                DeviceName = _platform.DeviceName,
                DeviceVersion = _platform.DeviceVersion,

                ApplicationName = _applicationName,
                ApplicationVersion = _appVersionProvider.AppVersion,
                SupportedOrientation = _platform.SupportedOrientation,

                IsSandbox = _internalConfigPrefs.IsSandbox.Value,
                IsBasicIntegration = _isBasicIntegration,

                SdkVersion = PackageInfo.Version,
                FirstLaunchDate = _prefs.FirstLaunchDate,

                LanguageCode = locale.LanguageWithRegion,
                CountryCode = locale.CountryCode,
            };
        }

        public void Dispose()
        {
            if (IsDisposed)
            {
                return;
            }
            IsDisposed = true;
            _disposable.Release();
        }
    }
}