﻿using System.Collections.Generic;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal partial class AghanimServerApi : IAghanimServerApi
    {
        [NotNull]
        private readonly AuthorizationToken _authorizationToken;
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Aghanim);

        [NotNull]
        private readonly ServerInteractionsProvider _interactionsProvider;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();

        [NotNull]
        private readonly RequestContext<AghanimOrderStatus?> _getOrderStatusContext;
        [NotNull]
        private readonly RequestContext<IReadOnlyList<AghanimProductCounting>> _getProductCountContext;
        [NotNull]
        private readonly RequestContext<AghanimOrder> _createOrderContext;

        public bool IsDisposed => _disposables.IsDisposed;

        public AghanimServerApi(
            EndpointUrl url,
            [NotNull] INetworkClient networkClient,
            [NotNull] AuthorizationToken authorizationToken,
            [NotNull] GetAuthTokenAsyncDelegate asyncAuthTokenGetter)
        {
            _authorizationToken = authorizationToken;
            _interactionsProvider = new ServerInteractionsProvider(url, networkClient, asyncAuthTokenGetter).AddTo(_disposables).AddTo(_disposables);

            _getOrderStatusContext = new RequestContext<AghanimOrderStatus?>(
                singleModeRequestTools: null,
                new IndependentModeRequestTools(),
                GetProductStatusResponseHandler
            ).AddTo(_disposables);

            _getProductCountContext = new RequestContext<IReadOnlyList<AghanimProductCounting>>(
                new SingleModeRequestTools<IReadOnlyList<AghanimProductCounting>>(),
                independentModeRequestTools: null,
                GetProductCountResponseHandler
            ).AddTo(_disposables);

            _createOrderContext = new RequestContext<AghanimOrder>(
                singleModeRequestTools: null,
                new IndependentModeRequestTools(),
                CreateOrderResponseHandler
            ).AddTo(_disposables);
        }

        public void Dispose()
        {
            _disposables.Release();
        }

        private void HandleBannedResponse()
        {
            _logger.Log("Player was banned, cancel trying to send requests for Aghanim");
            _createOrderContext.CancelAllRequests();
            _getProductCountContext.CancelAllRequests();
            _getOrderStatusContext.CancelAllRequests();
        }

        private void ThrowIfDisposed()
        {
            if (IsDisposed)
            {
                throw new MagifyObjectDisposedException(nameof(AghanimServerApi));
            }
        }
    }

    internal static class AghanimWebRequestMethods
    {
        [NotNull]
        public const string GetProductStatus = "getAghanimProductStatus";
        [NotNull]
        public const string GetProductsCount = "getAghanimProductsCount";
        [NotNull]
        public const string CreateOrder = "createAghanimOrder";
    }

    internal interface IAghanimServerApi : ICancelable
    {
        UniTask<AghanimOrderStatus?> GetOrderStatus([NotNull] string orderId, CancellationToken cancellationToken);
        [CanBeNull, ItemNotNull]
        UniTask<IReadOnlyList<AghanimProductCounting>> GetProductsCount(CancellationToken cancellationToken);
        [CanBeNull]
        UniTask<AghanimOrder> CreateOrder([NotNull] string productId, bool isSandbox, CancellationToken cancellationToken);
    }
}