using System.Collections.Generic;
using JetBrains.Annotations;
using Magify.Types;
using Newtonsoft.Json;

namespace Magify.Model
{
    internal class ProductPurchaseEvent : IAnalyticsEvent
    {
        [JsonProperty("campaignName")]
        public string CampaignName { get; init; }

        [JsonProperty("impressionNumber")]
        public int?  ImpressionNumber { get; init; }

        [JsonProperty("sessionImpressionNumber")]
        public int? SessionImpressionNumber { get; init; }

        [JsonProperty("campaignTypeImpressionNumber")]
        public int? CampaignTypeImpressionNumber { get; init; }

        [JsonProperty("campaignTypeSessionImpressionNumber")]
        public int? CampaignTypeSessionImpressionNumber { get; init; }

        [JsonProperty("eventName")]
        public string EventName { get; init; }

        [JsonProperty("eventNumber")]
        public int? EventNumber { get; init; }

        [JsonProperty("sessionEventNumber")]
        public int? SessionEventNumber { get; init; }

        [JsonProperty("sessionNumber")]
        public int SessionNumber { get; init; }

        [JsonProperty("isDefaultConfig"), CanBeNull]
        public bool? IsDefault { get; init; }

        [JsonProperty("isBasicIntegration"), CanBeNull]
        public bool? IsBasicIntegration { get; init; }

        [JsonProperty("product_id")]
        public string ProductId { get; init; }

        [JsonProperty("price")]
        public string Price { get; init; }

        [JsonProperty("currency")]
        public string Currency { get; init; }

        [JsonProperty("transaction_id")]
        public string TransactionId { get; init; }

        [JsonProperty("original_transaction_id")]
        public string OriginalTransactionId { get; init; }

        [JsonProperty("subscriptionStatus")]
        public SubscriptionStatus SubscriptionStatus { get; init; }

        [JsonProperty("inAppStatus")]
        public InAppStatus InAppStatus { get; init; }

        [JsonProperty("storefront"), CanBeNull]
        public string StoreCountry { get; init; }

        [JsonProperty("parameters"), CanBeNull]
        public Dictionary<string, object> Parameters { get; init; }

        [JsonProperty("mode")]
        public string Mode { get; init; }

        [JsonProperty("maxLevel")]
        public int? MaxLevel { get; init; }

        [JsonProperty("level")]
        public int? Level { get; init; }

        [JsonProperty("session_id"), CanBeNull]
        public string SessionId { get; init; }

        [JsonProperty("creative_id"), CanBeNull]
        public string ProductIdCreative { get; init; }

        [JsonProperty("store_name")]
        public PurchaseStore StoreName { get; set; }

        [JsonProperty("original_app_version")]
        public string AppVersion { get; init; }

        [JsonProperty("sdk_version"), CanBeNull]
        public string SdkVersion { get; init; }

        [JsonProperty("timestamp")]
        public double Timestamp { get; init; } = TimestampExtensions.UtcNowTimestamp;

        /// <remarks>
        /// For deserialization
        /// </remarks>
        public ProductPurchaseEvent()
        {
        }

        public ProductPurchaseEvent(
            [NotNull] CampaignImpression impression,
            PurchaseInfo purchase,
            SubscriptionStatus subscription,
            InAppStatus inApp,
            [NotNull] LevelStateInfo levelState,
            string sessionId,
            string storeCountry,
            string appVersion,
            string sdkVersion,
            bool? isBasicIntegration = null)
        {
            CampaignName = impression.CampaignName;
            ImpressionNumber = impression.ImpressionNumber;
            SessionImpressionNumber = impression.SessionImpressionNumber;
            CampaignTypeImpressionNumber = impression.CampaignTypeImpressionNumber;
            CampaignTypeSessionImpressionNumber = impression.CampaignTypeSessionImpressionNumber;
            EventName = impression.EventName;
            EventNumber = impression.EventNumber;
            SessionEventNumber = impression.SessionEventNumber;
            SessionNumber = impression.SessionNumber;
            IsDefault = impression.IsDefaultConfig;
            ProductId = purchase.Product.ProductId;
            ProductIdCreative = impression.ProductIdCreative;
            StoreName = purchase.Product.Store!.Value; // it must be not null for purchases
            Price = purchase.Product.Price ?? string.Empty;
            Currency = purchase.Product.Currency ?? string.Empty;
            TransactionId = purchase.TransactionId;
            OriginalTransactionId = purchase.OriginalTransactionId;
            SubscriptionStatus = subscription;
            InAppStatus = inApp;
            StoreCountry = storeCountry;
            Parameters = impression.Parameters;
            Mode = levelState.Mode;
            MaxLevel = levelState.MaxLevel;
            Level = levelState.Level;
            SessionId = sessionId;
            AppVersion = appVersion;
            SdkVersion = sdkVersion;
            IsBasicIntegration = isBasicIntegration;
        }
    }
}