using System;
using System.Collections.Generic;
using System.IO;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Aghanim;
using Magify.Features;
using Magify.Model;
using Magify.Rx;
using Magify.Types;
using UnityEngine;
using EventType = Magify.Model.EventType;

namespace Magify
{
    internal class MagifyClient : IMagifyClient
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get();

        [NotNull]
        private readonly string _configPath;
        [CanBeNull]
        private readonly CustomClientId _customClientId;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        [NotNull]
        private readonly SubsystemsCollection _subsystems = new();

        [NotNull]
        private readonly ClientIdProvider _clientIdProvider;
        [NotNull]
        private readonly PlatformAPI _platform;
        [NotNull]
        private readonly ContextFileStorage _contextFileStorage;
        [NotNull]
        private readonly SessionCounter _sessionCounter;
        [NotNull]
        private readonly InternalConfigPrefs _internalConfigPrefs;
        [NotNull]
        private readonly GeneralPrefs _generalPrefs;
        [NotNull]
        private readonly AppStatePrefs _appStatePrefs;
        [NotNull]
        private readonly AghanimPrefs _aghanimPrefs;
        [NotNull]
        private readonly AuthorizationConfigBuilder _authBuilder;
        [NotNull]
        private readonly IServerApi _serverApi;
        [NotNull]
        private readonly IAghanimServerApi _aghanimServerApi;
        [NotNull]
        private readonly AghanimManager _aghanimManager;
        [NotNull]
        private readonly PurchaseTracker _purchaseTracker;
        [NotNull]
        private readonly AnalyticsTracker _analyticsTracker;
        [NotNull]
        private readonly AnalyticsSessionTracker _analyticsSessionTracker;
        [NotNull]
        private readonly AnalyticsService _analyticsService;
        [NotNull]
        private readonly CampaignsTracker _campaignsTracker;
        [NotNull]
        private readonly CampaignsProvider _campaignProvider;
        [NotNull]
        private readonly CampaignsCollection _campaignsCollection;
        [NotNull]
        private readonly SegmentationsCollection _segmentations;
        [NotNull]
        private readonly AssignedAbTests _assignedAbTests;
        [NotNull]
        private readonly LimitsHolder _limitsHolder;
        [NotNull]
        private readonly FeaturesProvider _featuresProvider;
        [NotNull]
        private readonly StoredAppFeaturesProvider _storedAppFeaturesProvider;
        [NotNull]
        private readonly FeaturesUpdater _featuresUpdater;
        [NotNull]
        private readonly DefaultProductsContainer _defaultProducts;
        [NotNull]
        private readonly ContentProvider _contentProvider;
        [NotNull]
        private readonly CampaignUpdateHandler _campaignUpdateHandler;
        [NotNull]
        private readonly LtoCampaignManager _ltoCampaignManager;
        [NotNull]
        private readonly IAppStateManager _appStateManager;
        [NotNull]
        private readonly ContextApplicator _contextApplicator;
        [NotNull]
        private readonly AppVersionProvider _appVersionProvider;
        [NotNull]
        private readonly MagifyWorkersManager _workerManager;
        [NotNull]
        private readonly UniTaskCompletionSource _defaultContextParsingCompletionSource = new();
        [NotNull]
        private readonly UniTaskCompletionSource _savedContextParsingCompletionSource = new();

        private bool _isSetup;

        public bool IsDisposed => _disposables.IsDisposed;
        [NotNull]
        public IAghanimApi AghanimApi => _aghanimManager;
        public IFeatures Features => _featuresProvider;
        public IReactiveProperty<bool> IsGeoIpEnabled { get; }

        [NotNull]
        public IReadOnlyReactiveProperty<Environment> Environment => _generalPrefs.Environment;
        [NotNull]
        public IReadOnlyReactiveProperty<bool> IsSandbox => _internalConfigPrefs.IsSandbox;
        public bool HasChangedEnvironment { get; }
        [NotNull]
        public IReactiveProperty<IPurchaseVerificationHandler> ExternalPurchaseVerificationHandler => _purchaseTracker.ExternalPurchaseVerificationHandler;
        [NotNull]
        public IReactiveProperty<float> VerificationRetryInterval => _purchaseTracker.VerificationRetryInterval;
        [NotNull]
        public IReactiveProperty<bool> SyncStateEnabled => _generalPrefs.SyncStateEnabled;
        [NotNull]
        public IUniTaskSource<bool> ClientIdFromCloudLoadingPromise => _clientIdProvider.ClientIdFromCloudLoadingPromise;
        [NotNull]
        public IObservable<SyncStateResult> OnRestoreStateCompleted => _appStateManager.OnRestoreStateCompleted;
        [NotNull]
        public IReadOnlyReactiveProperty<AutoRestoreStateInfo> AutoRestoreStateInfo => _appStateManager.AutoRestoreStateInfo;
        [NotNull]
        public IReactiveProperty<bool> IsAutoRestoreStateEnabled => _appStateManager.IsAutoRestoreStateEnabled;
        public bool HasSocialAuthorizationData => _appStateManager.HasSocialAuthorizationData;
        public IReactiveProperty<SubscriptionStatus> SubscriptionStatus => _generalPrefs.SubscriptionStatus;
        public IReactiveProperty<InAppStatus> InAppStatus => _generalPrefs.InAppStatus;
        public IReactiveProperty<AuthorizationStatus> AuthorizationStatus => _generalPrefs.AuthorizationStatus;
        public IReactiveProperty<string> AdjustId => _generalPrefs.AdjustId;
        public IReactiveProperty<string> FirebaseInstanceId => _generalPrefs.FirebaseInstanceId;
        public IReactiveProperty<string> ReferrerId => _generalPrefs.ReferrerId;
        public IReactiveProperty<bool> IsGdprGranted => _generalPrefs.IsGdprGranted;
        public IReadOnlyReactiveProperty<string> GpsAdId => _generalPrefs.GpsAdId;
        public IReadOnlyReactiveProperty<bool?> AttAuthorizationStatus => _generalPrefs.AttAuthorizationStatus;
        [NotNull]
        public IReadOnlyReactiveProperty<long> LastSuccessfulSyncProgressUnixTimeSeconds => _appStatePrefs.LastSyncTime;
        public bool ShouldSendLastSyncProgressTime
        {
            get => _serverApi.ShouldSendLastSyncProgressTime;
            set => _serverApi.ShouldSendLastSyncProgressTime = value;
        }

        [NotNull]
        public IReactiveProperty<ConfigScope> RemoteContextScopes => _generalPrefs.RemoteContextScopes;

        [NotNull]
        public string AppVersion
        {
            get => _appVersionProvider.AppVersion;
            set
            {
                _appVersionProvider.Custom(value);
                _serverApi.ResetAuthorizationToken();
            }
        }

        public bool IsTablet => _platform.IsTablet;

        public bool IsPortrait => Screen.width < Screen.height;

        public ContextSyncTime LastContextSyncTime { get; private set; }

        public IReactiveProperty<TimeSpan> SessionsInterval => _sessionCounter.Interval;

        public string ClientId => _generalPrefs.ClientId.Value;
        public IReadOnlyReactiveProperty<string> CommonClientId => _clientIdProvider.CommonClientId;
        public bool CustomClientIdWasSet => _clientIdProvider.CustomClientIdWasSet.Value;
        public LimitsModel Limits => _limitsHolder.GetActiveLimits();

        public IReadOnlyReactiveProperty<int> SessionNumber => _generalPrefs.GlobalSessionCounter;
        public string FirstInstalledVersion => _generalPrefs.FirstInstalledVersion;
        public DateTime FirstLaunchDate => _generalPrefs.FirstLaunchDate;

        public IReadOnlyList<InAppProduct> InAppProducts => _defaultProducts.InAppProducts;

        public IReadOnlyList<SubscriptionProduct> SubscriptionProducts => _defaultProducts.SubscriptionProducts;

        [NotNull]
        public IReadOnlyReactiveProperty<IReadOnlyList<string>> Segmentations => _segmentations.Values;

        [NotNull]
        public IReadOnlyReactiveProperty<IReadOnlyList<AssignedAbTest>> AssignedAbTests => _assignedAbTests.Values;

        [NotNull]
        public IStoredAppFeaturesCollection StoredAppFeaturesProvider => _storedAppFeaturesProvider;

        public PlatformAPI Platform => _platform;

        public bool SkipClientIdFromCloudLoading
        {
            get => _clientIdProvider.SkipClientIdFromCloudLoading;
            set => _clientIdProvider.SkipClientIdFromCloudLoading = value;
        }

        public uint ClientIdFromCloudLoadingSecondsTimeout
        {
            get => _clientIdProvider.ClientIdFromCloudLoadingSecondsTimeout;
            set => _clientIdProvider.ClientIdFromCloudLoadingSecondsTimeout = value;
        }

        public event Action<ConfigKind> OnConfigParsed;
        public event Action<ConfigKind> OnConfigParsedOnMainThread;
        public event Action OnConfigLoaded;
        public event Action OnConfigSyncRequested;

        public event Action<LtoInfo> OnOfferAdded
        {
            add => _ltoCampaignManager.OnAdded += value;
            remove => _ltoCampaignManager.OnAdded -= value;
        }

        public event Action<LtoInfo> OnOfferUpdated
        {
            add => _ltoCampaignManager.OnUpdated += value;
            remove => _ltoCampaignManager.OnUpdated -= value;
        }

        public event Action<LtoInfo> OnOfferRemoved
        {
            add => _ltoCampaignManager.OnRemoved += value;
            remove => _ltoCampaignManager.OnRemoved -= value;
        }

        public event Action<LtoInfo> OnOfferFinished
        {
            add => _ltoCampaignManager.OnFinished += value;
            remove => _ltoCampaignManager.OnFinished -= value;
        }

        public event Action OnPurchasedProductsChanged;
#pragma warning disable CS0067 // The event is never used
        // ToDo: native screenshot tracking implementation
        public event Action OnUserDidTakeScreenshot;
#pragma warning restore
        public event Action OnApplicationEnterForeground;
        /// <inheritdoc cref="PlatformAndroid.OnBackground"/>
        public event Action OnApplicationEnterBackground;

        /// <inheritdoc/>
        internal MagifyClient([NotNull] string appName, [NotNull] string configPath, [NotNull] string rootPath, [NotNull] ClientStateConfig clientStateConfig, Environment environment, EditorDevice editorDevice = default)
            : this(appName, configPath, rootPath, customClientId: null, clientStateConfig, environment, false, editorDevice, new NetworkClient())
        {
        }
		/// <inheritdoc/>
        internal MagifyClient([NotNull] string appName, [NotNull] string configPath, [NotNull] string rootPath, [NotNull] ClientStateConfig clientStateConfig, Environment environment, bool isSandbox, EditorDevice editorDevice = default)
            : this(appName, configPath, rootPath, customClientId: null, clientStateConfig, environment, isSandbox, editorDevice, new NetworkClient())
        {
        }

        /// <inheritdoc/>
        internal MagifyClient([NotNull] string appName, [NotNull] string configPath, [NotNull] string rootPath, [NotNull] ClientStateConfig clientStateConfig, Environment environment, EditorDevice editorDevice, INetworkClient networkClient)
            : this(appName, configPath, rootPath, customClientId: null, clientStateConfig, environment, isSandbox: false, editorDevice, networkClient)
        {
        }

        /// <inheritdoc cref="MagifyDocs.CtorWithMagifyCustomClientIdWasJustSetException"/>
        internal MagifyClient([NotNull] string appName, [NotNull] string configPath, [NotNull] string rootPath, [CanBeNull] CustomClientId customClientId, [NotNull] ClientStateConfig clientStateConfig, Environment environment, bool isSandbox, EditorDevice editorDevice, INetworkClient networkClient)
        {
            _configPath = configPath;
            _customClientId = customClientId;

            _subsystems.AddTo(_disposables);

            var internalConfigPrefsPath = Path.Combine(rootPath, IMagifyClient.InternalConfigPrefsName);
            _internalConfigPrefs = InternalConfigPrefs.Create(internalConfigPrefsPath).AddTo(_subsystems);
            if (_internalConfigPrefs.IsSandbox.Value != isSandbox)
            {
                if (Directory.Exists(rootPath))
                    Directory.Delete(rootPath, true);
                HasChangedEnvironment = true;
                _internalConfigPrefs.IsSandbox.Value = isSandbox;
            }

            _appVersionProvider = new AppVersionProvider().AddTo(_subsystems);
            _platform = PlatformAPI.PreparePlatformBridge(editorDevice, rootPath, _internalConfigPrefs).AddTo(_subsystems);
            _contextFileStorage = new ContextFileStorage(rootPath).AddTo(_subsystems);
            _contextApplicator = new ContextApplicator(_subsystems).AddTo(_subsystems);
            var generalPrefsPath = Path.Combine(rootPath, IMagifyClient.GeneralPrefsName);
            var localGeneralPrefsPath = Path.Combine(rootPath, IMagifyClient.LocalGeneralPrefsName);
            _generalPrefs = GeneralPrefs.Create(generalPrefsPath, localGeneralPrefsPath, _appVersionProvider).AddTo(_subsystems);
            _appStatePrefs = AppStatePrefs.Create(Path.Combine(rootPath, IMagifyClient.AppStatePrefsName)).AddTo(_subsystems);
            _aghanimPrefs = AghanimPrefs.Create(Path.Combine(rootPath, IMagifyClient.AghanimPrefsName)).AddTo(_subsystems);

            HasChangedEnvironment = HasChangedEnvironment || _generalPrefs.Environment.Value != environment;
            _generalPrefs.Environment.Value = environment;

            _clientIdProvider = new ClientIdProvider(_generalPrefs, _appStatePrefs, _platform, customClientId).AddTo(_subsystems);

            var offersPrefs = OffersPrefs.Create(Path.Combine(rootPath, IMagifyClient.OffersPrefsName)).AddTo(_subsystems);
            var countersStorage = new CountersStorage(Path.Combine(rootPath, IMagifyClient.CountersFolderName)).AddTo(_subsystems);
            var counters = new Counters(countersStorage, AlternativeCountersGetter).AddTo(_subsystems);
            _authBuilder = new AuthorizationConfigBuilder(appName, _appVersionProvider, _platform, _generalPrefs, _internalConfigPrefs, isBasicIntegration: false).AddTo(_subsystems);

            _campaignsCollection = new CampaignsCollection().AddTo(_subsystems);
            _segmentations = new SegmentationsCollection().AddTo(_subsystems);
            _assignedAbTests = new AssignedAbTests().AddTo(_subsystems);

            _serverApi = new ServerApi(EndpointUrl.FromEnvironment(Environment.Value), networkClient, _generalPrefs, _appStatePrefs, _authBuilder, _platform).AddTo(_subsystems);
            _aghanimServerApi = new AghanimServerApi(EndpointUrl.FromEnvironment(Environment.Value), networkClient, _serverApi.AuthorizationToken, _serverApi.GetAuthorizationTokenAsync).AddTo(_subsystems);
            _aghanimManager = new AghanimManager(_aghanimServerApi, _aghanimPrefs).AddTo(_subsystems);
            _appStateManager = _clientIdProvider.CustomClientIdWasSet.Value || customClientId != null
                ? new DisabledAppStateManager("Custom client ID has been set. Application state sync does not support working with custom client ID's yet.")
                : new AppStateManager(_platform, _serverApi, _generalPrefs, _clientIdProvider, countersStorage, _appStatePrefs, clientStateConfig, rootPath).AddTo(_subsystems);
            _analyticsTracker = new AnalyticsTracker(_serverApi, _generalPrefs, _platform, _appStatePrefs, rootPath, _appVersionProvider).AddTo(_subsystems);
            _analyticsSessionTracker = new AnalyticsSessionTracker(_analyticsTracker, _generalPrefs, _appStatePrefs, _platform, _appVersionProvider).AddTo(_subsystems);

            _sessionCounter = new SessionCounter(_generalPrefs, _platform).AddTo(_subsystems);
            _limitsHolder = new LimitsHolder(_generalPrefs, _sessionCounter).AddTo(_subsystems);
            _campaignsTracker = new CampaignsTracker(_generalPrefs, counters, _limitsHolder, _platform).AddTo(_subsystems);
            _ltoCampaignManager = new LtoCampaignManager(_campaignsTracker, _campaignsCollection, _generalPrefs, offersPrefs, _platform, counters, rootPath, _appVersionProvider).AddTo(_subsystems);
            _featuresProvider = new FeaturesProvider(_generalPrefs, _analyticsTracker).AddTo(_subsystems);
            _storedAppFeaturesProvider = new StoredAppFeaturesProvider(_generalPrefs).AddTo(_subsystems);
            _featuresUpdater = new FeaturesUpdater(_featuresProvider, _storedAppFeaturesProvider, _generalPrefs).AddTo(_subsystems);
            _campaignProvider = new CampaignsProvider(_generalPrefs, _campaignsTracker, _campaignsCollection, counters, _limitsHolder, _platform, _ltoCampaignManager).AddTo(_subsystems);

            _purchaseTracker = new PurchaseTracker(_serverApi, _platform, rootPath).AddTo(_subsystems);
            _analyticsService = new AnalyticsService(_analyticsTracker, _campaignsTracker).AddTo(_subsystems);

            _defaultProducts = new DefaultProductsContainer().AddTo(_subsystems);
            _contentProvider = new ContentProvider().AddTo(_subsystems);
            _campaignUpdateHandler = new CampaignUpdateHandler(_campaignsTracker, _campaignProvider, _campaignsCollection, _platform, _generalPrefs, _ltoCampaignManager).AddTo(_subsystems);

            var workers = new IMagifyWorker[]
            {
                new MagifyContextSyncWorker(() => OnConfigSyncRequested?.Invoke(), syncTime => LastContextSyncTime = syncTime, _serverApi, _contextApplicator, _contextFileStorage),
                new SetupSavedContextWorker(_contextFileStorage, _contextApplicator),
                new SetupDefaultContextWorker(_platform, _contextApplicator),
            };
            _workerManager = new MagifyWorkersManager(workers).AddTo(_subsystems);

            IsGeoIpEnabled = _serverApi.IsGeoIpEnabled;
        }

        [CanBeNull]
        private int? AlternativeCountersGetter(CounterType type, CounterScope scope, CounterKey key)
        {
            if (!_generalPrefs.MigratedUser) return null;
            try
            {
                return _platform.GetGlobalCounterFromLegacyData(type, key);
            }
            catch (Exception e)
            {
                _logger.LogException(e);
            }
            return null;
        }

        public void InitializeSdk()
        {
            using(_generalPrefs.MultipleChangeScope())
                _subsystems.PreInitializeAll();
            try
            {
                if (!_generalPrefs.ClientId.HasValue || string.IsNullOrEmpty(_generalPrefs.ClientId.Value))
                {
                    var legacyData = _platform.LoadLegacyNativeDataForMigration();
#if MAGIFY_VERBOSE_LOGGING
                    try
                    {
                        // Remove after tests
                        if (!Directory.Exists(_contextFileStorage.RootFolderPath))
                            Directory.CreateDirectory(_contextFileStorage.RootFolderPath);
                        File.WriteAllText(Path.Combine(_contextFileStorage.RootFolderPath, "migrationdata.json"), legacyData?.ToString() ?? "Empty");
                    }
                    catch (Exception e)
                    {
                        _logger.LogException(e);
                    }
#endif
                    if (legacyData != null)
                    {
                        _subsystems.MigrateAll(legacyData);
                        _platform.CleanLegacyNativeDataAfterMigration();
                    }
                }
            }
            catch (Exception e)
            {
                _logger.LogException(new MagifyMigrationFailedException("All", string.Empty, e));
            }

            _contextApplicator.ContextApplied
                .Subscribe(contextKind =>
                {
                    _logger.Log($"{contextKind} context has been applied");
                    if (contextKind is ContextKind.Default)
                        _defaultContextParsingCompletionSource.TrySetResult();
                    if (contextKind is ContextKind.Saved)
                        _savedContextParsingCompletionSource.TrySetResult();
                    OnConfigParsed?.Invoke(contextKind.ToConfigKind());
                })
                .AddTo(_disposables);
            _contextApplicator.ContextSkipped
                .Subscribe(contextKind =>
                {
                    _logger.Log($"{contextKind} context has been skipped");
                    if (contextKind is ContextKind.Default)
                        _defaultContextParsingCompletionSource.TrySetResult();
                    if (contextKind is ContextKind.Saved)
                        _savedContextParsingCompletionSource.TrySetResult();
                })
                .AddTo(_disposables);
            _contextApplicator.ContextAppliedOnMainThread
                .Subscribe(contextKind => OnConfigParsedOnMainThread?.Invoke(contextKind.ToConfigKind()))
                .AddTo(_disposables);
            _contextApplicator.ContextAppliedOnMainThread
                .Where(kind => kind is ContextKind.Downloaded)
                .Subscribe(_ => OnConfigLoaded?.Invoke())
                .AddTo(_disposables);

            CheckInAppStatusIsActual();
            try
            {
                _workerManager.DoJob(new SetupSavedContextJob());
            }
            catch (Exception e) { _logger.LogException(new MagifyFailedToLoadContextFromDiskException(ContextKind.Saved, e)); }
            try
            {
                _workerManager.DoJob(new SetupDefaultContextJob(_configPath));
            }
            catch (Exception e) { _logger.LogException(new MagifyFailedToLoadContextFromDiskException(ContextKind.Default, e)); }

            if (_generalPrefs.VersionSessionCounter == 0)
            {
                PrepareForFirstSessionInVersion();
            }

            Environment
                .SkipLatestValueOnSubscribe()
                .Subscribe(value => _serverApi.ChangeServerUrl(EndpointUrl.FromEnvironment(value)))
                .AddTo(_disposables);
            RemoteContextScopes
                .SkipLatestValueOnSubscribe()
                .Subscribe(_ => _serverApi.CancelAllContextLoadings())
                .AddTo(_disposables);
            _platform.OnLocaleChanged
                .Subscribe(localeData => _serverApi.RefreshLocale(localeData).IfTrue(() => _serverApi.CancelAllContextLoadings().ContinueWith(Sync)))
                .AddTo(_disposables);
            _platform.OnForeground
                .Subscribe(_ =>
                {
                    _subsystems.ForEach<IForegroundListener>(s => s?.OnForeground());
                    OnApplicationEnterForeground?.Invoke();
                })
                .AddTo(_disposables);
            _platform.OnBackground
                .Subscribe(_ =>
                {
                    _subsystems.ForEach<IBackgroundListener>(s => s?.OnBackground());
                    OnApplicationEnterBackground?.Invoke();
                })
                .AddTo(_disposables);
            _generalPrefs.AttAuthorizationStatus
                .SkipLatestValueOnSubscribe()
                .Subscribe(CancelRequestsAndSync)
                .AddTo(_disposables);
            _generalPrefs.SessionId
                .SkipLatestValueOnSubscribe()
                .Where(_ => !(_appStatePrefs.IsPendingRestore.Value || _appStatePrefs.IsRestoreInProgress.Value))
                .Subscribe(Sync)
                .AddTo(_disposables);
            if (!_clientIdProvider.CustomClientIdWasSet.Value)
            {
                _appStateManager.OnRestoreStateCompleted
                    .Subscribe(RestoreStateCompletedHandler)
                    .AddTo(_disposables);
                _clientIdProvider.CloudClientIdReplacedLocal
                    .Where(replaced => replaced is true)
                    .Subscribe(_ => _serverApi.CancelAllServerInteractions().Forget())
                    .AddTo(_disposables);
            }
#if MAGIFY_VERBOSE_LOGGING
            _platform.OnForeground
                .Subscribe(_ => _logger.Log($"{nameof(_platform.OnForeground)} called"))
                .AddTo(_disposables);
            _platform.OnBackground
                .Subscribe(_ => _logger.Log($"{nameof(_platform.OnBackground)} called"))
                .AddTo(_disposables);
            _platform.OnApplicationDidBecomeActive
                ?.Subscribe(_ => _logger.Log($"{nameof(_platform.OnApplicationDidBecomeActive)} called"))
                .AddTo(_disposables);
            _platform.OnApplicationWillResignActive
                ?.Subscribe(_ => _logger.Log($"{nameof(_platform.OnApplicationWillResignActive)} called"))
                .AddTo(_disposables);
#endif

            _campaignsTracker.OnPurchasedProductsChanged += () =>
            {
                _logger.Log($"{nameof(OnPurchasedProductsChanged)} has been called.");
                OnPurchasedProductsChanged?.Invoke();
            };

            using(_generalPrefs.MultipleChangeScope())
                _subsystems.InitializeAll();
            _generalPrefs.LastInitializedVersion = _appVersionProvider.AppVersion;
        }

        public UniTask AwaitInitialConfigParsing(CancellationToken cancellationToken)
        {
            if (_defaultContextParsingCompletionSource.IsCompleted() && _savedContextParsingCompletionSource.IsCompleted())
            {
                return UniTask.CompletedTask;
            }
            var task = UniTask.WhenAll(_defaultContextParsingCompletionSource.Task, _savedContextParsingCompletionSource.Task);
            return task.AttachExternalCancellation(cancellationToken);
        }

        private void CheckInAppStatusIsActual()
        {
            if (_generalPrefs.PurchasedInAppProducts is { Count: > 0 })
            {
                _logger.Log($"There is {_generalPrefs.PurchasedInAppProducts} purchased InApp products, {nameof(Magify.InAppStatus)} will be changed to {Magify.InAppStatus.Purchased}");
                _generalPrefs.InAppStatus.Value = Magify.InAppStatus.Purchased;
            }
            else
            {
                _logger.Log($"There is no purchased in app products, {nameof(CheckInAppStatusIsActual)} will be skipped");
            }
        }

        public void Setup()
        {
            if (_isSetup)
            {
                _logger.Log("Magify client has already been setup.");
                return;
            }
            _isSetup = true;

            Sync();
        }

        public void Sync<T>(T _) => Sync();
        public void Sync()
        {
            if (!_isSetup)
            {
                _logger.Log("Cannot perform sync before initial setup.");
                return;
            }
            if (IsDisposed)
            {
                _logger.Log("Cannot perform sync after disposal.");
                return;
            }

            _workerManager.DoJob(new MagifyContextSyncJob(RemoteContextScopes.Value, _disposables.GetOrCreateToken()));
        }

        private void CancelRequestsAndSync<T>(T _) => CancelRequestsAndSync();
        private void CancelRequestsAndSync()
        {
            _serverApi.CancelAllServerInteractions().ContinueWith(Sync);
        }

        private void RestoreStateCompletedHandler([CanBeNull] SyncStateResult result)
        {
            if (result is { InternalStateSuccessfullyRestored: true } || _clientIdProvider.CloudClientIdReplacedLocal.Value is true)
            {
                _authBuilder.UpdateConfig();
                CheckInAppStatusIsActual();
                ResetContext(cancelIfLoading: true);
                ResetCachedConfig();
                _serverApi.ResetAuthorizationToken();
                _campaignsTracker.UpdateRelatedCounters();
                _sessionCounter.ForceStartNewSession();
                Sync();
                _analyticsTracker.ForceSync();
                _ltoCampaignManager.OnAppStateChanged();
            }
        }

        public ICampaign CampaignFor(string eventName, IReadOnlyDictionary<string, object> customParams = null, bool silent = false)
        {
            if (silent)
            {
                return _campaignProvider.GetCampaignSilently(eventName, customParams);
            }

            var result = _campaignProvider.ProvideCampaign(eventName, customParams);
            if (result == null)
            {
                return null;
            }

            if (result.IsDefaultCampaign)
            {
                _generalPrefs.UsedDefaultCampaignTypes.Add(result.Campaign.Type.ToEnumString());
            }

            _campaignUpdateHandler.ClearSubscriptions(result.Campaign.Name);
            return result.Campaign;
        }

        public bool IsCampaignAvailable(string campaignName, string eventName, Dictionary<string, object> customParams = null)
        {
            return _campaignProvider.IsCampaignAvailable(campaignName, eventName, customParams);
        }

        [NotNull]
        public IReadOnlyCollection<LtoInfo> GetActiveLtoOffers()
        {
            return _ltoCampaignManager.GetActiveLtoOffers();
        }

        public void CompleteOffer(string campaignName)
        {
            _ltoCampaignManager.CompleteOffer(campaignName);
        }

        public CampaignImpression GetCampaignImpression(string campaignName)
        {
            return _campaignsTracker.CreateCampaignImpression(campaignName);
        }

        public IList<ContentItem> GetContentList([NotNull] string group, [NotNull] string key, [CanBeNull] ICollection<string> tags = null)
        {
            return _contentProvider.GetContentList(group, key, tags);
        }

        public ContentItem GetEarliestContent(string group, string key)
        {
            return _contentProvider.GetEarliestContent(group, key);
        }

        public ContentItem GetLatestContent(string group, string key)
        {
            return _contentProvider.GetLatestContent(group, key);
        }

        public bool HasProcessedPurchase(string productId)
        {
            return _campaignsTracker.AllUsedProducts.Contains(productId);
        }

        public bool SubscribeCampaignUpdates([NotNull] ICampaign campaign, Action onUpdate)
        {
            if (_campaignsTracker.GetCampaignRequest(campaign.Name)?.Campaign != campaign) return false;
            _campaignUpdateHandler.SubscribeCampaignUpdates(campaign.Name, onUpdate);
            return true;
        }

        public void UnsubscribeCampaignUpdates(string campaignName, Action onUpdate)
        {
            _campaignUpdateHandler.UnsubscribeCampaignUpdates(campaignName, onUpdate);
        }

        public void ClearSubscriptions(string campaignName)
        {
            _campaignUpdateHandler.ClearSubscriptions(campaignName);
        }

        private void ResetCachedConfig()
        {
            _campaignsCollection.ResetCurrentCampaigns();
            _limitsHolder.Reset();
            _ltoCampaignManager.ResetContext();
            _defaultProducts.ResetDefaultProducts();
            _contentProvider.ResetContentMap();
            _featuresProvider.ResetFeatures();

            _workerManager.DoJob(new SetupDefaultContextJob(_configPath));
        }

        public void TweakAnalyticsConfig(int eventsGroupSize, int syncTimeInterval)
        {
            _analyticsService.SetupAnalyticsConfig(eventsGroupSize, syncTimeInterval);
        }

        public void ResetAnalyticsConfig()
        {
            _analyticsService.ResetAnalyticsConfig();
        }

        public UniTask<string> ForceReissueAuthToken(CancellationToken cancellationToken)
        {
            _serverApi.ResetAuthorizationToken();
            return _serverApi.GetAuthorizationTokenUntilSuccessAsync(cancellationToken);
        }

        public void RefreshGpsAdId()
        {
            _serverApi.RefreshGpsAdId();
        }

        public void TweakFirstLaunchDate(DateTime date)
        {
            AssertProductionTweak(string.Intern("first launch date"));
            if (_isSetup)
            {
                _logger.Log($"Tweak first launch date to: {date.ToString()}");
                _generalPrefs.FirstLaunchDate = date;
                _authBuilder.UpdateConfig();
                _serverApi.ResetAuthorizationToken();
                Sync();
            }
        }

        public void TweakUserLocale([NotNull] string languageTag)
        {
            AssertProductionTweak(string.Intern("user locale"));
            if (_isSetup)
            {
                _logger.Log($"Tweak user locale to: {languageTag}");
                _platform.TweakUserLocale(languageTag);
            }
        }

        public void SetAttStatus(bool authorized)
        {
            _generalPrefs.AttAuthorizationStatus.Value = authorized;
        }

        public void SetGameLevel(int level)
        {
            _analyticsService.SetGameLevel(level);
        }

        public void SetGameMaxLevel(int maxLevel)
        {
            _analyticsService.SetGameMaxLevel(maxLevel);
        }

        public void SetGameMode([CanBeNull] string mode)
        {
            _analyticsService.SetGameMode(mode);
        }

        public void SetUserEmail(string email)
        {
            _analyticsTracker.TrackMailingStatusMappingEvent(email);
        }

        public UniTask<bool> RequestSocialAuthTokenFor([NotNull] string provider, [NotNull] string token, CancellationToken cancellationToken)
        {
            return _appStateManager.RequestSocialAuthTokenFor(provider, token, cancellationToken);
        }

        public void ResetSocialAuthToken()
        {
            _appStateManager.ResetSocialAuthToken();
        }

        public UniTask<SyncStateResult> SaveState(int? weight, CancellationToken cancellationToken)
        {
            return _appStateManager.SaveState(weight, cancellationToken);
        }

        public UniTask<SyncStateResult> RestoreState(int? weight, CancellationToken cancellationToken)
        {
            return _appStateManager.RestoreState(weight, cancellationToken);
        }

        public void SetMediaSource(string networkName, string campaignName, string adGroup)
        {
            _logger.Log($"{nameof(SetMediaSource)} with networkName={networkName}, campaignName={campaignName}, adGroup={adGroup}");
            var mediaSource = new MediaSource(networkName, campaignName, adGroup);
            if (_generalPrefs.MediaSource.Equals(mediaSource))
            {
                return;
            }
            _generalPrefs.MediaSource = mediaSource;
            Sync();
        }

        public void SetupConversionTracker(string revenuePerCountryPath, string revenueLevelsPath, string defaultCurrencyRatesPath, string subscriptionMultipliersPath)
        {
            _logger.Log($"Setup conversion tracker.\n{nameof(revenuePerCountryPath)}: {revenuePerCountryPath}\n{nameof(revenueLevelsPath)}: {revenueLevelsPath}\n{nameof(defaultCurrencyRatesPath)}: {defaultCurrencyRatesPath}\n{nameof(subscriptionMultipliersPath)}: {subscriptionMultipliersPath}");
            _platform.SetupConversionTracker(revenuePerCountryPath, revenueLevelsPath, defaultCurrencyRatesPath, subscriptionMultipliersPath);
        }

        public void TrackCustomEvent(string eventName, Dictionary<string, object> customParams = null)
        {
            _analyticsService.TrackCustomSessionEvent(eventName, customParams);
        }

        public void TrackImpression(string campaignName)
        {
            _campaignsTracker.TrackImpression(campaignName);
            _analyticsService.TrackCampaignImpression(campaignName, includeNested: true);
        }

        public void TrackParentCampaignImpression(string campaignName)
        {
            _campaignsTracker.TrackImpression(campaignName);
            _analyticsService.TrackCampaignImpression(campaignName, includeNested: false);
        }

        public void TrackProductsImpression([CanBeNull] string campaignName, IReadOnlyList<string> productIds)
        {
            _analyticsService.TrackNestedCampaignImpressions(campaignName, productIds);
        }

        public void TrackImpressionFailFor([CanBeNull] string campaignName, string reason)
        {
            _analyticsService.TrackImpressionFailEvent(campaignName, reason);
        }

        public void TrackAdsImpression(string campaignName, IAdsImpression impression)
        {
            _campaignsTracker.TrackAdsImpression(campaignName);
            _analyticsService.TrackAdsImpression(campaignName, impression);
        }

        public void TrackClickFor(string campaignName)
        {
            _campaignsTracker.TrackClick(campaignName);
            _analyticsService.TrackClickEvent(EventType.Click, campaignName);
        }

        public void TrackProductClickFor(string campaignName, string productId, PurchaseStore? store)
        {
            _campaignsTracker.TrackClick(campaignName, productId, store);
            _analyticsService.TrackClickEvent(EventType.Click, campaignName, productId, store);
        }

        public void TrackAdsClickFor(string campaignName)
        {
            _campaignsTracker.TrackClick(campaignName);
            _analyticsService.TrackClickEvent(EventType.AdsClick, campaignName);
        }

        public void TrackAdsProductClickFor(string campaignName, string productId)
        {
            _analyticsService.TrackClickEvent(EventType.AdsClick, campaignName, productId);
        }

        public void TrackSubscriptionActivation(bool isTrial, string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt, bool needValidation, PurchaseStore store)
        {
            _analyticsService.TrackSubscription(isTrial, productId, price, currency, transactionId, originalTransactionId, store, Option<string>.None);
            _campaignsTracker.TrackSubscriptionPurchase(productId, InAppSourceKind.Internal);
            _featuresUpdater.OnProductPurchase();
            _ltoCampaignManager.OnProductPurchase(productId);
            _campaignUpdateHandler.OnProductUsed(productId);
            if (needValidation)
            {
                _purchaseTracker.ValidateSubscription(productId, transactionId, purchaseToken, receipt);
            }
        }

        public void TrackExternalSubscriptionActivation(bool isTrial, string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt, bool needValidation, PurchaseStore store)
        {
            _campaignsTracker.TrackSubscriptionPurchase(productId, InAppSourceKind.External);
            _ltoCampaignManager.OnProductPurchase(productId);
            _featuresUpdater.OnProductPurchase();

            var purchaseInfo = new PurchaseInfo
            {
                Product = new ProductInfo(productId, price, currency, store),
                TransactionId = transactionId,
                OriginalTransactionId = originalTransactionId,
            };

            var eventType = isTrial ? EventType.TrialActivation : EventType.PaidSubscriptionActivation;
            _analyticsTracker.TrackPurchaseEvent(eventType, _campaignsTracker.BuildExternalSubscriptionImpression(productId), purchaseInfo);
            if (needValidation)
            {
                _purchaseTracker.ValidateSubscription(productId, transactionId, purchaseToken, receipt);
            }
        }

        public void TrackInAppFor(string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt, bool needValidation, PurchaseStore store)
        {
            _analyticsService.TrackInApp(isExternal: false, productId, price, currency, transactionId, originalTransactionId, store, Option<string>.None);
            _campaignsTracker.TrackInAppPurchase(productId, InAppSourceKind.Internal);
            _featuresUpdater.OnProductPurchase();
            _ltoCampaignManager.OnProductPurchase(productId);
            _campaignUpdateHandler.OnProductUsed(productId);
            if (needValidation)
            {
                _purchaseTracker.ValidateInApp(productId, transactionId, purchaseToken, receipt);
            }
        }

        public void TrackExternalInAppFor(string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt, bool needValidation, PurchaseStore store)
        {
            _analyticsService.TrackInApp(isExternal: true, productId, price, currency, transactionId, originalTransactionId, store, Option<string>.None);
            _campaignsTracker.TrackInAppPurchase(productId, InAppSourceKind.External);
            _featuresUpdater.OnProductPurchase();
            _ltoCampaignManager.OnProductPurchase(productId);
            _campaignUpdateHandler.OnProductUsed(productId);
            if (needValidation)
            {
                _purchaseTracker.ValidateInApp(productId, transactionId, purchaseToken, receipt);
            }
        }

        public void TrackTrustedPurchase([NotNull] TrustedPurchaseRecord record, bool? isSubscription, bool? isExternal)
        {
            if (isSubscription is null && isExternal is null)
                _purchaseTracker.TrackTrustedPurchase(record);
            else if (isSubscription is true && isExternal is true)
                TrackExternalTrustedSubscriptionActivation(record);
            else if (isSubscription is true && isExternal is false)
                TrackTrustedSubscriptionActivation(record);
            else if (isExternal is true)
                TrackExternalTrustedInAppFor(record);
            else TrackTrustedInAppFor(record);
        }

        private void TrackTrustedInAppFor([NotNull] TrustedPurchaseRecord record)
        {
            _analyticsService.TrackInApp(isExternal: false, record.ProductId, record.Price, record.Currency, record.TransactionId, record.OriginalTransactionId, record.StoreName, new Option<string>(record.StoreFront));
            _campaignsTracker.TrackInAppPurchase(record.ProductId, InAppSourceKind.Internal);
            _featuresUpdater.OnProductPurchase();
            _ltoCampaignManager.OnProductPurchase(record.ProductId);
            _campaignUpdateHandler.OnProductUsed(record.ProductId);
            _purchaseTracker.TrackTrustedPurchase(record);
        }

        private void TrackExternalTrustedInAppFor([NotNull] TrustedPurchaseRecord record)
        {
            _analyticsService.TrackInApp(isExternal: true, record.ProductId, record.Price, record.Currency, record.TransactionId, record.OriginalTransactionId, record.StoreName, new Option<string>(record.StoreFront));
            _campaignsTracker.TrackInAppPurchase(record.ProductId, InAppSourceKind.External);
            _featuresUpdater.OnProductPurchase();
            _ltoCampaignManager.OnProductPurchase(record.ProductId);
            _campaignUpdateHandler.OnProductUsed(record.ProductId);
            _purchaseTracker.TrackTrustedPurchase(record);
        }

        private void TrackTrustedSubscriptionActivation([NotNull] TrustedPurchaseRecord record)
        {
            _analyticsService.TrackSubscription(record.PeriodType is PeriodType.Trial, record.ProductId, record.Price, record.Currency, record.TransactionId, record.OriginalTransactionId, record.StoreName, new Option<string>(record.StoreFront));
            _campaignsTracker.TrackSubscriptionPurchase(record.ProductId, InAppSourceKind.Internal);
            _featuresUpdater.OnProductPurchase();
            _ltoCampaignManager.OnProductPurchase(record.ProductId);
            _campaignUpdateHandler.OnProductUsed(record.ProductId);
            _purchaseTracker.TrackTrustedPurchase(record);
        }

        private void TrackExternalTrustedSubscriptionActivation([NotNull] TrustedPurchaseRecord record)
        {
            _campaignsTracker.TrackSubscriptionPurchase(record.ProductId, InAppSourceKind.External);
            _ltoCampaignManager.OnProductPurchase(record.ProductId);
            _featuresUpdater.OnProductPurchase();

            var purchaseInfo = new PurchaseInfo
            {
                Product = new ProductInfo(record.ProductId, record.Price, record.Currency, record.StoreName),
                TransactionId = record.TransactionId,
                OriginalTransactionId = record.OriginalTransactionId,
                CustomStoreFront = new Option<string>(record.StoreFront),
            };

            var eventType = record.PeriodType is PeriodType.Trial ? EventType.TrialActivation : EventType.PaidSubscriptionActivation;
            _analyticsTracker.TrackPurchaseEvent(eventType, _campaignsTracker.BuildExternalSubscriptionImpression(record.ProductId), purchaseInfo);
            _purchaseTracker.TrackTrustedPurchase(record);
        }

        public void TrackRestoredInAppFor(string productId)
        {
            _campaignsTracker.TrackInAppPurchase(productId, InAppSourceKind.External); // restored InApps are considered as external
            _featuresUpdater.OnProductPurchase();
            _ltoCampaignManager.OnProductPurchase(productId);
            _campaignUpdateHandler.OnProductUsed(productId);
        }

        public void TrackRestoredSubscription(string productId)
        {
            _campaignsTracker.TrackSubscriptionPurchase(productId, InAppSourceKind.External); // restored Subscriptions are considered as external
            _featuresUpdater.OnProductPurchase();
            _ltoCampaignManager.OnProductPurchase(productId);
            _campaignUpdateHandler.OnProductUsed(productId);
        }

        public void TrackRewardGranted(string productId)
        {
            _campaignsTracker.TrackRewardGranted(productId);
            _ltoCampaignManager.OnRewardGranted(productId);
            _campaignUpdateHandler.OnProductUsed(productId);
        }

        public void TrackFreeBonusGranted(string productId)
        {
            _campaignsTracker.TrackBonusGranted(productId);
            _ltoCampaignManager.OnBonusGranted(productId);
            _campaignUpdateHandler.OnProductUsed(productId);
        }

        public void TrackOrdinaryProductUsed(string productId)
        {
            _campaignsTracker.TrackOrdinaryProductUsed(productId);
            _campaignUpdateHandler.OnProductUsed(productId);
        }

        public void TrackIncomeTransaction(string source, [NotNull] List<BonusInfo> bonuses, ProductInfo product = null)
        {
            _analyticsService.TrackIncomeTransaction(source, bonuses, product);
        }

        public void TrackExpenseTransaction([NotNull] List<BonusInfo> bonuses)
        {
            _analyticsService.TrackExpenseTransaction(bonuses);
        }

        public void TrackCorrectionTransaction([NotNull] List<BonusInfo> bonuses)
        {
            _analyticsService.TrackCorrectionTransaction(bonuses);
        }

        [CanBeNull]
        public CampaignRequest GetLastCampaignRequest(string campaignName)
        {
            return _campaignsTracker.GetCampaignRequest(campaignName);
        }

        [CanBeNull]
        public CampaignRequest GetLastCampaignRequest(CampaignType campaignType)
        {
            return _campaignsTracker.GetCampaignRequest(campaignType);
        }

        public bool IsApplicationInstalled([CanBeNull] string identifier)
        {
            return _platform.IsApplicationInstalled(identifier);
        }

        internal void HandleContext([NotNull] CampaignsContext context, ContextKind kind)
        {
            _subsystems.UpdateContextForAll(context, kind);
        }

        private void PrepareForFirstSessionInVersion()
        {
            _logger.Log($"Start of preparing for the current App version: {_appVersionProvider.AppVersion}.");
            using (_generalPrefs.MultipleChangeScope())
            {
                _logger.Log("Used Default Campaign Types list will be clear.");
                _generalPrefs.UsedDefaultCampaignTypes.Clear();
                _logger.Log("Authorization Token will be reset.");
                _serverApi.ResetAuthorizationToken();
                _logger.Log("Version counters will be reset.");
                _campaignsTracker.ResetVersionCounters();
                _logger.Log("Notifying analytics session tracker.");
                _analyticsSessionTracker.TrackFirstSessionInVersion();
            }
            _logger.Log("Preparing for the current App version finished.");
        }

        private static EndpointUrl GetEnvironmentUrl(Environment environment)
        {
            return environment switch
            {
                Magify.Environment.Production => EndpointUrl.Production,
                Magify.Environment.Staging => EndpointUrl.Staging,
                Magify.Environment.Offline => EndpointUrl.Offline,
                _ => throw new InvalidMagifyEnvironmentException(environment)
            };
        }

        public void ResetContext(bool cancelIfLoading)
        {
            if (cancelIfLoading) _serverApi.CancelAllContextLoadings();
            _contextFileStorage.ResetContext();
            ResetCachedConfig();
        }

        public void Reset(bool clearNativeStorage = true, bool clearCloudStorage = true)
        {
            _generalPrefs.Reset();
            _appStatePrefs.Reset();
            if (clearNativeStorage) ClearNativeStorage();
            if (clearCloudStorage) _platform.RemoveValueFromCloudStorage(CloudStorageKeys.ClientId);
        }

        /// <inheritdoc cref="PlatformAPI.ClearNativeStorage"/>
        public void ClearNativeStorage()
        {
            _platform.ClearNativeStorage();
        }

        public void Dispose()
        {
            try
            {
                _disposables.Release();
                typeof(MagifyClient).ResetEvents(this);
            }
            catch (Exception e)
            {
                _logger.LogError($"During dispose of MagifyClient an error occurred: {e}");
            }
        }

        private void AssertProductionTweak([NotNull] string tweakName)
        {
            if (Environment.Value is Magify.Environment.Production)
            {
                _logger.LogWarning($"Attempting to tweak the {tweakName} in production mode. This is not recommended as it may break user segmentation.");
            }
        }
    }
}