using System.Linq;
using JetBrains.Annotations;
using Magify.Model;

namespace Magify
{
    internal static class NestedCampaignExtensions
    {
        internal static int? GetLimitForScope(this NestedCampaign nested, NestedCounterScope scope)
        {
            return (nested.Type, scope) switch
            {
                (ProductType.Rewarded, NestedCounterScope.Global) => nested.GlobalRewardLimit,
                (ProductType.Rewarded, NestedCounterScope.Session) => nested.SessionRewardLimit,
                (ProductType.Rewarded, NestedCounterScope.Daily) => nested.DailyRewardLimit,
                (ProductType.Rewarded, NestedCounterScope.Activation) => nested.ActivationRewardLimit,

                (ProductType.Consumable or ProductType.NonConsumable, NestedCounterScope.Global) => nested.GlobalPurchaseLimit,
                (ProductType.Consumable or ProductType.NonConsumable, NestedCounterScope.Session) => nested.SessionPurchaseLimit,
                (ProductType.Consumable or ProductType.NonConsumable, NestedCounterScope.Daily) => nested.DailyPurchaseLimit,
                (ProductType.Consumable or ProductType.NonConsumable, NestedCounterScope.Activation) => nested.ActivationPurchaseLimit,

                (ProductType.Bonus, NestedCounterScope.Global) => nested.GlobalBonusLimit,
                (ProductType.Bonus, NestedCounterScope.Session) => nested.SessionBonusLimit,
                (ProductType.Bonus, NestedCounterScope.Daily) => nested.DailyBonusLimit,
                (ProductType.Bonus, NestedCounterScope.Activation) => nested.ActivationBonusLimit,

                (ProductType.Info or ProductType.ExternalLink or ProductType.InternalLink or ProductType.CrossPromo, NestedCounterScope.Global) => nested.GlobalOrdinaryLimit,
                (ProductType.Info or ProductType.ExternalLink or ProductType.InternalLink or ProductType.CrossPromo, NestedCounterScope.Session) => nested.SessionOrdinaryLimit,
                (ProductType.Info or ProductType.ExternalLink or ProductType.InternalLink or ProductType.CrossPromo, NestedCounterScope.Daily) => nested.DailyOrdinaryLimit,
                (ProductType.Info or ProductType.ExternalLink or ProductType.InternalLink or ProductType.CrossPromo, NestedCounterScope.Activation) => nested.ActivationOrdinaryLimit,

                _ => null
            };
        }

        [CanBeNull]
        public static ProductCreative ExtractProductCreative(this NestedCampaign nestedCampaign)
        {
            return nestedCampaign.CreativeModel?.Info?.ToProductCreative();
        }

        [NotNull]
        public static ProductCreative ToProductCreative(this CardCreativeInfo info)
        {
            return new ProductCreative
            {
                Title = info.Title,
                Text = info.Text,
                Resource = CreateFrom(info),
                ActionButton = info.ActionBtn,
                CloseButton = info.CloseBtn,
                Attributes = info.Attributes
            };
        }

        internal static CreativeResource CreateFrom(CardCreativeInfo info)
        {
            return info.ImageUrl != null
                ? new CreativeResource { Type = CreativeResource.ArtifactType.Image, Url = info.ImageUrl }
                : new CreativeResource { Type = CreativeResource.ArtifactType.Bundle, Url = info.CompressedLottieUrl };
        }

        internal static ReferrerLimits GetReferrerLimits(this NestedCampaign campaign)
        {
            return new ReferrerLimits
            {
                IncludeIds = campaign.IncludeReferrerIds?.Select(r => r.Id).ToHashSet(),
                ExcludeIds = campaign.ExcludeReferrerIds?.Select(r => r.Id).ToHashSet(),
            };
        }

        internal static PurchaseLimits GetPurchaseLimits(this NestedCampaign campaign)
        {
            return new PurchaseLimits
            {
                ExcludeIfAll = campaign.ExcludeIfAllPurchased?.Select(r => r.Product).ToHashSet(),
                ExcludeIfAny = campaign.ExcludeIfAnyPurchased?.Select(r => r.Product).ToHashSet(),
                IncludeIfAll = campaign.IncludeIfAllPurchased?.Select(r => r.Product).ToHashSet(),
                IncludeIfAny = campaign.IncludeIfAnyPurchased?.Select(r => r.Product).ToHashSet(),
            };
        }
    }
}