using System;
using Magify.Model;

namespace Magify
{
    public static class LtoCampaignExtensions
    {
        internal static DateTime GetOfferEndTime(this CampaignModel campaign, DateTime startTime)
        {
            var durationEndTime = campaign.Duration.HasValue ? startTime + campaign.Duration.Value : startTime;
            var forcedFinishTime = campaign.GetForcedFinishTime();

            var endTime = durationEndTime;
            if (forcedFinishTime.HasValue)
            {
                endTime = TimeExtensions.MinOf(durationEndTime, forcedFinishTime.Value);
            }

            if (campaign.ActivationPeriod?.ForceStop == true)
            {
                endTime = TimeExtensions.MinOf(endTime, campaign.ActivationPeriod.GetEndTime());
            }

            return endTime;
        }

        internal static bool MeetsImpressionsPerActivationLimit(this CampaignModel campaign, Counter<ImpressionsCounterScope> counter)
        {
            var limitValue = campaign.ImpressionLimitPerActivation;
            if (limitValue == null)
                return true;

            var triggerImpressions = counter[ImpressionsCounterScope.Activation, CounterKey.GetKeyCampaignBySourceType(campaign.Name, SourceType.Trigger)];
            var eventImpressions = counter[ImpressionsCounterScope.Activation, CounterKey.GetKeyCampaignBySourceType(campaign.Name, SourceType.Event)];
            return triggerImpressions + eventImpressions < limitValue;
        }

        internal static bool IsWithinActivationLimits(this CampaignModel campaign, Counter<LtoStartCounterScope> counter)
        {
            var keyCampaignName = CounterKey.GetKey(campaignName: campaign.Name);
            var global = isWithinLimit(campaign.ActivationLimit, LtoStartCounterScope.Global);
            var daily = isWithinLimit(campaign.DailyActivationLimit, LtoStartCounterScope.Daily);

            return global && daily;

            bool isWithinLimit(int? limitValue, LtoStartCounterScope scope)
            {
                if (limitValue == null)
                {
                    return true;
                }
                return counter[scope, keyCampaignName] < limitValue;
            }
        }

        internal static bool IsWithinAppVersionLimits(this CampaignModel campaign, string version)
        {
            if (campaign.AppVersionLimit?.VersionFrom == null && campaign.AppVersionLimit?.VersionTo == null)
            {
                return true;
            }

            if (!Version.TryParse(version, out var appVersion))
            {
                return false;
            }

            Version limitVersionFrom = null;
            if (campaign.AppVersionLimit?.VersionFrom != null && !Version.TryParse(campaign.AppVersionLimit?.VersionFrom!, out limitVersionFrom))
            {
                return false;
            }

            Version limitVersionTo = null;
            if (campaign.AppVersionLimit?.VersionTo != null && !Version.TryParse(campaign.AppVersionLimit?.VersionTo!, out limitVersionTo))
            {
                return false;
            }

            return (limitVersionFrom == null || limitVersionFrom <= appVersion) && (limitVersionTo == null || appVersion <= limitVersionTo);
        }

        internal static bool IsLaunchTimeIntervalPassed(this CampaignModel campaign, DateTime prevEndTime)
        {
            if (campaign.Delay == null)
            {
                return true;
            }

            return DateTime.UtcNow - prevEndTime > campaign.Delay;
        }

        internal static bool IsTimeWithinActivationPeriod(this CampaignModel campaign)
        {
            if (campaign.ActivationPeriod == null)
            {
                return true;
            }

            var currentTime = DateTime.UtcNow;
            return currentTime > campaign.ActivationPeriod.GetStartTime() && currentTime < campaign.ActivationPeriod.GetEndTime();
        }

        internal static bool IsWithinActivationsPerPeriodLimit(this CampaignModel campaign, DateTime prevStartTime, Counter<LtoStartCounterScope> counter)
        {
            if (campaign.ActivationPeriod?.ActivationLimit == null)
            {
                return true;
            }

            // Skip activations limit in case previous activation was outside period.
            if (prevStartTime == default || prevStartTime < campaign.ActivationPeriod.GetStartTime())
            {
                return true;
            }

            var keyCampaignName = CounterKey.GetKey(campaignName: campaign.Name);
            return counter[LtoStartCounterScope.Period, keyCampaignName] < campaign.ActivationPeriod?.ActivationLimit;
        }

        internal static bool IsWithinLaunchTimeLimits(this CampaignModel campaign)
        {
            var currentTime = DateTime.UtcNow;
            var isStartSuitable = campaign.StartsAt == null || currentTime > campaign.Timezone?.Get(campaign.StartsAt.Value);
            var isEndSuitable = campaign.EndsAt == null || currentTime < campaign.Timezone?.Get(campaign.EndsAt.Value);

            var forcedFinishTime = campaign.GetForcedFinishTime();
            var isForceFinishSuitable = forcedFinishTime == null || currentTime < forcedFinishTime;
            return isStartSuitable && isEndSuitable && isForceFinishSuitable;
        }

        internal static bool CanBeUpdatedWhileActive(this CampaignModel campaign)
        {
            return campaign.IsLtoCached is false;
        }

        internal static int? GetStopRewardAfterWatchLimit(this CampaignModel campaign)
        {
            return campaign.ShouldStopRewardOfferAfterWatch is true ? 1 : campaign.StopRewardOfferAfterWatchLimit;
        }

        internal static DateTime? GetForcedFinishTime(this CampaignModel campaign)
        {
            return campaign.FinishAt.HasValue ? campaign.Timezone?.Get(campaign.FinishAt.Value) : null;
        }
    }
}