﻿using System;
using System.Reflection;

namespace Magify
{
    public static class Base64Utils
    {
        [JetBrains.Annotations.Pure]
        [System.Diagnostics.Contracts.Pure]
        public static int CalculateAndValidateOutputLength(int inputLength, bool insertLineBreaks)
        {
            try
            {
                // It is better to use the internal method from the Convert class if we can. But it also duplicated below for the case of failure.
                return (int)typeof(Convert)
                    .GetMethod("ToBase64_CalculateAndValidateOutputLength", BindingFlags.NonPublic | BindingFlags.Static | BindingFlags.NonPublic)
                    ?.Invoke(null, new object[] { inputLength, false })!; // (int inputLength, bool insertLineBreaks)
            }
            catch
            {
                // ignored
            }

            // look: https://referencesource.microsoft.com/#mscorlib/system/convert.cs,54651c7adb7d244d
            const int base64LineBreakPosition = 76;
            var outlen = (long)inputLength / 3 * 4; // the base length - we want integer division here.
            outlen += (inputLength % 3) != 0 ? 4 : 0; // at most 4 more chars for the remainder
            if (outlen == 0)
            {
                return 0;
            }
            if (insertLineBreaks)
            {
                var newLines = outlen / base64LineBreakPosition;
                if (outlen % base64LineBreakPosition == 0)
                {
                    --newLines;
                }
                outlen += newLines * 2; // the number of line break chars we'll add, "\r\n"
            }

            return outlen switch
            {
                // If we overflow an int then we cannot allocate enough memory to output the value so throw
                > int.MaxValue => throw new OutOfMemoryException(),
                _ => (int)outlen,
            };
        }
    }
}