using System;
using System.Collections.Generic;
using JetBrains.Annotations;
using Magify.Rx;
using Magify.Types;
using UnityEngine;

namespace Magify
{
    internal class GeneralPrefs : ILegacyMigrator, IInitializable, ICancelable
    {
        private const string KeyMigratedUser = "migrated_user";

        internal const string KeyClientId = "client_id";
        private const string KeyCustomClientIdWasSet = "custom_client_id_was_set";
        private const string KeyIsClientIdRetrievedFromNativeStorage = "client_id_retrieved_from_native_storage";
        private const string KeyIsMigratedFromOldStorageScheme = "is_migrated_from_old_storage_scheme";
        private const string KeyFirstInstalledVersion = "first_version";
        private const string KeyLastInitializedVersion = "last_initialized_version";
        private const string KeyFirstLaunchDate = "first_launch";

        private const string KeySessionCounter = "session_counter";
        private const string KeyVersionSessionCounter = "version_session_counter";
        private const string KeySessionId = "session_id";
        private const string KeySessionOpenTime = "session_open_time";
        private const string KeySessionLastTime = "session_last_time";

        private const string KeyAuthToken = "auth_token";
        private const string KeyAuthTokenExpired = "auth_token_expired";

        private const string KeyLanguageCode = "language_code";
        private const string KeyCountryCode = "country_code";

        private const string KeySubscriptionState = "subscription_state";
        private const string KeyInAppStatus = "in_app_status";
        private const string KeyAuthorizationStatus = "authorization_status";
        private const string KeyReferrerId = "referrer_id";
        private const string KeyAdjustId = "adjust_id";
        private const string KeyGpsAdId = "gps_ad_id";
        private const string KeyGdprStatus = "gdpr_granted";
        private const string KeyAttAuthorizationStatus = "att_authorization_status";
        private const string KeySyncStateEnabled = "sync_state_enabled";

        private const string KeySourceNetwork = "source_network";
        private const string KeySourceCampaign = "source_campaign";
        private const string KeySourceGroup = "source_group";

        private const string KeyDefaultCampaignTypes = "default_campaign_types";
        private const string KeyProductsReceiptTime = "products_by_dates";

        private const string KeyPurchasedInApps = "purch_products";
        private const string KeyPurchasedSubscriptions = "purch_subs_products";
        private const string KeyUsedFreeProducts = "free_products";

        private const string KeyFirebaseInstanceID = "firebase_instance_id";

        private const string KeyNextDailyResetTime = "next_daily_reset_time";

        private const string KeyCampaignsContextScopes = "CampaignsContextScopes";

        private const string KeyEnvironment = "environment";

        internal const SubscriptionStatus DefaultSubscriptionStatus = Magify.SubscriptionStatus.Inactive;
        internal const InAppStatus DefaultInAppStatus = Magify.InAppStatus.NeverPurchased;
        internal const AuthorizationStatus DefaultAuthorizationStatus = Magify.AuthorizationStatus.Unknown;

        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get();

        [NotNull]
        private readonly object _lockStorage = new();
        [NotNull]
        private readonly object _lockLocalStorage = new();
        [NotNull]
        private readonly object _lockMiscs = new();
        [NotNull]
        private readonly BinaryStorage _storage;
        [NotNull]
        private readonly BinaryStorage _localSubStorage;
        [NotNull]
        private readonly IDictionary<string, int> _versionSessionCounter;
        [NotNull]
        private readonly AppVersionProvider _appVersionProvider;
        public bool IsDisposed { get; private set; }

        [NotNull]
        public NotNullReactiveProperty<string> ClientId { get; }
        [NotNull]
        public IReactiveProperty<bool> CustomClientIdWasSet { get; }
        [NotNull]
        public IReactiveProperty<bool> IsFirstAppLaunchTracked { get; } = new ReactiveProperty<bool>(false);
        [NotNull]
        public IReactiveProperty<bool> AuthorizationTokenExpired { get; }
        [NotNull]
        public IReactiveProperty<string> AuthorizationToken { get; }
        [NotNull]
        public IReactiveProperty<string> LanguageCode { get; }
        [NotNull]
        public IReactiveProperty<string> CountryCode { get; }
        [NotNull]
        public IReactiveProperty<SubscriptionStatus> SubscriptionStatus { get; }
        [NotNull]
        public IReactiveProperty<InAppStatus> InAppStatus { get; }
        [NotNull]
        public IReactiveProperty<AuthorizationStatus> AuthorizationStatus { get; }
        [NotNull]
        public IReactiveProperty<string> ReferrerId { get; }
        [NotNull]
        public IReactiveProperty<string> AdjustId { get; }
        [NotNull]
        public IReactiveProperty<string> GpsAdId { get; }
        [NotNull]
        public IReactiveProperty<bool> IsGdprGranted { get; }
        [NotNull]
        public IReactiveProperty<bool?> AttAuthorizationStatus { get; }
        [NotNull]
        public ISet<string> UsedDefaultCampaignTypes { get; }
        [NotNull]
        public IDictionary<string, DateTime> ProductsReceiptTimestamps { get; }
        [NotNull]
        public IReactiveProperty<bool> SyncStateEnabled { get; }

        [NotNull]
        public ISet<string> PurchasedInAppProducts { get; }
        [NotNull]
        public ISet<string> PurchasedSubscriptionProducts { get; }
        [NotNull]
        public ISet<string> UsedFreeProducts { get; }
        [NotNull]
        public ICollection<string> AllUsedProducts { get; }

        [NotNull]
        public IReactiveProperty<string> FirebaseInstanceId { get; }

        [NotNull]
        public IReactiveProperty<int> GlobalSessionCounter { get; }

        [NotNull]
        public IReactiveProperty<string> SessionId { get; }

        [NotNull]
        public IReactiveProperty<ConfigScope> RemoteContextScopes { get; }

        [NotNull]
        public IReactiveProperty<Environment> Environment { get; }

        public bool MigratedUser
        {
            get
            {
                lock (_lockStorage)
                {
                    return _storage.Get<bool>(KeyMigratedUser);
                }
            }
            private set
            {
                lock (_lockStorage)
                {
                    _storage.Set(KeyMigratedUser, value);
                }
            }
        }

        public bool IsNewUser
        {
            get
            {
                lock (_lockMiscs)
                {
                    return FirstInstalledVersion == _appVersionProvider.AppVersion;
                }
            }
        }

        public bool IsMigratedFromOldStorageScheme
        {
            get
            {
                lock (_lockLocalStorage)
                {
                    return _localSubStorage.Get<bool>(KeyIsMigratedFromOldStorageScheme);
                }
            }
            set
            {
                lock (_lockLocalStorage)
                {
                    _localSubStorage.Set(KeyIsMigratedFromOldStorageScheme, value);
                }
            }
        }

        public bool IsClientIdRetrievedFromNativeStorage
        {
            get
            {
                lock (_lockStorage)
                {
                    return _storage.Get<bool>(KeyIsClientIdRetrievedFromNativeStorage);
                }
            }
            set
            {
                lock (_lockStorage)
                {
                    _storage.Set(KeyIsClientIdRetrievedFromNativeStorage, value);
                }
            }
        }

        public string FirstInstalledVersion
        {
            get
            {
                lock (_lockStorage)
                {
                    return _storage.Get<string>(KeyFirstInstalledVersion);
                }
            }
            set
            {
                lock (_lockStorage)
                {
                    _storage.Set(KeyFirstInstalledVersion, value);
                }
            }
        }

        public string LastInitializedVersion
        {
            get
            {
                lock (_lockLocalStorage)
                {
                    return _localSubStorage.Get<string>(KeyLastInitializedVersion);
                }
            }
            set
            {
                lock (_lockLocalStorage)
                {
                    _localSubStorage.Set(KeyLastInitializedVersion, value);
                }
            }
        }

        public DateTime FirstLaunchDate
        {
            get
            {
                lock (_lockStorage)
                {
                    return _storage.Get(KeyFirstLaunchDate, default(DateTime));
                }
            }
            set
            {
                lock (_lockStorage)
                {
                    _storage.Set(KeyFirstLaunchDate, value);
                }
            }
        }

        public int VersionSessionCounter
        {
            get
            {
                lock (_lockMiscs)
                {
                    return _versionSessionCounter.TryGetValue(_appVersionProvider.AppVersion, out var value) ? value : 0;
                }
            }
            set
            {
                lock (_lockMiscs)
                {
                    _versionSessionCounter[_appVersionProvider.AppVersion] = value;
                }
            }
        }

        public long SessionStarted
        {
            get
            {
                lock (_lockStorage)
                {
                    return _storage.Get(KeySessionOpenTime, 0L);
                }
            }
            set
            {
                lock (_lockStorage)
                {
                    _storage.Set(KeySessionOpenTime, value);
                }
            }
        }

        public long LastActivity
        {
            get
            {
                lock (_lockStorage)
                {
                    return _storage.Get(KeySessionLastTime, 0L);
                }
            }
            set
            {
                lock (_lockStorage)
                {
                    _storage.Set(KeySessionLastTime, value);
                }
            }
        }

        public MediaSource MediaSource
        {
            get
            {
                lock (_lockLocalStorage)
                {
                    var network = _localSubStorage.Get<string>(KeySourceNetwork);
                    var campaign = _localSubStorage.Get<string>(KeySourceCampaign);
                    var group = _localSubStorage.Get<string>(KeySourceGroup);
                    return new MediaSource(network, campaign, group);
                }
            }
            set
            {
                lock (_lockLocalStorage)
                {
                    using var _ = _localSubStorage.MultipleChangeScope();
                    _localSubStorage.Set(KeySourceNetwork, value.NetworkName);
                    _localSubStorage.Set(KeySourceCampaign, value.CampaignName);
                    _localSubStorage.Set(KeySourceGroup, value.GroupName);
                }
            }
        }

        public DateTime NextDailyResetTime
        {
            get
            {
                lock (_lockStorage)
                {
                    return _storage.Get(KeyNextDailyResetTime, default(DateTime));
                }
            }
            set
            {
                lock (_lockStorage)
                {
                    _storage.Set(KeyNextDailyResetTime, value);
                }
            }
        }

        [NotNull]
        public static GeneralPrefs Create(
            [NotNull] string storagePath,
            [NotNull] string localSubStoragePath,
            [NotNull] AppVersionProvider appVersionProvider)
        {
            var storage = configure(BinaryStorage.Construct(storagePath)).Build();
            var localSubStorage = configure(BinaryStorage.Construct(localSubStoragePath)).Build();
            var prefs = new GeneralPrefs(storage, localSubStorage, appVersionProvider);
            if (!prefs.IsMigratedFromOldStorageScheme)
            {
                _logger.Log("Migration from old general prefs storage scheme...");
                prefs.TryMigrateFromOldStorageScheme(storage);
                prefs.IsMigratedFromOldStorageScheme = true;
            }
            return prefs;

            [NotNull]
            BinaryStorage.Builder configure([NotNull] BinaryStorage.Builder storageBuilder)
            {
                return storageBuilder
                    .AddPrimitiveTypes()
                    .SupportNullable<bool>()
                    .SupportSetsOf<string>()
                    .SupportDictionariesOf<string, long>()
                    .SupportDictionariesOf<string, int>()
                    .SupportDictionariesOf<string, DateTime>()
                    .SupportEnum<SubscriptionStatus>()
                    .SupportEnum<InAppStatus>()
                    .SupportEnum<AuthorizationStatus>()
                    .SupportEnum<ConfigScope>()
                    .SupportEnum<Environment>();
            }
        }

        void ILegacyMigrator.Migrate([NotNull] MigrationData migrationData)
        {
            lock (_lockStorage)
            {
                using (MultipleChangeScopeThreadUnsafe())
                {
                    MigratedUser = true;
                    if (migrationData.ClientId != null)
                        ClientId.Value = migrationData.ClientId;
                    FirstInstalledVersion = migrationData.FirstInstalledVersion;
                    FirstLaunchDate = migrationData.FirstLaunchDate ?? default;
                    if (migrationData.SubscriptionStatus.HasValue)
                        SubscriptionStatus.Value = migrationData.SubscriptionStatus.Value;
                    InAppStatus.Value = migrationData.InAppStatus ?? DefaultInAppStatus;
                    if (migrationData.AuthorizationStatus.HasValue)
                        AuthorizationStatus.Value = migrationData.AuthorizationStatus.Value;
                    ReferrerId.Value = migrationData.ReferrerId;
                    AdjustId.Value = migrationData.AdjustId;
                    GpsAdId.Value = migrationData.GpsAdId;
                    IsGdprGranted.Value = migrationData.IsGdprGranted ?? default;
                    AttAuthorizationStatus.Value = migrationData.AttAuthorizationStatus ?? default;
                    UsedDefaultCampaignTypes.AddRange(migrationData.UsedDefaultCampaignTypes);
                    ProductsReceiptTimestamps.AddRange(migrationData.ProductsReceiptTimestamps);
                    PurchasedInAppProducts.AddRange(migrationData.PurchasedInAppProducts);
                    PurchasedSubscriptionProducts.AddRange(migrationData.PurchasedSubscriptionProducts);
                    UsedFreeProducts.AddRange(migrationData.UsedFreeProducts);

                    FirebaseInstanceId.Value = migrationData.FirebaseInstanceId;
                    GlobalSessionCounter.Value = migrationData.GlobalSessionCounter ?? default;
                    SessionId.Value = migrationData.SessionId;
                    SessionStarted = migrationData.SessionStarted ?? default;
                    LastActivity = migrationData.LastActivity ?? default;
                    MediaSource = migrationData.MediaSource ?? default;
                }
            }
        }

        private void TryMigrateFromOldStorageScheme([NotNull] BinaryStorage oldStorage)
        {
            // Migrate from syncable to local (added during sync progress implementation)
            using (MultipleChangeScope())
            {
                tryMigrateToLocal(KeyClientId, ClientId);
                tryMigrateToLocal(KeyAuthTokenExpired, AuthorizationTokenExpired);
                tryMigrateToLocal(KeyAuthToken, AuthorizationToken);
                tryMigrateToLocal(KeyInAppStatus, InAppStatus);
                tryMigrateToLocal(KeySubscriptionState, SubscriptionStatus);
                tryMigrateToLocal(KeyAuthorizationStatus, AuthorizationStatus);
                tryMigrateToLocal(KeyReferrerId, ReferrerId);
                tryMigrateToLocal(KeyLanguageCode, LanguageCode);
                tryMigrateToLocal(KeyCountryCode, CountryCode);
                tryMigrateToLocal(KeyGpsAdId, GpsAdId);
                tryMigrateToLocal(KeyGdprStatus, IsGdprGranted);
                tryMigrateToLocal(KeyAttAuthorizationStatus, AttAuthorizationStatus);
                tryMigrateToLocal(KeyAdjustId, AdjustId);
                tryMigrateToLocal(KeyFirebaseInstanceID, FirebaseInstanceId);
            }

            void tryMigrateToLocal<T>([NotNull] string key, [NotNull] IReactiveProperty<T> property)
            {
                if (oldStorage.Has<T>(key))
                {
                    property.Value = oldStorage.Get<T>(key);
                    oldStorage.Remove<T>(key);
                }
            }
        }

        void IInitializable.Initialize()
        {
            if (string.IsNullOrEmpty(FirstInstalledVersion))
            {
                FirstInstalledVersion = _appVersionProvider.AppVersion;
            }
            if (FirstLaunchDate == default)
            {
                FirstLaunchDate = DateTime.Now;
            }
        }

        void IDisposable.Dispose()
        {
            lock (_lockStorage)
            {
                lock (_lockLocalStorage)
                {
                    if (IsDisposed)
                    {
                        return;
                    }
                    IsDisposed = true;
                    _storage.Dispose();
                    _localSubStorage.Dispose();
                }
            }
        }

        private GeneralPrefs([NotNull] BinaryStorage storage, [NotNull] BinaryStorage localSubStorage, [NotNull] AppVersionProvider appVersionProvider)
        {
            _storage = storage;
            _localSubStorage = localSubStorage;
            _appVersionProvider = appVersionProvider;
            using (MultipleChangeScopeThreadUnsafe())
            {
                // General data, that must be synced with server
                GlobalSessionCounter = _storage.GetReactiveProperty(KeySessionCounter, 0);
                SessionId = _storage.GetReactiveProperty(KeySessionId, string.Empty);

                UsedDefaultCampaignTypes = _storage.GetSet<string>(KeyDefaultCampaignTypes);
                ProductsReceiptTimestamps = _storage.GetDictionary<string, DateTime>(KeyProductsReceiptTime);

                PurchasedInAppProducts = _storage.GetSet<string>(KeyPurchasedInApps);
                PurchasedSubscriptionProducts = _storage.GetSet<string>(KeyPurchasedSubscriptions);
                UsedFreeProducts = _storage.GetSet<string>(KeyUsedFreeProducts);
                AllUsedProducts = new UnionReadOnlyCollection<string>(PurchasedInAppProducts, PurchasedSubscriptionProducts, UsedFreeProducts);

                FirebaseInstanceId = _storage.GetReactiveProperty<string>(KeyFirebaseInstanceID);

                _versionSessionCounter = _storage.GetDictionary<string, int>(KeyVersionSessionCounter);
                _versionSessionCounter.TryAdd(_appVersionProvider.AppVersion, 0);

                // Local data, that won't be synced with server
                ClientId = _localSubStorage.GetReactiveProperty<string>(KeyClientId).AsNotNull(null);
                CustomClientIdWasSet = _localSubStorage.GetReactiveProperty<bool>(KeyCustomClientIdWasSet); // here while sync progress is not supported for custom id's
                Environment = _localSubStorage.GetReactiveProperty<Environment>(KeyEnvironment);
                AuthorizationTokenExpired = _localSubStorage.GetReactiveProperty(KeyAuthTokenExpired, false);
                AuthorizationToken = _localSubStorage.GetReactiveProperty(KeyAuthToken, string.Empty);
                InAppStatus = _localSubStorage.GetReactiveProperty(KeyInAppStatus, DefaultInAppStatus);
                SubscriptionStatus = _localSubStorage.GetReactiveProperty(KeySubscriptionState, DefaultSubscriptionStatus);
                AuthorizationStatus = _localSubStorage.GetReactiveProperty(KeyAuthorizationStatus, DefaultAuthorizationStatus);
                ReferrerId = _localSubStorage.GetReactiveProperty(KeyReferrerId, string.Empty);
                LanguageCode = _localSubStorage.GetReactiveProperty(KeyLanguageCode, string.Empty);
                CountryCode = _localSubStorage.GetReactiveProperty(KeyCountryCode, string.Empty);
                GpsAdId = _localSubStorage.GetReactiveProperty<string>(KeyGpsAdId);
                IsGdprGranted = _localSubStorage.GetReactiveProperty<bool>(KeyGdprStatus);
                AttAuthorizationStatus = _localSubStorage.GetReactiveProperty<bool?>(KeyAttAuthorizationStatus);
                AdjustId = _localSubStorage.GetReactiveProperty(KeyAdjustId, default(string));
                FirebaseInstanceId = _localSubStorage.GetReactiveProperty<string>(KeyFirebaseInstanceID);
                SyncStateEnabled = _localSubStorage.GetReactiveProperty(KeySyncStateEnabled, false);
                RemoteContextScopes = _localSubStorage.GetReactiveProperty<ConfigScope>(KeyCampaignsContextScopes);
            }
        }

        /// <summary>
        /// If you are planning to change many properties of this class, you should use this method.
        /// This method temporarily blocks the writing prefs file to disk inside <b>using</b> construction.
        /// </summary>
        /// <example>
        /// <code>
        /// using (MultipleChangeScope())
        /// {
        ///     GeneralPrefs.ClientId = Guid.NewGuid().ToString();
        ///     GeneralPrefs.AuthorizationStatus.Value = Magify.AuthorizationStatus.Authorized;
        /// }
        /// </code>
        /// </example>
        /// <returns>IDisposable which must disposed automatically after <b>using</b> construction (preferred) or disposed manually.</returns>
        public IDisposable MultipleChangeScope()
        {
            lock (_lockStorage)
            {
                lock (_lockLocalStorage)
                {
                    return MultipleChangeScopeThreadUnsafe();
                }
            }
        }

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        private IDisposable MultipleChangeScopeThreadUnsafe()
        {
            return new DisposableTuple<IDisposable>((_storage.MultipleChangeScope(), _localSubStorage.MultipleChangeScope()));
        }

        public void Reset()
        {
            lock (_lockStorage)
            {
                _storage.ResetAll();
            }
            lock (_lockLocalStorage)
            {
                _localSubStorage.ResetAll();
            }
        }

        /// <inheritdoc cref="BinaryStorage.GetFileContent"/>
        [NotNull]
        internal string GetFileContent()
        {
            lock (_lockStorage)
            {
                return _storage.GetFileContent();
            }
        }

        /// <inheritdoc cref="BinaryStorage.RewriteContent"/>
        internal void RewriteContent([NotNull] string contentBase64)
        {
            lock (_lockStorage)
            {
                _storage.RewriteContent(contentBase64);
            }
        }
    }
}