using System;
using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using Magify.Model;
using UnityEngine;

namespace Magify.Features
{
    internal class FeaturesProvider : IFeatures, IContextListener, IDisposable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Features);

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        public event Action<ConfigKind> OnUpdate;

        public Dictionary<string, object> AllFeatures { get; private set; }
        public Dictionary<string, object> AllDefaultFeatures { get; private set; }

        [NotNull]
        private SortedDictionary<string, bool> _bufferBooleans = new();
        [NotNull]
        private SortedDictionary<string, double> _bufferNumbers = new();
        [NotNull]
        private SortedDictionary<string, string> _bufferStrings = new();

        [NotNull]
        private SortedDictionary<string, bool> _defaultBooleans = new();
        [NotNull]
        private SortedDictionary<string, double> _defaultNumbers = new();
        [NotNull]
        private SortedDictionary<string, string> _defaultStrings = new();
        [NotNull]
        private SortedDictionary<string, bool> _currentBooleans = new();
        [NotNull]
        private SortedDictionary<string, double> _currentNumbers = new();
        [NotNull]
        private SortedDictionary<string, string> _currentStrings = new();

        [NotNull]
        private readonly Dictionary<FeatureSource, HashSet<string>> _ignoredFeatures;

        [CanBeNull]
        private Dictionary<string, FeatureConfig> _featureConfig;
        [CanBeNull]
        private Dictionary<string, FeatureConfig> _defaultFeatureConfig;

        [NotNull]
        private readonly IAnalyticsTracker _analyticsTracker;
        [NotNull]
        private readonly FeaturesBuilder _featuresBuilder;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        [NotNull]
        private readonly object _lock = new();

        public ConfigScope SuitableScope => ConfigScope.AppFeatures;

        #region IFeatures implementaion

        public bool IsCurrentFeaturesParsed { get; private set; }

        public bool TryGetBool([NotNull] string featureName, out Feature<bool> result)
        {
            lock (_lock)
            {
                return TryGetFeature(featureName, _defaultBooleans, _currentBooleans, _ignoredFeatures, _analyticsTracker, out result);
            }
        }

        public Feature<bool> GetBool([NotNull] string featureName)
        {
            if (!TryGetBool(featureName, out var result))
            {
                throw new MagifyFeatureNotFoundException(typeof(Boolean), featureName);
            }
            return result;
        }

        public bool TryGetNumber([NotNull] string featureName, out Feature<double> result)
        {
            lock (_lock)
            {
                return TryGetFeature(featureName, _defaultNumbers, _currentNumbers, _ignoredFeatures, _analyticsTracker, out result);
            }
        }

        public Feature<double> GetNumber([NotNull] string featureName)
        {
            if (!TryGetNumber(featureName, out var result))
            {
                throw new MagifyFeatureNotFoundException(typeof(Double), featureName);
            }
            return result;
        }

        public bool TryGetString([NotNull] string featureName, out Feature<string> result)
        {
            lock (_lock)
            {
                return TryGetFeature(featureName, _defaultStrings, _currentStrings, _ignoredFeatures, _analyticsTracker, out result);
            }
        }

        public Feature<string> GetString([NotNull] string featureName)
        {
            if (!TryGetString(featureName, out var result))
            {
                throw new MagifyFeatureNotFoundException(typeof(String), featureName);
            }
            return result;
        }

        private static bool TryGetFeature<T>(
            [NotNull] string featureName,
            [NotNull] IReadOnlyDictionary<string, T> defaultMap,
            [NotNull] IReadOnlyDictionary<string, T> currentMap,
            [NotNull] Dictionary<FeatureSource, HashSet<string>> ignoredFeatures,
            [NotNull] IAnalyticsTracker analyticsTracker,
            out Feature<T> result)
        {
            if (!CheckFeatureIsIgnored(ignoredFeatures, FeatureSource.Current, featureName))
            {
                if (currentMap.TryGetValue(featureName, out var value))
                {
                    result = new Feature<T>(value, isDefault: false);
                    return true;
                }
            }
            if (!CheckFeatureIsIgnored(ignoredFeatures, FeatureSource.Default, featureName))
            {
                if (defaultMap.TryGetValue(featureName, out var value))
                {
                    analyticsTracker.TrackUsedApplicationDefaultFeaturesEvent(featureName);
                    result = new Feature<T>(value, isDefault: true);
                    return true;
                }
            }

            result = default;
            return false;
        }

        #endregion

        public FeaturesProvider([NotNull] GeneralPrefs generalPrefs, [NotNull] IAnalyticsTracker analyticsTracker)
        {
            _featuresBuilder = new FeaturesBuilder(generalPrefs);
            _analyticsTracker = analyticsTracker;
            _ignoredFeatures = new Dictionary<FeatureSource, HashSet<string>>();
        }

        void IDisposable.Dispose()
        {
            lock (_lock)
            {
                _disposables.Release();
            }
        }

        public void UpdateContext([NotNull] CampaignsContext context, ContextKind kind)
        {
            bool isUpdated;
            lock (_lock)
            {
                if (kind is ContextKind.Default)
                {
                    AllDefaultFeatures = _featuresBuilder.BuildFeatureValueMap(context.Features).ToDictionary(c => c.Key, c => c.Value);
                }
                else
                {
                    AllFeatures = _featuresBuilder.BuildFeatureValueMap(context.Features).ToDictionary(c => c.Key, c => c.Value);
                }
                isUpdated = UpdateFeaturesThreadUnsafe(context.Features, kind is ContextKind.Default);
            }
            if (isUpdated)
            {
                OnUpdate?.Invoke(kind.ToConfigKind());
            }
        }

        public void ResetFeatures()
        {
            bool isUpdated;
            lock (_lock)
            {
                isUpdated = UpdateFeaturesThreadUnsafe(default, true);
            }
            if (isUpdated)
            {
                OnUpdate?.Invoke(ConfigKind.Default);
            }
        }

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        /// <returns>
        /// True if updated
        /// </returns>
        private bool UpdateFeaturesThreadUnsafe(Dictionary<string, FeatureConfig> features, bool isDefault)
        {
            bool isUpdated;
            if (isDefault)
            {
                _defaultFeatureConfig = features;
                isUpdated = UpdateDefaultFeatureValuesThreadUnsafe(_defaultFeatureConfig);
            }
            else
            {
                _featureConfig = features;
                isUpdated = UpdateFeatureValuesThreadUnsafe(_featureConfig);
                IsCurrentFeaturesParsed = _featureConfig != null;
            }
            return isUpdated;
        }

        internal void UpdateFeaturesOnTrigger()
        {
            var isFeaturesUpdated = false;
            var isDefaultFeaturesUpdated = false;
            lock (_lock)
            {
                isFeaturesUpdated = UpdateFeatureValuesThreadUnsafe(_featureConfig);
                isDefaultFeaturesUpdated = UpdateDefaultFeatureValuesThreadUnsafe(_defaultFeatureConfig);
            }
            if (isFeaturesUpdated)
            {
                OnUpdate?.Invoke(ConfigKind.Saved);
            }
            if (isDefaultFeaturesUpdated)
            {
                OnUpdate?.Invoke(ConfigKind.Default);
            }
        }

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        private bool UpdateFeatureValuesThreadUnsafe([CanBeNull] Dictionary<string, FeatureConfig> config)
        {
            return UpdateFeaturesThreadUnsafe(config, ref _currentBooleans, ref _currentNumbers, ref _currentStrings);
        }

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        private bool UpdateDefaultFeatureValuesThreadUnsafe([CanBeNull] Dictionary<string, FeatureConfig> config)
        {
            return UpdateFeaturesThreadUnsafe(config, ref _defaultBooleans, ref _defaultNumbers, ref _defaultStrings);
        }

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        private bool UpdateFeaturesThreadUnsafe(
            [CanBeNull] Dictionary<string, FeatureConfig> config,
            ref SortedDictionary<string, bool> booleans,
            ref SortedDictionary<string, double> numbers,
            ref SortedDictionary<string, string> strings)
        {
            _bufferBooleans.Clear();
            _bufferNumbers.Clear();
            _bufferStrings.Clear();

            if (config != null)
            {
                foreach (var (key, rawValue) in _featuresBuilder.BuildFeatureValueMap(config))
                {
                    ParseFeature(key, rawValue, _bufferBooleans, _bufferNumbers, _bufferStrings);
                }
            }

            if (!booleans.SequenceEqual(_bufferBooleans) || !numbers.SequenceEqual(_bufferNumbers) || !strings.SequenceEqual(_bufferStrings))
            {
                (booleans, _bufferBooleans) = (_bufferBooleans, booleans);
                (numbers, _bufferNumbers) = (_bufferNumbers, numbers);
                (strings, _bufferStrings) = (_bufferStrings, strings);
                return true;
            }

            return false;
        }

        public void SetIgnoredFeatures([CanBeNull] IEnumerable<string> featuresToIgnore, FeatureSource source)
        {
            var features = featuresToIgnore?.ToHashSet();

            if (features == null || features.Count == 0)
            {
                foreach (var sourceItem in EnumerateFeatureSourceFlags(source))
                {
                    if (_ignoredFeatures.Remove(sourceItem))
                    {
                        _logger.Log($"Clear features from ignore list with {nameof(FeatureSource)} = {source}");
                    }
                }

                return;
            }

            foreach (var sourceItem in EnumerateFeatureSourceFlags(source))
            {
                _ignoredFeatures[sourceItem] = features;
                _logger.Log($"Features was added to ignore list with {nameof(FeatureSource)}");
            }
        }

        [NotNull]
        private static IEnumerable<FeatureSource> EnumerateFeatureSourceFlags(FeatureSource source)
        {
            foreach (FeatureSource sourceItem in Enum.GetValues(typeof(FeatureSource)))
            {
                if (sourceItem == FeatureSource.None)
                {
                    continue;
                }
                if (source.HasFlag(sourceItem))
                {
                    yield return sourceItem;
                }
            }
        }

        private static bool CheckFeatureIsIgnored(
            [NotNull] Dictionary<FeatureSource, HashSet<string>> ignoredFeatures,
            FeatureSource source,
            [NotNull] string featureName)
        {
            if (ignoredFeatures.TryGetValue(source, out var features))
            {
                return features!.Contains(featureName);
            }

            return false;
        }

        private static void ParseFeature(
            [NotNull] string key,
            [CanBeNull] object rawValue,
            [NotNull] in IDictionary<string, bool> booleans,
            [NotNull] in IDictionary<string, double> numbers,
            [NotNull] in IDictionary<string, string> strings)
        {
            _logger.Log($"Key={key}. Type={(rawValue != null ? rawValue.GetType().Name : "N/A")}. Value={rawValue}");
            switch (rawValue)
            {
                case bool value:
                    booleans[key] = value;
                    break;
                case short value:
                    numbers[key] = value;
                    break;
                case int value:
                    numbers[key] = value;
                    break;
                case long value:
                    numbers[key] = value;
                    break;
                case float value:
                    numbers[key] = value;
                    break;
                case double value:
                    numbers[key] = value;
                    break;
                case string value:
                    strings[key] = value;
                    break;
                default:
                    strings[key] = JsonFacade.SerializeObject(rawValue);
                    _logger.Log("Convert to json");
                    _logger.Log(strings[key]);
                    break;
            }
        }
    }
}