﻿using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using Magify.Features;
using Magify.Model;
using UnityEngine;

namespace Magify
{
    internal class FeaturesBuilder
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Features);
        [NotNull]
        private readonly GeneralPrefs _generalPrefs;

        public FeaturesBuilder([NotNull] GeneralPrefs generalPrefs)
        {
            _generalPrefs = generalPrefs;
        }

        [NotNull]
        public IEnumerable<(string Key, object Value)> BuildFeatureValueMap([CanBeNull] IReadOnlyDictionary<string, FeatureConfig> config)
        {
            var configs = config ?? Enumerable.Empty<KeyValuePair<string, FeatureConfig>>();
            foreach (var (key, featureConfig) in configs)
            {
                if (key == null)
                {
                    _logger.LogError("Failed to parse feature, null was found as feature name");
                    continue;
                }
                if (featureConfig == null)
                {
                    _logger.LogError($"Failed to parse feature {key}, null was found as feature value");
                    continue;
                }
                var option = featureConfig.Options?.FirstOrDefault(IsSuitable);
                var rawValue = option?.Value ?? featureConfig.DefaultValue;
                if (rawValue is not null)
                {
                    yield return (key, rawValue);
                }
            }
        }
        private bool IsSuitable([CanBeNull] FeatureOption option)
        {
            if (option?.Trigger is null)
            {
                return false;
            }

            if (option.Trigger.SubscriptionStatus?.IsCorresponded(_generalPrefs.SubscriptionStatus.Value) == false)
            {
                return false;
            }
            if (option.Trigger.InAppStatus?.IsCorresponded(_generalPrefs.InAppStatus.Value) == false)
            {
                return false;
            }
            if (option.Trigger.AuthorizationStatus?.IsCorresponded(_generalPrefs.AuthorizationStatus.Value) == false)
            {
                return false;
            }
            if (option.Trigger.PayingStatus?.IsCorresponded(_generalPrefs.SubscriptionStatus.Value, _generalPrefs.InAppStatus.Value) == false)
            {
                return false;
            }
            if (!option.Trigger.GetReferrerLimits().IsCorresponded(_generalPrefs.ReferrerId.Value))
            {
                return false;
            }
            if (!option.Trigger.GetPurchaseLimits().IsCorresponded(_generalPrefs.AllUsedProducts))
            {
                return false;
            }

            return true;
        }
    }
}