﻿using System;
using System.Collections.Generic;
using JetBrains.Annotations;
using Magify.Model;
using UnityEngine;

namespace Magify
{
    internal class SdkByPlatformStateFileStorage : FileStorage
    {
        private const EncodingMode EncodingMode = Magify.EncodingMode.ToJsonToZipToBase64;
        private const string FileName = "bps.dat";

        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Storage);
        [NotNull]
        private readonly IMagifyEncoder _encoder = new ToJsonToZipToBase64MagifyEncoder();
        [NotNull]
        private readonly object _lock = new();

        public SdkByPlatformStateFileStorage(string storagePath) : base(storagePath)
        {
        }

        public void SaveState([CanBeNull] Dictionary<RuntimePlatform, ApplicationState.PlatformSdk> byPlatform)
        {
            var text = byPlatform == null || byPlatform.Count == 0
                ? string.Empty
                : _encoder.Encode(byPlatform, EncodingMode.ToPrefixString());
            lock (_lock)
            {
                Save(FileName, text);
            }
        }

        [NotNull]
        public Dictionary<RuntimePlatform, ApplicationState.PlatformSdk> LoadState()
        {
            try
            {
                string text;
                lock (_lock)
                {
                    text = Load(FileName);
                }
                if (!string.IsNullOrEmpty(text))
                {
                    var decoded = _encoder.Decode<Dictionary<RuntimePlatform, ApplicationState.PlatformSdk>>(text, EncodingMode.ToPrefixString().Length);
                    if (decoded != null)
                        return decoded;
                }
            }
            catch (Exception e)
            {
                _logger.LogException(e);
            }
            return new Dictionary<RuntimePlatform, ApplicationState.PlatformSdk>(1);
        }
    }
}