using System;
using System.Collections.Generic;
using System.Linq;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using UnityEngine;
using UnityEngine.Pool;
using EventType = Magify.Model.EventType;

namespace Magify
{
    internal interface IAnalyticsEventHandler
    {
        void Initialize();
        void OnForeground();
        void OnBackground();
        void ForceSync();
        void SetupAnalyticsConfig(AnalyticsConfiguration config);
        void ResetAnalyticsConfig();
    }

    internal interface IAnalyticsEventHandler<T> : IAnalyticsEventHandler
        where T : class, IAnalyticsEvent
    {
        [NotNull]
        AnalyticsEventStorage<T> Storage { get; }
        void HandleEvent(T analyticsEvent);
        void HandleEvents(IEnumerable<T> analyticsEvents);
    }

    internal class EventHandler<T> : IAnalyticsEventHandler<T>, IDisposable
        where T : class, IAnalyticsEvent
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Analytics);
        [NotNull]
        private readonly IServerApi _serverApi;
        [NotNull]
        private readonly AppStatePrefs _appStatePrefs;
        private readonly EventType _type;
        [NotNull]
        private readonly SyncTimer _syncTimer;
        [NotNull]
        private readonly AnalyticsConfiguration _defaultConfig;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        [NotNull]
        private readonly object _lock = new();
        [NotNull]
        private AnalyticsConfiguration _currentConfig;
        [NotNull]
        private List<T> _events = new(0);
        private bool _isSendEventsInProgress;

        private int GroupMinSize => _currentConfig.GroupSize;
        private int MaxSyncInterval => _currentConfig.SyncIntervalInSeconds;
        [NotNull]
        public AnalyticsEventStorage<T> Storage { get; }

        public EventHandler(EventType type, [NotNull] IServerApi serverApi, [NotNull] AppStatePrefs appStatePrefs, string storagePath)
        {
            _serverApi = serverApi;
            _appStatePrefs = appStatePrefs;
            _type = type;
            _syncTimer = new SyncTimer(ForceSync);
            Storage = new AnalyticsEventStorage<T>(type, storagePath);
            _currentConfig = _defaultConfig = new AnalyticsConfiguration
            {
                GroupSize = GetGroupMizSizeForEventType(type),
                SyncIntervalInSeconds = GetMaxSyncIntervalForEventType(type)
            };
        }

        public void Initialize()
        {
            lock (_lock)
            {
                InitWithStorageThreadUnsafe();
            }
            ForceSync();
            _syncTimer.Start(MaxSyncInterval * 1000);
        }

        void IDisposable.Dispose()
        {
            _syncTimer.Dispose();
            _disposables.Release();
        }

        public void OnForeground()
        {
            _syncTimer.Start(MaxSyncInterval * 1000);
        }

        public void OnBackground()
        {
            _syncTimer.Stop();
            ForceSync();
        }

        public void ForceSync()
        {
            PerformSync(true).Forget();
        }

        public void HandleEvent(T @event)
        {
            lock (_lock)
            {
                _events.Add(@event);
                _logger.Log($"For {nameof(EventHandler<T>)} of type {typeof(T)} called {nameof(HandleEvent)} with event: \n{JsonFacade.SerializeObject(@event)}");
                StoreUnsentEventsThreadUnsafe();
            }
            PerformSync_Forget();
        }

        public void HandleEvents([NotNull] IEnumerable<T> events)
        {
            lock (_lock)
            {
                _events.AddRange(events);
                _logger.Log($"For {nameof(EventHandler<T>)} of type {typeof(T)} called {nameof(HandleEvents)} with events:\n{string.Join(",\n", _events.Select(JsonFacade.SerializeObject))}");
                StoreUnsentEventsThreadUnsafe();
            }
            PerformSync().Forget();
        }

        public void SetupAnalyticsConfig([NotNull] AnalyticsConfiguration config)
        {
            _currentConfig = config;
            OnConfigChanged();
        }

        public void ResetAnalyticsConfig()
        {
            _currentConfig = _defaultConfig;
            OnConfigChanged();
        }

        private void OnConfigChanged()
        {
            PerformSync().Forget();
            _syncTimer.Restart(MaxSyncInterval * 1000);
        }

        private void InitWithStorageThreadUnsafe()
        {
            try
            {
                _events = Storage.LoadEvents() ?? _events;
                _logger.Log($"For {nameof(EventHandler<T>)} of type {typeof(T)} loaded events:\n{string.Join(",\n", _events.Select(JsonFacade.SerializeObject))}");
            }
            catch (Exception e)
            {
                _logger.LogError($"Magify failed to load analytics events with exception: {e.Message}");
            }
        }

        private void StoreUnsentEventsThreadUnsafe()
        {
            try
            {
                Storage.SaveEvents(_events);
                _logger.Log($"For {nameof(EventHandler<T>)} of type {typeof(T)} saved events:\n{string.Join(",\n", _events.Select(JsonFacade.SerializeObject))}");
            }
            catch (Exception e)
            {
                throw new MagifyFailedToSaveEventsOnDiskException(e);
            }
        }

        private void PerformSync_Forget() => PerformSync().Forget();
        private async UniTaskVoid PerformSync(bool isForced = false)
        {
            T[] analyticsEvents;
            lock (_lock)
            {
                if (_isSendEventsInProgress) return;
                if (_appStatePrefs.IsPendingRestore.Value) return;
                if (_disposables.IsDisposed) return;
                if (_events.Count == 0) return;
                if (!isForced && _events.Count < GroupMinSize) return;

                _logger.Log($"For {nameof(EventHandler<T>)} of type {typeof(T)} called {nameof(PerformSync)} with events:\n{string.Join(",\n", _events.Select(JsonFacade.SerializeObject))}");
                _isSendEventsInProgress = true;
                analyticsEvents = _events.ToArray();
            }

            var sendResult = false;
            try
            {
                var cancellationToken = _disposables.GetOrCreateToken();
                await TaskScheduler.SwitchToThreadPool(cancellationToken);
                sendResult = await _serverApi.SendEvents(_type, analyticsEvents, cancellationToken);
                await TaskScheduler.SwitchToMainThread(cancellationToken);
            }
            catch (OperationCanceledException)
            {
                _logger.Log($"Send events({nameof(T)}) request was cancelled");
            }
            catch (Exception e)
            {
                _logger.LogError($"An exception was caught in the process of sending events: {e.Message}");
            }
            finally
            {
                lock (_lock)
                {
                    _isSendEventsInProgress = false;
                    if (sendResult)
                    {
                        try
                        {
                            for (var i = analyticsEvents.Length - 1; i >= 0; i--)
                            {
                                if (analyticsEvents[i] is CustomSessionEvent customSessionEvent && customSessionEvent.Params is not null)
                                {
                                   DictionaryPool<string, object>.Release(customSessionEvent.Params);
                                }
                                _events.Remove(analyticsEvents[i]);
                            }
                        }
                        catch (Exception e)
                        {
                            _logger.LogError($"Something went wrong during removing of sent analytics events: {e.Message}");
                        }
                        StoreUnsentEventsThreadUnsafe();
                    }
                }
                if (sendResult)
                {
                    PerformSync().Forget();
                }
                if (!_syncTimer.IsDisposed)
                    _syncTimer.Restart();
            }
        }

        private static int GetGroupMizSizeForEventType(EventType eventType)
        {
            switch (eventType)
            {
                case EventType.Impression:
                case EventType.ApplovinAdsImpression:
                case EventType.IronSourceAdsImpression:
                case EventType.SessionEvent:
                    return 10;
                default:
                    return 1;
            }
        }

        private static int GetMaxSyncIntervalForEventType(EventType eventType)
        {
            switch (eventType)
            {
                case EventType.Impression:
                    return 10;
                case EventType.ApplovinAdsImpression:
                case EventType.IronSourceAdsImpression:
                case EventType.SessionEvent:
                    return 60;
                default:
                    return 0;
            }
        }
    }
}