using System;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal class AnalyticsSessionTracker : IInitializable, IForegroundListener, IBackgroundListener, IDisposable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Analytics);

        [NotNull]
        private readonly IAnalyticsTracker _analyticsTracker;
        private readonly GeneralPrefs _generalPrefs;
        [NotNull]
        private readonly AppVersionProvider _appVersionProvider;
        [NotNull]
        private readonly AppStatePrefs _appStatePrefs;
        private readonly PlatformAPI _platform;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();

        private bool _isFirstSessionInVersion;
        private IDisposable _intervalDisposable;

        private long SessionStarted
        {
            get => _generalPrefs.SessionStarted;
            set => _generalPrefs.SessionStarted = value;
        }

        private long LastActivity
        {
            get => _generalPrefs.LastActivity;
            set => _generalPrefs.LastActivity = value;
        }

        public AnalyticsSessionTracker(
            [NotNull] IAnalyticsTracker analyticsTracker,
            GeneralPrefs generalPrefs,
            [NotNull] AppStatePrefs appStatePrefs,
            PlatformAPI platform,
            [NotNull] AppVersionProvider appVersionProvider)
        {
            _analyticsTracker = analyticsTracker;
            _generalPrefs = generalPrefs;
            _appStatePrefs = appStatePrefs;
            _platform = platform;
            _appVersionProvider = appVersionProvider;
        }

        void IInitializable.Initialize()
        {
            _generalPrefs.GlobalSessionCounter
                .SkipLatestValueOnSubscribe()
                .Where(_ => !_appStatePrefs.IsRestoreInProgress.Value)
                .Subscribe(_ => TrackAppLaunch())
                .AddTo(_disposables);

            _analyticsTracker.OnEventSent
                .Subscribe(_ => TrackActivity())
                .AddTo(_disposables);

            CheckFailedSession();
            StartSessionActivityListening();
        }

        void IDisposable.Dispose()
        {
            _intervalDisposable?.Dispose();
            _intervalDisposable = null;
            _disposables.Release();
        }

        void IBackgroundListener.OnBackground()
        {
            TrackAppBackgroundEvent(SessionStarted, DateTime.UtcNow.ToUnixMilliseconds(), _appVersionProvider.AppVersion);
            _analyticsTracker.HandeWentBackground();
        }

        void IForegroundListener.OnForeground()
        {
            if (SessionStarted > 0)
            {
                // Unexpected SessionStarted value on foreground event. It should be zero at this point.
                // Possible problem: Magify Initialized in Awake method in first loaded scene (means before first OnApplicationFocus call)
                // Just do nothing in this case because session already started
                return;
            }
            _analyticsTracker.HandeWentForeground();
            StartSessionActivityListening();
        }

        private void StartSessionActivityListening()
        {
            SessionStarted = DateTime.UtcNow.ToUnixMilliseconds();
            _intervalDisposable = Observable
                .Interval(TimeSpan.FromSeconds(30))
                .Subscribe(_ => TrackActivity());
            TrackActivity();
        }

        private void CheckFailedSession()
        {
            var sessionId = _generalPrefs.SessionId.Value;
            var openTimestamp = SessionStarted;
            var closeTimestamp = LastActivity;

            if (!string.IsNullOrEmpty(sessionId) && openTimestamp > 0 && closeTimestamp > 0)
            {
                var appVersion = _isFirstSessionInVersion ? _generalPrefs.LastInitializedVersion : _appVersionProvider.AppVersion;
                TrackAppBackgroundEvent(openTimestamp, closeTimestamp, appVersion);
            }
        }

        private void TrackAppBackgroundEvent(long start, long end, string appVersion)
        {
            _analyticsTracker.TrackAppBackgroundEvent(start, end, appVersion);

            SessionStarted = 0;
            LastActivity = 0;
            _intervalDisposable?.Dispose();
            _intervalDisposable = null;
        }

        internal void TrackAppLaunch()
        {
            _analyticsTracker.TrackAppLaunch();
            _generalPrefs.IsFirstAppLaunchTracked.Value = true;
        }

        internal void TrackFirstSessionInVersion()
        {
            _isFirstSessionInVersion = true;
        }

        private void TrackActivity()
        {
            LastActivity = DateTimeOffset.UtcNow.ToUnixTimeMilliseconds();
        }
    }
}