using System;
using System.IO;
using System.Threading;
using Cysharp.Threading.Tasks;
using UnityEngine;
using UnityEngine.Networking;
using Object = UnityEngine.Object;
using JetBrains.Annotations;

namespace Magify
{
    public class BundleRemoteStorage : RemoteStorage<AssetBundle>
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Storage);

        protected readonly Transform RootTransform;

        public BundleRemoteStorage(Transform rootTransform, string storagePath) : base(storagePath, "bundles")
        {
            RootTransform = rootTransform;
        }

        protected override void ReleaseContent(AssetBundle cache)
        {
            cache.Unload(true);
        }

        protected override AssetBundle LoadFromDisk(string filePath)
        {
            var assetBundle = AssetBundle.LoadFromFile(filePath);
            try
            {
                File.SetLastAccessTime(filePath, DateTime.Now);
            }
            catch (Exception e)
            {
                _logger.LogWarning(MagifyLogs.FailedToSetLastAccessTime(filePath, e));
            }
            return assetBundle;
        }

        protected override async UniTask<(StorageResultCode Code, AssetBundle Value)> DownloadFromUrlWithoutTimeout(string url, CancellationToken cancellationToken)
        {
            LogWithPath($"{nameof(DownloadFromUrlWithoutTimeout)} using {nameof(UnityWebRequestTexture)}", url);
            using var www = UnityWebRequest.Get(url);
            var storageResultCode = await SendWebRequest(www, cancellationToken);
            if (storageResultCode != StorageResultCode.Success)
            {
                return (storageResultCode, null);
            }

            LogWithPath($"Getting {nameof(BundledCreative)} from downloaded bundle", url);
            var bundleBytes = www.downloadHandler.data;
            var bundle = AssetBundle.LoadFromMemory(bundleBytes);
            if (bundle == null)
            {
                return (StorageResultCode.InvalidAssetError, null);
            }
            LogWithPath($"{nameof(AssetBundle)} with name {bundle.name} extracted", url);

            var isValid = IsContentValid(bundle);
            if (!isValid)
            {
                bundle.Unload(true);
                return (StorageResultCode.InvalidAssetError, null);
            }

            SaveToDisk(url, bundleBytes);
            return (StorageResultCode.Success, bundle);
        }

        private bool IsContentValid(AssetBundle bundle)
        {
            var content = bundle.GetMainGameObject();
            if (content == null) return false;
            if (content.GetComponent<BundledPrefab>() != null) return true;
            return IsBundledCreativeValid(content);
        }

        private bool IsBundledCreativeValid(GameObject content)
        {
            var componentInBundle = content.GetComponent<BundledCreative>();
            if (componentInBundle == null) return false;

            var instance = Object.Instantiate(content, RootTransform);
            var creativeContent = instance.GetComponent<BundledCreative>();
            creativeContent.Initialize();

            var hasRenderTexture = creativeContent.RenderTexture != null;
            Object.Destroy(instance.gameObject);

            return hasRenderTexture;
        }
    }
}