using System;
using System.Collections.Generic;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using Newtonsoft.Json;
using Newtonsoft.Json.Utilities;
using UnityEngine;
using Object = UnityEngine.Object;

namespace Magify
{
    [System.Diagnostics.CodeAnalysis.SuppressMessage("ReSharper", "AnnotationRedundancyAtValueType")]
    [System.Diagnostics.CodeAnalysis.SuppressMessage("ReSharper", "ContainerAnnotationRedundancy")]
    public static partial class MagifyManager
    {
        /// <summary>
        /// A callback notifying that a config of a certain <see cref="ConfigKind"/> has been loaded (from disk or network) and processed by SDK
        /// </summary>
        /// <remarks>
        /// Since config processing is a multithreaded operation, this callback is called on the same thread as the processing.
        /// For synchronized to Unity's main thread, see <see cref="OnConfigParsedOnMainThread"/>
        /// </remarks>
        public static event Action<ConfigKind> OnConfigParsed;

        /// <summary>
        /// <see cref="OnConfigParsed"/> callback variant synchronized with Unity main thread
        /// </summary>
        public static event Action<ConfigKind> OnConfigParsedOnMainThread;

        public static event Action OnConfigLoaded;

        public static event Action<int> OnSessionChanged;

        public static event Action OnSubscriptionStatusChanged;
        public static event Action OnInAppStatusChanged;
        public static event Action OnAuthorizationStatusChanged;
        public static event Action OnReferrerChanged;
        public static event Action OnPurchasedProductsChanged;

        public static event Action OnApplicationEnterForeground;
        public static event Action OnApplicationEnterBackground;

        /// <summary>
        /// Wrapper over <see cref="OnApplicationEnterForeground"/> <i>(invokes with true)</i> and <see cref="OnApplicationEnterBackground"/> <i>(invokes with false)</i>
        /// </summary>
        public static event Action<bool> OnApplicationFocus;

        /// <summary>
        /// Only supported on iOS
        /// </summary>
        public static event Action OnUserDidTakeScreenshot;

        [NotNull]
        private static event Action<MagifyPlatformAPI> OnMagifyPlatformAPIChanged = delegate {};

        [CanBeNull]
        private static MagifyLogger _logger;
        [CanBeNull]
        private static Transform _root;
        [CanBeNull]
        private static MagifyPresenter _presenter;
        [CanBeNull]
        private static MagifyPlatformAPI _magifyPlatformAPI;

        /// <remarks>
        /// Null can only be set at ShutDown
        /// </remarks>
        [CanBeNull]
        internal static MagifyPlatformAPI MagifyPlatformAPI
        {
            get => _magifyPlatformAPI;
            set
            {
                _magifyPlatformAPI = value;
                OnMagifyPlatformAPIChanged(value);
            }
        }

        public static bool Initialized => MagifyPlatformAPI is { IsSdkInitialized: true };

        [CanBeNull]
        public static ContextSyncTime LastContextSyncTime
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(LastContextSyncTime));
                return MagifyPlatformAPI!.LastContextSyncTime;
            }
        }

        public static bool IsPortrait
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(IsPortrait));
                return MagifyPlatformAPI!.IsPortrait;
            }
        }

        [Obsolete("You should no longer use this property. " +
                  "It is outdated and will be removed around the second half of 2026. " +
                  "Consider using " + nameof(MagifyConfig.IsSandbox))]
        public static Environment Environment
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(Environment));
                return MagifyPlatformAPI!.Environment.Value;
            }
        }

        [NotNull]
        [Obsolete("You should no longer use this property. It is outdated and will be removed around the second half of 2026.")]
        public static IObservable<Environment> OnEnvironmentChanged
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(OnEnvironmentChanged));
                return MagifyPlatformAPI!.Environment;
            }
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <summary>
        /// Allows you to check whether the SDK is in sandbox mode.
        /// </summary>
        public static bool IsSandbox
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(IsSandbox));
                return MagifyPlatformAPI!.IsSandbox.Value;
            }
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <summary>
        /// Allows to handle the sandbox mode changing.
        /// </summary>
        [NotNull]
        public static IObservable<bool> OnIsSandboxChanged
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(OnIsSandboxChanged));
                return MagifyPlatformAPI!.IsSandbox;
            }
        }

        /// <summary>
        /// Allows you to control purchases verification process and get the result of it
        /// </summary>
        [NotNull]
        public static IReactiveProperty<IPurchaseVerificationHandler> ExternalPurchaseVerificationHandler
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(ExternalPurchaseVerificationHandler));
                return MagifyPlatformAPI!.ExternalPurchaseVerificationHandler;
            }
        }

        /// <summary>
        /// The time in seconds between purchase verification attempts.
        /// </summary>
        /// <remarks>
        /// Minimum allowed value is 1 second.
        /// </remarks>
        [NotNull]
        public static IReactiveProperty<float> VerificationRetryInterval
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(VerificationRetryInterval));
                return MagifyPlatformAPI!.VerificationRetryInterval;
            }
        }

        public static SubscriptionStatus SubscriptionStatus
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(SubscriptionStatus));
                return MagifyPlatformAPI!.SubscriptionStatus;
            }
            set
            {
                ThrowIfMagifyIsNotReady(nameof(SubscriptionStatus));
                _logger!.Log($"Set {nameof(SubscriptionStatus)} to {value}");
                MagifyPlatformAPI!.SubscriptionStatus = value;
            }
        }

        public static AuthorizationStatus AuthorizationStatus
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(AuthorizationStatus));
                return MagifyPlatformAPI!.AuthorizationStatus;
            }
            set
            {
                ThrowIfMagifyIsNotReady(nameof(AuthorizationStatus));
                _logger!.Log($"Set {nameof(AuthorizationStatus)} to {value}");
                MagifyPlatformAPI!.AuthorizationStatus = value;
            }
        }

        public static InAppStatus InAppStatus
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(InAppStatus));
                return MagifyPlatformAPI!.InAppStatus;
            }
            set
            {
                ThrowIfMagifyIsNotReady(nameof(InAppStatus));
                MagifyPlatformAPI!.InAppStatus = value;
            }
        }

        public static string ReferrerId
        {
            [CanBeNull]
            get
            {
                ThrowIfMagifyIsNotReady(nameof(ReferrerId));
                return MagifyPlatformAPI!.ReferrerId;
            }
            [CanBeNull]
            set
            {
                ThrowIfMagifyIsNotReady(nameof(ReferrerId));
                _logger!.Log($"Set {nameof(ReferrerId)} to {value}");
                MagifyPlatformAPI!.ReferrerId = value;
            }
        }

        public static string AdjustId
        {
            [CanBeNull]
            get
            {
                ThrowIfMagifyIsNotReady(nameof(AdjustId));
                return MagifyPlatformAPI!.AdjustId;
            }
            [CanBeNull]
            set
            {
                ThrowIfMagifyIsNotReady(nameof(AdjustId));
                _logger!.Log($"Set {nameof(AdjustId)} to {value}");
                MagifyPlatformAPI!.AdjustId = value;
            }
        }

        public static string FirebaseInstanceId
        {
            [CanBeNull]
            get
            {
                ThrowIfMagifyIsNotReady(nameof(FirebaseInstanceId));
                return MagifyPlatformAPI!.FirebaseInstanceId;
            }
            [CanBeNull]
            set
            {
                ThrowIfMagifyIsNotReady(nameof(FirebaseInstanceId));
                _logger!.Log($"Set {nameof(FirebaseInstanceId)} to {value}");
                MagifyPlatformAPI!.FirebaseInstanceId = value;
            }
        }

        [NotNull]
        public static Limits Limits
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(Limits));
                return MagifyPlatformAPI!.Limits;
            }
        }

        /// <summary>
        /// The primary method of user identification in Magify SDK.
        /// Randomly generated GUID string.
        /// Used for linking analytics, saving progress and many other processes.
        /// Also, can be passed to other SDKs for linking.
        /// </summary>
        [NotNull]
        public static string ClientId
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(ClientId));
                return MagifyPlatformAPI!.ClientId;
            }
        }

        /// <summary>
        /// A method for identifying a user who has restored their progress from another device.
        /// </summary>
        /// <remarks>
        /// Most often, it will be identical to <see cref="ClientId"/>
        /// from the very first device on which this user saved their progress. </br>
        /// The value is always null if <see cref="Authorization.HasSocialAuthorizationData"/> is false.
        /// </remarks>
        [NotNull]
        public static IReadOnlyReactiveProperty<string> CommonClientId
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(CommonClientId));
                return MagifyPlatformAPI!.CommonClientId;
            }
        }

        public static bool CustomClientIdWasSet
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(CustomClientIdWasSet));
                return MagifyPlatformAPI!.CustomClientIdWasSet;
            }
        }

        [NotNull]
        public static IReadOnlyList<InAppProduct> InAppProducts
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(InAppProducts));
                return MagifyPlatformAPI!.InAppProducts;
            }
        }

        [NotNull]
        public static IReadOnlyList<SubscriptionProduct> SubscriptionProducts
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(SubscriptionProducts));
                return MagifyPlatformAPI!.SubscriptionProducts;
            }
        }

        /// <summary>
        /// Returns a list of segments for which the user is suitable.
        /// </summary>
        /// <remarks>
        /// Might be received only from remote config. There is no in default config.
        /// </remarks>
        [NotNull]
        public static IReadOnlyReactiveProperty<IReadOnlyList<string>> Segmentations
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(Segmentations));
                return MagifyPlatformAPI!.Segmentations;
            }
        }

        /// <summary>
        /// Returns a list of a/b-tests and groups for which the user is suitable.
        /// </summary>
        /// <remarks>
        /// Might be received only from remote config. There is no in default config.
        /// </remarks>
        [NotNull]
        public static IReadOnlyReactiveProperty<IReadOnlyList<AssignedAbTest>> AssignedAbTests
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(AssignedAbTests));
                return MagifyPlatformAPI!.AssignedAbTests;
            }
        }

        /// <summary>
        /// Be careful with this property, in the Full version of Magify SDK it will be loaded with app config/context.
        /// </summary>
        [NotNull]
        public static IReactiveProperty<TimeSpan> SessionsInterval
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(SessionsInterval));
                return MagifyPlatformAPI!.SessionsInterval;
            }
        }

        public static int SessionNumber
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(SessionNumber));
                return MagifyPlatformAPI!.SessionNumber;
            }
        }

        /// <summary>
        /// Returns first installed version of the app (that had Magify SDK integrated)
        /// </summary>
        [NotNull]
        public static string FirstInstalledVersion
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(FirstInstalledVersion));
                return MagifyPlatformAPI!.FirstInstalledVersion;
            }
        }

        /// <summary>
        /// Returns the date of the first installation of the app (that had Magify SDK integrated)
        /// </summary>
        public static DateTime FirstLaunchDate
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(FirstLaunchDate));
                return MagifyPlatformAPI!.FirstLaunchDate;
            }
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <inheritdoc cref="ConfigScope"/>
        [NotNull]
        public static IReactiveProperty<ConfigScope> RemoteConfigScopes
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(RemoteConfigScopes));
                return MagifyPlatformAPI!.RemoteContextScope;
            }
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <summary>
        /// Allows you to toggle user's and application's state synchronization.
        /// </summary>
        /// <remarks>
        /// Without this set to true <see cref="MagifyManager.Synchronization"/> and <see cref="MagifyManager.Authorization"/> are useless. <br/>
        /// If you want to control this property since initialization you can use <see cref="MagifyConfig.SyncStateEnabled"/>. <br/>
        /// If this flag was set to 'true' since initialization, the application state will attempt to be restored automatically. <br/>
        /// In case the flag was set to 'true' after initialization, no automatic restore will take place.
        /// </remarks>
        [NotNull]
        public static IReactiveProperty<bool> SyncStateEnabled
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(SyncStateEnabled));
                return MagifyPlatformAPI!.SyncStateEnabled;
            }
        }

        [NotNull]
        public static MagifyPresenter Presenter
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(Presenter));
                return _presenter!;
            }
        }

        [NotNull]
        public static Transform Root
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(Root));
                return _root!;
            }
        }

        public static bool IsSandboxHasChanged
        {
            get
            {
                ThrowIfMagifyIsNotReady(nameof(IsSandboxHasChanged));
                return MagifyPlatformAPI!.IsSandboxHasChanged;
            }
        }

        /// <exception cref="MagifyAlreadyInitializedException">
        /// If you try to initialize <see cref="MagifyManager"/> more than once.
        /// </exception>
        /// <exception cref="NoAppNameForMagifyInitializationException">
        /// If you try to initialize <see cref="MagifyManager"/> without <see cref="MagifyConfig.AppNameGP"/> or <see cref="MagifyConfig.AppNameIOS"/> in the settings.
        /// </exception>
        /// <exception cref="MagifyClientIdCannotBeChangedException">
        /// If you are attempting to pass <see cref="MagifyConfig.CustomClientId"/> for the second+ time, and it's now different from the previous one.
        /// </exception>
        public static void Initialize([NotNull] MagifyConfig config, [CanBeNull] MagifyDebugConfig debugConfig = null)
        {
            AotHelper.EnsureList<LtoModel>();
            AotHelper.EnsureDictionary<string, object>();

            if (MagifyPlatformAPI != null)
            {
                throw new MagifyAlreadyInitializedException();
            }

            _logger = MagifyLogger.Get();
            MagifyLogger.IsLoggingEnabled = config.IsLoggingEnabled;

            var editorDevice = config.EditorDevice ?? EditorDevice.GetDefaultAndroidDevice();
            var platform = GetCurrentPlatform(editorDevice);
            var appName = platform switch
            {
                RuntimePlatform.Android => config.AppNameGP,
                RuntimePlatform.IPhonePlayer => config.AppNameIOS,
                _ => string.Empty
            };
            var configPath = platform switch
            {
                RuntimePlatform.Android => config.ConfigPathGP,
                RuntimePlatform.IPhonePlayer => config.ConfigPathIOS,
                _ => string.Empty
            };
            configPath ??= string.Empty;

            if (string.IsNullOrEmpty(appName))
            {
                throw new NoAppNameForMagifyInitializationException(platform);
            }

            MagifyPlatformAPI = new MagifyCSharp(
                config.EditorMoq,
                config.IsDeveloperMode,
                !config.UseAdvancedVersion,
                editorDevice,
                appName,
                configPath,
                config.CustomClientId,
                config.StoragePath,
                config.ClientStateConfig,
#pragma warning disable CS0618 // Type or member is obsolete
                config.Environment,
#pragma warning restore CS0618 // Type or member is obsolete
                config.IsSandbox);

            try
            {

#if MAGIFY_VERBOSE_LOGGING || MAGIFY_DEBUG || UNITY_EDITOR
                try
                {
                    if (debugConfig != null)
                    {
                        (MagifyPlatformAPI as MagifyCSharp)!.Client!.Platform!.CustomAndroidId = debugConfig.CustomAndroidId;
                        (MagifyPlatformAPI as MagifyCSharp)!.Client!.Platform!.CustomBuildNumber = debugConfig.CustomBuildNumber;
                        if (!string.IsNullOrEmpty(debugConfig.CustomAppVersion))
                        {
                            (MagifyPlatformAPI as MagifyCSharp)!.Client!.AppVersion = debugConfig.CustomAppVersion;
                        }
                    }
                }
                catch (Exception)
                {
                    /* Ignore */
                }
#endif

                _logger.Log("Create root GameObject");
                var gameObject = new GameObject("Magify SDK");
                Object.DontDestroyOnLoad(gameObject);
                _root = gameObject.transform;
                _root.position = Vector3.zero;
                _root.rotation = Quaternion.identity;
                _root.localScale = Vector3.one;
                var taskScheduler = gameObject.AddComponent<TaskScheduler>()!;

                _logger.Log("Initialize LTO");
                Lto.Initialize(taskScheduler);
                _logger.Log("Initialize Features");
                Features.Initialize(MagifyPlatformAPI);
                _logger.Log("Init native callbacks");
                InitCallbacks(MagifyPlatformAPI, _logger);

                // Specific for migration from iOS native SDK
                // ToDo: remove after removing of migration from native iOS SDK
                SubscriptionStatus = config.SubscriptionStatus;
                AuthorizationStatus = config.AuthorizationStatus;

                if (config.CustomClientId == null && !MagifyPlatformAPI.CustomClientIdWasSet)
                {
                    if (config.SyncStateEnabled != null)
                    {
                        MagifyPlatformAPI.SyncStateEnabled.Value = config.SyncStateEnabled.Value;
                    }
                    MagifyPlatformAPI.IsAutoRestoreStateEnabled.Value = config.IsAutoRestoreStateEnabled;
                    MagifyPlatformAPI.ClientIdFromCloudLoadingSecondsTimeout = config.ClientIdFromCloudLoadingTimeout;
                    MagifyPlatformAPI.SkipClientIdFromCloudLoading = config.SkipClientIdFromCloudLoading;
                    MagifyPlatformAPI.RemoteContextScope.Value = config.RemoteConfigScopes;
                }

                _logger.Log($"Initialize Magify SDK ({MagifyPlatformAPI.GetType().Name}). {nameof(appName)}={appName}; {nameof(configPath)}={configPath}");
                MagifyPlatformAPI.InitializeSdk();
                _logger.Log($"Magify SDK has been initialized. {nameof(ClientId)} is {ClientId}");
                MagifyPlatformAPI.IsLoggingEnabled = config.IsLoggingEnabled;

                SubscriptionStatus = config.SubscriptionStatus;
                AuthorizationStatus = config.AuthorizationStatus;
                if (config.IsGdprApplied != null)
                {
                    TrackGdprAccessState(config.IsGdprApplied.Value);
                }
                if (config.IsAttAuthorized != null)
                {
                    SetAttStatus(config.IsAttAuthorized.Value);
                }

                MagifyPlatformAPI.SetupConversionTracker(
                    $"{Application.streamingAssetsPath}/{config.RevenuePerCountryPath}",
                    $"{Application.streamingAssetsPath}/{config.RevenueLevelsPath}",
                    $"{Application.streamingAssetsPath}/{config.DefaultCurrencyRatesPath}",
                    $"{Application.streamingAssetsPath}/{config.SubscriptionMultipliersPath}"
                );

                _logger.Log("Setup SDK");
                MagifyPlatformAPI.Setup(taskScheduler);
                _logger.Log("Prepare storages");
                Storage.Setup(config.StoragePath);
                _logger.Log("Setup lto offers");
                Lto.Setup();
                _logger.Log("Prepare presenter");
                _presenter = MagifyPresenter.BuildDefaultPresenter(config.PresenterSettings, Root);
            }
            catch (Exception)
            {
                try
                {
                    ShutDown();
                }
                catch (Exception)
                {
                    // ignored
                }
                throw;
            }
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <summary>
        /// This method allows to wait until completion of parsing and handling of the initial configs (default and saved)
        /// </summary>
        /// <remarks>
        /// Finishes right before <see cref="OnConfigParsed"/> call
        /// </remarks>
        public static UniTask AwaitInitialConfigParsing(CancellationToken cancellationToken)
        {
            ThrowIfMagifyIsNotReady(nameof(AwaitInitialConfigParsing));
            return MagifyPlatformAPI!.AwaitInitialConfigParsing(cancellationToken);
        }

        private static RuntimePlatform GetCurrentPlatform([CanBeNull] EditorDevice editorDevice)
        {
            if (!Application.isEditor)
            {
                return Application.platform;
            }
            return editorDevice?.Platform ?? Application.platform;
        }

        private static void InitCallbacks([NotNull] MagifyPlatformAPI platformApi, [NotNull] MagifyLogger logger)
        {
            platformApi.OnConfigParsed += configKind =>
            {
                logger.Log($"{nameof(OnConfigParsed)} of {configKind} kind has been called.");
                OnConfigParsed?.Invoke(configKind);
            };

            platformApi.OnConfigParsedOnMainThread += configKind =>
            {
                logger.Log($"{nameof(OnConfigParsedOnMainThread)} of {configKind} kind has been called.");
                OnConfigParsedOnMainThread?.Invoke(configKind);
            };

            platformApi.OnConfigLoaded += () =>
            {
                logger.Log($"{nameof(OnConfigLoaded)} has been called.");
                OnConfigLoaded?.Invoke();
            };

            platformApi.OnSessionChanged += (sessionNumber) =>
            {
                logger.Log($"{nameof(OnSessionChanged)} has been called.");
                OnSessionChanged?.Invoke(sessionNumber);
            };

            platformApi.OnSubscriptionStatusChanged += () =>
            {
                logger.Log($"{nameof(OnSubscriptionStatusChanged)} has been called.");
                OnSubscriptionStatusChanged?.Invoke();
            };

            platformApi.OnInAppStatusChanged += () =>
            {
                logger.Log($"{nameof(OnInAppStatusChanged)} has been called.");
                OnInAppStatusChanged?.Invoke();
            };

            platformApi.OnAuthorizationStatusChanged += () =>
            {
                logger.Log($"{nameof(OnAuthorizationStatusChanged)} has been called.");
                OnAuthorizationStatusChanged?.Invoke();
            };

            platformApi.OnReferrerChanged += () =>
            {
                logger.Log($"{nameof(OnReferrerChanged)} has been called.");
                OnReferrerChanged?.Invoke();
            };

            platformApi.OnPurchasedProductsChanged += () =>
            {
                logger.Log($"{nameof(OnPurchasedProductsChanged)} has been called.");
                OnPurchasedProductsChanged?.Invoke();
            };

            platformApi.OnApplicationEnterForeground += () =>
            {
                logger.Log($"{nameof(OnApplicationEnterForeground)} has been called.");
                OnApplicationEnterForeground?.Invoke();
                OnApplicationFocus?.Invoke(true);
            };

            platformApi.OnApplicationEnterBackground += () =>
            {
                logger.Log($"{nameof(OnApplicationEnterBackground)} has been called.");
                OnApplicationEnterBackground?.Invoke();
                OnApplicationFocus?.Invoke(false);
            };

            platformApi.OnUserDidTakeScreenshot += () =>
            {
                logger.Log($"{nameof(OnUserDidTakeScreenshot)} has been called.");
                OnUserDidTakeScreenshot?.Invoke();
            };
        }

        public static void ShutDown()
        {
            if (MagifyPlatformAPI != null)
            {
                MagifyPlatformAPI.Dispose();
                MagifyPlatformAPI = default;
            }

            OnConfigLoaded = null;
            OnSessionChanged = null;
            OnSubscriptionStatusChanged = null;
            OnInAppStatusChanged = null;
            OnAuthorizationStatusChanged = null;
            OnReferrerChanged = null;
            OnPurchasedProductsChanged = null;
            OnApplicationEnterForeground = null;
            OnApplicationEnterBackground = null;
            OnUserDidTakeScreenshot = null;

            Lto.Reset();
            Features.Dispose();
            Storage.Clear();
            Logging.Reset();

            if (_presenter != null) destroy(_presenter.gameObject);
            if (_root != null) destroy(_root.gameObject);
            _root = default;
            _presenter = default;
            _logger = default;

            void destroy(Object obj)
            {
                switch (obj != null, Application.isPlaying)
                {
                    case (true, true):
                        Object.Destroy(obj);
                        break;
                    case (true, false):
                        Object.DestroyImmediate(obj);
                        break;
                }
            }
        }

        public static void Sync()
        {
            ThrowIfMagifyIsNotReady(nameof(Sync));
            _logger!.Log("Sync config with backend");
            MagifyPlatformAPI!.Sync();
        }

        public static void Reset(bool clearNativeStorage = true, bool clearCloudStorage = true)
        {
            ThrowIfMagifyIsNotReady(nameof(Reset));

            Lto.Reset();
            Storage.Clear();
            Logging.Reset();
            MagifyPlatformAPI!.Reset(clearNativeStorage, clearCloudStorage);
        }

        public static void Reset(Environment newEnvironment, bool isSandbox, bool clearNativeStorage = true, bool clearCloudStorage = true)
        {
            ThrowIfMagifyIsNotReady(nameof(Reset));

            Lto.Reset();
            Storage.Clear();
            MagifyPlatformAPI!.Reset(newEnvironment, isSandbox, clearNativeStorage, clearCloudStorage);
        }

        /// <inheritdoc cref="MagifyCSharp.ClearNativeStorage"/>
        /// <exception cref="MagifyNotInitializedException">
        ///     <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// </exception>
        /// <remarks>
        ///     <inheritdoc cref="MagifyDocs.TestOnly" path="/remarks"/>
        /// </remarks>
        public static void ClearNativeStorage()
        {
            ThrowIfMagifyIsNotReady(nameof(ClearNativeStorage));
            MagifyPlatformAPI!.ClearNativeStorage();
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <remarks><inheritdoc cref="MagifyDocs.TestOnly" path="/remarks"/></remarks>
        public static void ResetAndForceUpdate()
        {
            ThrowIfMagifyIsNotReady(nameof(ResetAndForceUpdate));
            MagifyPlatformAPI!.ResetAndForceUpdate();
        }

        /// <summary>
        /// Tweak environment and isSandbox flag.
        /// The SDK would be reset if the environment and isSandbox flag values were not saved.
        /// </summary>
        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <remarks><inheritdoc cref="MagifyDocs.TestOnly" path="/remarks"/></remarks>
        public static void TweakEnvironment(Environment? environment, bool? isSandbox)
        {
#pragma warning disable CS0618 // Type or member is obsolete
            ThrowIfMagifyIsNotReady(nameof(TweakEnvironment));
            if (!environment.HasValue) environment = Environment;
            if (!isSandbox.HasValue) isSandbox = IsSandbox;

            if (environment.Value == Environment && isSandbox.Value == IsSandbox)
            {
                _logger!.Log($"Setting {nameof(Environment)} and {nameof(IsSandbox)} is skipped, because values are the save");
                return;
            }

            _logger!.Log($"Set {nameof(Environment)} to {environment.ToString()}({(isSandbox.Value ? string.Intern("sandbox") : string.Intern("non sandbox"))})");
            Reset(environment.Value, isSandbox.Value);
#pragma warning restore CS0618 // Type or member is obsolete
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <remarks><inheritdoc cref="MagifyDocs.TestOnly" path="/remarks"/></remarks>
        public static void TweakAnalyticsConfig(int eventsGroupSize, int syncTimeInterval)
        {
            ThrowIfMagifyIsNotReady(nameof(TweakAnalyticsConfig));
            MagifyPlatformAPI!.TweakAnalyticsConfig(eventsGroupSize, syncTimeInterval);
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <remarks><inheritdoc cref="MagifyDocs.TestOnly" path="/remarks"/></remarks>
        public static void ResetAnalyticsConfig()
        {
            ThrowIfMagifyIsNotReady(nameof(ResetAnalyticsConfig));
            MagifyPlatformAPI!.ResetAnalyticsConfig();
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <remarks><inheritdoc cref="MagifyDocs.TestOnly" path="/remarks"/></remarks>
        public static void TweakFirstLaunchDate(DateTime date)
        {
            ThrowIfMagifyIsNotReady(nameof(TweakFirstLaunchDate));
            MagifyPlatformAPI!.TweakFirstLaunchDate(date);
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <remarks><inheritdoc cref="MagifyDocs.TestOnly" path="/remarks"/></remarks>
        public static void TweakUserLocale([NotNull] string languageTag)
        {
            ThrowIfMagifyIsNotReady(nameof(TweakUserLocale));
            MagifyPlatformAPI!.TweakUserLocale(languageTag);
        }

        public static void SetAttStatus(bool authorized)
        {
            ThrowIfMagifyIsNotReady(nameof(SetAttStatus));
            MagifyPlatformAPI!.SetAttStatus(authorized);
        }

        public static void SetMediaSource([CanBeNull] string networkName = null, [CanBeNull] string campaignName = null, [CanBeNull] string adGroup = null)
        {
            ThrowIfMagifyIsNotReady(nameof(SetMediaSource));
            MagifyPlatformAPI!.SetMediaSource(networkName, campaignName, adGroup);
        }

        public static void SetUserEmail([NotNull] string email)
        {
            ThrowIfMagifyIsNotReady(nameof(SetUserEmail));
            if (EmailValidator.IsValidEmail(email, out var error))
            {
                MagifyPlatformAPI!.SetUserEmail(email);
            }
            else
            {
                _logger!.LogError($"The user's email has an invalid format, validation failed with an error: {error}");
            }
        }

        [CanBeNull]
        public static ICampaign CampaignFor([NotNull] string eventName, CampaignRequestFlags flags, [CanBeNull] Dictionary<string, object> customParams = null)
        {
            ThrowIfMagifyIsNotReady(nameof(CampaignFor));
            var silent = flags.HasFlag(CampaignRequestFlags.Silent);
            _logger!.Log($"{nameof(CampaignFor)} called with flags [{flags}] for '{eventName}' (silent={silent}).\nParams:\n{JsonFacade.SerializeObject(customParams, Formatting.Indented)}");
            if (flags.HasFlag(CampaignRequestFlags.ReplaceCampaignTypeWithString) && customParams is { Count: > 0 })
            {
                customParams.ReplaceCampaignTypeWithString();
            }
            return CampaignFor(eventName, customParams, silent);
        }

        [CanBeNull]
        public static ICampaign CampaignFor([NotNull] string eventName, [CanBeNull] IReadOnlyDictionary<string, object> customParams = null, bool silent = false)
        {
            ThrowIfMagifyIsNotReady(nameof(CampaignFor));
            _logger!.Log($"Finding campaign for '{eventName}' (silent={silent}).\nParams:\n{JsonFacade.SerializeObject(customParams, Formatting.Indented)}");
            var campaign = MagifyPlatformAPI?.CampaignFor(eventName, customParams, silent);
            _logger.Log($"Campaign for '{eventName}' (silent={silent}): {(campaign != null ? $"\n{JsonFacade.SerializeObject(campaign, Formatting.Indented)}" : "null")}");
            return campaign;
        }

        public static bool IsCampaignAvailable([NotNull] string campaignName, [NotNull] string eventName, [CanBeNull] Dictionary<string, object> customParams = null)
        {
            ThrowIfMagifyIsNotReady(nameof(IsCampaignAvailable));
            _logger!.Log($"Checking is campaign '{campaignName}' for '{eventName}' available. Params: \n{JsonFacade.SerializeObject(customParams, Formatting.Indented)}");
            var available = MagifyPlatformAPI!.IsCampaignAvailable(campaignName, eventName, customParams);
            _logger.Log($"Campaign '{campaignName}' for '{eventName}' is {(available ? "available" : "not available")}");
            return available;
        }

        [CanBeNull]
        public static CampaignImpression LastImpressionFor(CampaignType campaignType, [NotNull] string campaignName, [NotNull] string eventName)
        {
            ThrowIfMagifyIsNotReady(nameof(LastImpressionFor));
            return MagifyPlatformAPI!.LastImpressionFor(campaignType, campaignName, eventName);
        }

        // ToDo: make obsolete, replace to SubscribeCampaignUpdates(ICampaign campaign, Action onUpdate)
        public static void SubscribeCampaignUpdates([NotNull] string campaignName, [NotNull] Action<ICampaign> updateAction)
        {
            ThrowIfMagifyIsNotReady(nameof(SubscribeCampaignUpdates));
            MagifyPlatformAPI!.SubscribeCampaignUpdates(campaignName, updateAction);
        }

        // ToDo: make obsolete, replace to UnsubscribeCampaignUpdates(string campaignName, Action onUpdate)
        public static void UnsubscribeCampaignUpdates([NotNull] string campaignName)
        {
            ThrowIfMagifyIsNotReady(nameof(UnsubscribeCampaignUpdates));
            MagifyPlatformAPI!.UnsubscribeCampaignUpdates(campaignName);
        }

        public static bool HasProcessedPurchase([NotNull] string productId)
        {
            ThrowIfMagifyIsNotReady(nameof(HasProcessedPurchase));
            return MagifyPlatformAPI!.HasProcessedPurchase(productId);
        }

        /// <summary>
        /// Will return true if creative has default version to show
        /// </summary>
        public static bool HasDefaultCreative([NotNull] ICampaignWithCreative campaign)
        {
            ThrowIfMagifyIsNotReady(nameof(HasDefaultCreative));
            return campaign.Creative switch
            {
                ImageCreative imageCreative when IsPortrait => Storage.HasDefaultCreative(imageCreative.Portrait),
                ImageCreative imageCreative when !IsPortrait => Storage.HasDefaultCreative(imageCreative.Landscape),
                BundleCreative bundleCreative => Storage.HasDefaultBundle(bundleCreative.Url),
                _ => false,
            };
        }

        /// <summary>
        /// Will return true if remote creative has been downloaded
        /// </summary>
        public static bool IsRemoteCreativeDownloaded([NotNull] ICampaignWithCreative campaign)
        {
            ThrowIfMagifyIsNotReady(nameof(IsRemoteCreativeDownloaded));
            return campaign.Creative switch
            {
                ImageCreative imageCreative when IsPortrait => Storage.ImageIsCached(imageCreative.Portrait),
                ImageCreative imageCreative when !IsPortrait => Storage.ImageIsCached(imageCreative.Landscape),
                BundleCreative bundleCreative => Storage.BundleIsCached(bundleCreative.Url),
                _ => false,
            };
        }

        /// <summary>
        /// Will return true if everything has already been downloaded to show the creative.
        /// </summary>
        [Obsolete("Use " + nameof(IsRemoteCreativeDownloaded) + " instead", true)]
        public static bool IsCreativeCached(ICampaignWithCreative campaign)
        {
            throw new NotSupportedException("Use " + nameof(IsRemoteCreativeDownloaded) + " instead");
        }

        /// <summary>
        /// Checks whether an application with provided <paramref name="identifier"/> is installed on this device
        /// </summary>
        /// <param name="identifier">
        /// For Android it is `Bundle ID` of an application
        /// For iOS it is `URL scheme` of an application
        /// </param>
        /// <remarks>
        /// Also used internally to verify suitability of <see cref="CrossPromoProduct"/>
        /// </remarks>
        public static bool IsApplicationInstalled([CanBeNull] string identifier)
        {
            ThrowIfMagifyIsNotReady(nameof(IsApplicationInstalled));
            return MagifyPlatformAPI!.IsApplicationInstalled(identifier);
        }

        #region Tracking

        public static void TrackGdprAccessState(bool accessState)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackGdprAccessState));
            _logger!.Log("Track Gdpr Access State");
            MagifyPlatformAPI!.TrackGdprAccessState(accessState);
        }

        public static void TrackAppLaunch()
        {
            ThrowIfMagifyIsNotReady(nameof(TrackAppLaunch));
            _logger!.Log("Track app launch");
            MagifyPlatformAPI!.TrackAppLaunch();
        }

        /// <summary>
        /// Allows you to track subscription product purchases initiated from a Magify campaign.
        /// This will track analytics, track used campaign product, initiate validation of purchase, update data for filtering and segmentation (update list of received products).
        /// </summary>
        /// <param name="isTrial">Was it the activation of a free trial subscription</param>
        /// <param name="productId">The ID of the purchased subscription product, as defined in the store</param>
        /// <param name="price">Price of the purchased subscription</param>
        /// <param name="currency">Currency of the purchased subscription</param>
        /// <param name="transactionId">
        /// ID number of the performed purchase (not the product ID, but the transaction itself).<br/>
        /// For Android it is called `Order ID` from product's receipt
        /// For iOS it is called `Transaction ID`
        /// </param>
        /// <param name="purchaseToken"> Represents Android purchase token. Null for all other platforms.</param>
        /// <param name="originalTransactionId">
        /// If this is a subscription renewal, you must provide the ID of the very first transaction that activated the subscription.<br/>
        /// For Android it is called Order Id from product's receipt (same as <paramref name="transactionId"/>)
        /// For iOS it is called `Original Transaction ID`
        /// </param>
        /// <param name="receipt">The receipt of the transaction</param>
        /// <param name="store">
        /// The store where the product is defined and purchased. See <see cref="PurchaseStore"/> for more details.
        /// </param>
        public static void TrackSubscriptionActivation(bool isTrial, [NotNull] string productId, [CanBeNull] string price, [CanBeNull] string currency, [CanBeNull] string transactionId, [CanBeNull] string purchaseToken, [CanBeNull] string originalTransactionId, [CanBeNull] string receipt, PurchaseStore store)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackSubscriptionActivation));
            _logger!.Log($"Track subscription activation. {nameof(isTrial)}={isTrial}; {nameof(productId)}={productId}; {nameof(price)}={price}; {nameof(currency)}={currency}; {nameof(transactionId)}={transactionId}; {nameof(purchaseToken)}={purchaseToken}; {nameof(originalTransactionId)}={originalTransactionId}, {nameof(store)}={store}, {nameof(receipt)}={receipt}");
            MagifyPlatformAPI!.TrackSubscriptionActivation(isTrial, productId, price, currency, transactionId, purchaseToken, originalTransactionId, receipt, needVerification: true, store);
        }

        /// <summary>
        /// Allows you to track subscription product purchases thar weren't initiated from a Magify campaign (if you were to implement your own purchase offer).
        /// This will track analytics, initiate validation of purchase, update data for filtering and segmentation (update list of received products).
        /// </summary>
        /// <param name="isTrial">Was it the activation of a free trial subscription</param>
        /// <param name="productId">The ID of the purchased subscription product, as defined in the store</param>
        /// <param name="price">Price of the purchased subscription</param>
        /// <param name="currency">Currency of the purchased subscription</param>
        /// <param name="transactionId">
        /// ID number of the performed purchase (not the product ID, but the transaction itself).<br/>
        /// For Android it is called `Order ID` from product's receipt
        /// For iOS it is called `Transaction ID`
        /// </param>
        /// <param name="purchaseToken"> Represents Android purchase token. Null for all other platforms.</param>
        /// <param name="originalTransactionId">
        /// If this is a subscription renewal, you must provide the ID of the very first transaction that activated the subscription.<br/>
        /// For Android it is called Order Id from product's receipt (same as <paramref name="transactionId"/>)
        /// For iOS it is called `Original Transaction ID`
        /// </param>
        /// <param name="receipt">The receipt of the transaction</param>
        /// <param name="store">
        /// The store where the product is defined and purchased. See <see cref="PurchaseStore"/> for more details.
        /// </param>
        public static void TrackExternalSubscriptionActivation(bool isTrial, [NotNull] string productId, [CanBeNull] string price, [CanBeNull] string currency, [CanBeNull] string transactionId, [CanBeNull] string purchaseToken, [CanBeNull] string originalTransactionId, [CanBeNull] string receipt, PurchaseStore store)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackExternalSubscriptionActivation));
            _logger!.Log($"Track external subscription activation. {nameof(isTrial)}={isTrial}; {nameof(productId)}={productId}; {nameof(price)}={price}; {nameof(currency)}={currency}; {nameof(transactionId)}={transactionId}; {nameof(purchaseToken)}={purchaseToken}; {nameof(originalTransactionId)}={originalTransactionId}, {nameof(store)}={store}, {nameof(receipt)}={receipt}");
            MagifyPlatformAPI!.TrackExternalSubscriptionActivation(isTrial, productId, price, currency, transactionId, purchaseToken, originalTransactionId, receipt, needVerification: true, store);
        }

        /// <summary>
        /// Allows you to track in-app product purchases initiated from a Magify campaign.
        /// This will track analytics, track used campaign product, initiate validation of purchase, update data for filtering and segmentation
        /// (update list of received products, <see cref="MagifyManager.InAppStatus"/> change to <see cref="InAppStatus.Purchased"/>).
        /// </summary>
        /// <param name="productId">The ID of the purchased product, as defined in the store</param>
        /// <param name="price">Price of the purchased subscription</param>
        /// <param name="currency">Currency of the purchased subscription</param>
        /// <param name="transactionId">
        /// ID number of the performed purchase (not the product ID, but the transaction itself).<br/>
        /// For Android it is called `Order ID` from product's receipt
        /// For iOS it is called `Transaction ID`
        /// </param>
        /// <param name="purchaseToken"> Represents Android purchase token. Null for all other platforms.</param>
        /// <param name="originalTransactionId">
        /// If this is a subscription renewal, you must provide the ID of the very first transaction that activated the subscription.<br/>
        /// For Android it is called Order Id from product's receipt (same as <paramref name="transactionId"/>)
        /// For iOS it is called `Original Transaction ID`
        /// </param>
        /// <param name="receipt">The receipt of the transaction</param>
        /// <param name="store">
        /// The store where the product is defined and purchased. See <see cref="PurchaseStore"/> for more details.
        /// </param>
        public static void TrackInAppFor([NotNull] string productId, [CanBeNull] string price, [CanBeNull] string currency, [CanBeNull] string transactionId, [CanBeNull] string purchaseToken, [CanBeNull] string originalTransactionId, [CanBeNull] string receipt, PurchaseStore store)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackInAppFor));
            _logger!.Log($"Track inapp. {nameof(productId)}={productId}; {nameof(price)}={price}; {nameof(currency)}={currency}; {nameof(transactionId)}={transactionId}; {nameof(purchaseToken)}={purchaseToken}; {nameof(originalTransactionId)}={originalTransactionId}, {nameof(store)}={store}, {nameof(receipt)}={receipt}");
            MagifyPlatformAPI!.TrackInAppFor(productId, price, currency, transactionId, purchaseToken, originalTransactionId, receipt, needVerification: true, store);
        }

        /// <summary>
        /// Allows you to track in-app product purchases thar weren't initiated from a Magify campaign (if you were to implement your own purchase offer).
        /// This will track analytics, initiate validation of purchase, update data for filtering and segmentation
        /// (update list of received products, <see cref="MagifyManager.InAppStatus"/> change to <see cref="InAppStatus.Purchased"/>).
        /// </summary>
        /// <param name="productId">The ID of the purchased product, as defined in the store</param>
        /// <param name="price">Price of the purchased subscription</param>
        /// <param name="currency">Currency of the purchased subscription</param>
        /// <param name="transactionId">
        /// ID number of the performed purchase (not the product ID, but the transaction itself).<br/>
        /// For Android it is called `Order ID` from product's receipt
        /// For iOS it is called `Transaction ID`
        /// </param>
        /// <param name="purchaseToken"> Represents Android purchase token. Null for all other platforms.</param>
        /// <param name="originalTransactionId">
        /// If this is a subscription renewal, you must provide the ID of the very first transaction that activated the subscription.<br/>
        /// For Android it is called Order Id from product's receipt (same as <paramref name="transactionId"/>)
        /// For iOS it is called `Original Transaction ID`
        /// </param>
        /// <param name="receipt">The receipt of the transaction</param>
        /// <param name="store">
        /// The store where the product is defined and purchased. See <see cref="PurchaseStore"/> for more details.
        /// </param>
        public static void TrackExternalInAppFor([NotNull] string productId, [CanBeNull] string price, [CanBeNull] string currency, [CanBeNull] string transactionId, [CanBeNull] string purchaseToken, [CanBeNull] string originalTransactionId, [CanBeNull] string receipt, PurchaseStore store)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackExternalInAppFor));
            _logger!.Log($"Track external inapp. {nameof(productId)}={productId}; {nameof(price)}={price}; {nameof(currency)}={currency}; {nameof(transactionId)}={transactionId}; {nameof(purchaseToken)}={purchaseToken}; {nameof(originalTransactionId)}={originalTransactionId}, {nameof(store)}={store}, {nameof(receipt)}={receipt}");
            MagifyPlatformAPI!.TrackExternalInAppFor(productId, price, currency, transactionId, purchaseToken, originalTransactionId, receipt, needVerification: true, store);
        }

        public static void TrackSubscriptionActivationWithoutVerification(bool isTrial, [NotNull] string productId, [CanBeNull] string price, [CanBeNull] string currency, [CanBeNull] string transactionId, [CanBeNull] string originalTransactionId, PurchaseStore store)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackSubscriptionActivationWithoutVerification));
            _logger!.Log($"Track subscription activation without validation. {nameof(isTrial)}={isTrial}; {nameof(productId)}={productId}; {nameof(price)}={price}; {nameof(currency)}={currency}; {nameof(transactionId)}={transactionId}; {nameof(originalTransactionId)}={originalTransactionId}, {nameof(store)}={store}");
            MagifyPlatformAPI!.TrackSubscriptionActivation(isTrial, productId, price, currency, transactionId, purchaseToken: null, originalTransactionId, receipt: null, needVerification: false, store);
        }

        public static void TrackExternalSubscriptionActivationWithoutVerification(bool isTrial, [NotNull] string productId, [CanBeNull] string price, [CanBeNull] string currency, [CanBeNull] string transactionId, [CanBeNull] string originalTransactionId, PurchaseStore store)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackExternalSubscriptionActivationWithoutVerification));
            _logger!.Log($"Track external subscription activation without validation. {nameof(isTrial)}={isTrial}; {nameof(productId)}={productId}; {nameof(price)}={price}; {nameof(currency)}={currency}; {nameof(transactionId)}={transactionId}; {nameof(originalTransactionId)}={originalTransactionId}, {nameof(store)}={store}");
            MagifyPlatformAPI!.TrackExternalSubscriptionActivation(isTrial, productId, price, currency, transactionId, purchaseToken: null, originalTransactionId, receipt: null, needVerification: false, store);
        }

        public static void TrackInAppWithoutVerification([NotNull] string productId, [CanBeNull] string price, [CanBeNull] string currency, [CanBeNull] string transactionId, [CanBeNull] string originalTransactionId, PurchaseStore store)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackInAppWithoutVerification));
            _logger!.Log($"Track inapp without validation. {nameof(productId)}={productId}; {nameof(price)}={price}; {nameof(currency)}={currency}; {nameof(transactionId)}={transactionId}; {nameof(originalTransactionId)}={originalTransactionId}, {nameof(store)}={store}");
            MagifyPlatformAPI!.TrackInAppFor(productId, price, currency, transactionId, purchaseToken: null, originalTransactionId, receipt: null, needVerification: false, store);
        }

        public static void TrackExternalInAppWithoutVerification([NotNull] string productId, [CanBeNull] string price, [CanBeNull] string currency, [CanBeNull] string transactionId, [CanBeNull] string originalTransactionId, PurchaseStore store)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackExternalInAppWithoutVerification));
            _logger!.Log($"Track external inapp without validation. {nameof(productId)}={productId}; {nameof(price)}={price}; {nameof(currency)}={currency}; {nameof(transactionId)}={transactionId}; {nameof(originalTransactionId)}={originalTransactionId}, {nameof(store)}={store}");
            MagifyPlatformAPI!.TrackExternalInAppFor(productId, price, currency, transactionId, purchaseToken: null, originalTransactionId: originalTransactionId, receipt: null, needVerification: false, store);
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <summary>
        /// This method is used for the track initial verified in-app purchase that the Magify SDK can trust and not request verification on the Magify services side<br/>
        /// </summary>
        /// <param name="record">Magify expects this model to be created after the purchase has been verified and all data from the verification service is already present, which can be reliably displayed in monetization reports</param>
        /// <remarks>
        /// If user repeatedly buys consumable products - then all individual purchases will count for the initial purchase.
        /// </remarks>
        public static void TrackTrustedInAppFor([NotNull] TrustedPurchaseRecord record)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackTrustedInAppFor));
            _logger!.Log($"Track trusted inapp. {nameof(record)}={JsonFacade.SerializeObject(record)}");
            MagifyPlatformAPI!.TrackTrustedPurchase(record, isSubscription: false, isExternal: false);
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <summary>
        /// This method is used for the track initial external verified in-app purchase that the Magify SDK can trust and not request verification on the Magify services side<br/>
        /// </summary>
        /// <param name="record">Magify expects this model to be created after the purchase has been verified and all data from the verification service is already present, which can be reliably displayed in monetization reports</param>
        /// <remarks>
        /// If user repeatedly buys consumable products - then all individual purchases will count for the initial purchase.
        /// </remarks>
        public static void TrackExternalTrustedInAppFor([NotNull] TrustedPurchaseRecord record)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackExternalTrustedInAppFor));
            _logger!.Log($"Track external trusted inapp. {nameof(record)}={JsonFacade.SerializeObject(record)}");
            MagifyPlatformAPI!.TrackTrustedPurchase(record, isSubscription: false, isExternal: true);
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <summary>
        /// This method is used for the track initial verified subscription purchase that the Magify SDK can trust and not request verification on the Magify services side<br/>
        /// </summary>
        /// <param name="record">Magify expects this model to be created after the purchase has been verified and all data from the verification service is already present, which can be reliably displayed in monetization reports</param>
        /// <remarks>
        /// If one subscription was renewed, it doesn't count for the initial purchase.
        /// </remarks>
        public static void TrackTrustedSubscriptionActivation([NotNull] TrustedPurchaseRecord record)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackTrustedSubscriptionActivation));
            _logger!.Log($"Track trusted subscription activation. {nameof(record)}={JsonFacade.SerializeObject(record)}");
            MagifyPlatformAPI!.TrackTrustedPurchase(record, isSubscription: true, isExternal: false);
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <summary>
        /// This method is used for the track initial verified external subscription purchase that the Magify SDK can trust and not request verification on the Magify services side<br/>
        /// </summary>
        /// <param name="record">Magify expects this model to be created after the purchase has been verified and all data from the verification service is already present, which can be reliably displayed in monetization reports</param>
        /// <remarks>
        /// If one subscription was renewed, it doesn't count for the initial purchase.
        /// </remarks>
        public static void TrackExternalTrustedSubscriptionActivation([NotNull] TrustedPurchaseRecord record)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackExternalTrustedSubscriptionActivation));
            _logger!.Log($"Track external trusted subscription activation. {nameof(record)}={JsonFacade.SerializeObject(record)}");
            MagifyPlatformAPI!.TrackTrustedPurchase(record, isSubscription: true, isExternal: true);
        }

        /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
        /// <summary>
        /// This method is used for the track additional purchasing information about trusted purchases. For example, information about renewing or canceling a subscription, or to send refund information for an in-app purchase.
        /// </summary>
        /// <param name="record">Magify expects this model to be created after the purchase has been verified and all data from the verification service is already present, which can be reliably displayed in monetization reports</param>
        public static void TrackTrustedPurchase([NotNull] TrustedPurchaseRecord record)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackTrustedPurchase));
            _logger!.Log($"Track trusted purchase. {nameof(record)}={JsonFacade.SerializeObject(record)}");
            MagifyPlatformAPI!.TrackTrustedPurchase(record);
        }

        public static void TrackRestoredInAppFor([NotNull] string productId)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackRestoredInAppFor));
            _logger!.Log($"Track restored inapp. {nameof(productId)}={productId}");
            MagifyPlatformAPI!.TrackRestoredInAppFor(productId);
        }

        public static void TrackRestoredSubscription([NotNull] string productId)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackRestoredSubscription));
            _logger!.Log($"Track restored subscription. {nameof(productId)}={productId}");
            MagifyPlatformAPI!.TrackRestoredSubscription(productId);
        }

        public static void TrackProductClickFor(CampaignType campaignType, [NotNull] string productId, [CanBeNull] PurchaseStore? store = null)
        {
            switch (productId)
            {
                case ProductDef.FakeProductId:
                    TrackClickFor(campaignType);
                    break;
                default:
                    ThrowIfMagifyIsNotReady(nameof(TrackProductClickFor));
                    _logger!.Log($"Track product click. {nameof(campaignType)}={campaignType}; {nameof(productId)}={productId}");
                    MagifyPlatformAPI!.TrackProductClickFor(campaignType, productId, store);
                    break;
            }
        }

        public static void TrackAdsProductClickFor(CampaignType campaignType, [NotNull] string productId)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackAdsProductClickFor));
            _logger!.Log($"Track ads product click. {nameof(campaignType)}={campaignType}; {nameof(productId)}={productId}");
            MagifyPlatformAPI!.TrackAdsProductClickFor(campaignType, productId);
        }

        public static void TrackClickFor(CampaignType campaignType)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackClickFor));
            _logger!.Log($"Track click. {nameof(campaignType)}={campaignType}");
            MagifyPlatformAPI!.TrackClickFor(campaignType);
        }

        public static void TrackAdsClickFor(CampaignType campaignType)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackAdsClickFor));
            _logger!.Log($"Track ads click. {nameof(campaignType)}={campaignType}");
            MagifyPlatformAPI!.TrackAdsClickFor(campaignType);
        }

        public static void TrackImpressionFailFor(CampaignType campaignType, [CanBeNull] string reason)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackImpressionFailFor));
            _logger!.Log($"Track fail. {nameof(campaignType)}={campaignType}; {nameof(reason)}={reason}");
            MagifyPlatformAPI!.TrackImpressionFailFor(campaignType, reason);
        }

        public static void TrackAdsImpression(CampaignType campaignType, [NotNull] IAdsImpression impression)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackAdsImpression));
            _logger!.Log($"Track ads impression. {nameof(campaignType)}={campaignType}; {nameof(impression)}={JsonFacade.SerializeObject(impression)}");
            MagifyPlatformAPI!.TrackAdsImpression(campaignType, impression);
        }

        public static void TrackImpression(CampaignType campaignType)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackImpression));
            _logger!.Log($"Track impression. {nameof(campaignType)}={campaignType}");
            MagifyPlatformAPI!.TrackImpression(campaignType);
        }

        public static void TrackParentCampaignImpression(CampaignType campaignType)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackParentCampaignImpression));
            _logger!.Log($"Track parent campaign impression. {nameof(campaignType)}={campaignType}");
            MagifyPlatformAPI!.TrackParentCampaignImpression(campaignType);
        }

        public static void TrackProductsImpression(CampaignType campaignType, [NotNull, ItemNotNull] List<string> productIds)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackProductsImpression));
            productIds.RemoveAll(c => c == ProductDef.FakeProductId);
            if (productIds.Count == 0) return;
            _logger!.Log($"Track products impression. {nameof(campaignType)}={campaignType}; {nameof(productIds)}={JsonFacade.SerializeObject(productIds)}");
            MagifyPlatformAPI!.TrackProductsImpression(campaignType, productIds);
        }

        public static void TrackCustomEvent([NotNull] string eventName, [CanBeNull] IReadOnlyDictionary<string, object> customParams = null)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackCustomEvent));
            _logger!.Log($"Track custom event. {nameof(eventName)}={eventName}; {nameof(customParams)}={(customParams is null ? string.Empty : JsonFacade.SerializeObject(customParams))}");
            MagifyPlatformAPI!.TrackCustomEvent(eventName, customParams);
        }

        public static void TrackIncomeTransaction([NotNull] string source, [NotNull, ItemNotNull] List<BonusInfo> bonuses, [CanBeNull] ProductInfo product = null)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackIncomeTransaction));
            _logger!.Log($"Track income transaction. {nameof(source)}={source}; {nameof(bonuses)}={JsonFacade.SerializeObject(bonuses)}; {nameof(product)}={(product is null ? string.Empty : JsonFacade.SerializeObject(product))}");
            MagifyPlatformAPI!.TrackIncomeTransaction(source, bonuses, product);
        }

        public static void TrackExpenseTransaction([NotNull] List<BonusInfo> bonuses)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackExpenseTransaction));
            _logger!.Log($"Track expense transaction. {nameof(bonuses)}={JsonFacade.SerializeObject(bonuses)}");
            MagifyPlatformAPI!.TrackExpenseTransaction(bonuses);
        }

        public static void TrackCorrectionTransaction([NotNull] List<BonusInfo> bonuses)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackCorrectionTransaction));
            _logger!.Log($"Track correction transaction. {nameof(bonuses)}={JsonFacade.SerializeObject(bonuses)}");
            MagifyPlatformAPI!.TrackCorrectionTransaction(bonuses);
        }

        public static void TrackRewardGranted([NotNull] string productId)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackRewardGranted));
            _logger!.Log($"Track reward granted. {nameof(productId)}={productId}");
            MagifyPlatformAPI!.TrackRewardGranted(productId);
        }

        public static void TrackFreeBonusGranted([NotNull] string productId)
        {
            ThrowIfMagifyIsNotReady(nameof(TrackFreeBonusGranted));
            _logger!.Log($"Track free bonus granted. {nameof(productId)}={productId}");
            MagifyPlatformAPI!.TrackFreeBonusGranted(productId);
        }

        public static void TrackOrdinaryProductUsed([NotNull] string productId)
        {
            if (productId is ProductDef.FakeProductId) return;
            ThrowIfMagifyIsNotReady(nameof(TrackOrdinaryProductUsed));
            _logger!.Log($"Track ordinary product used. {nameof(productId)}={productId}");
            MagifyPlatformAPI!.TrackOrdinaryProductUsed(productId);
        }

        #endregion

        #region Debugging

        /// <exception cref="MagifyNotInitializedException">
        /// If you try to call this method before magify has been initialized.
        /// </exception>
        internal static void ThrowIfMagifyIsNotReady(string caller)
        {
            if (MagifyPlatformAPI == null)
            {
                throw new MagifyNotInitializedException(caller);
            }
        }

        #endregion
    }
}
