﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;
using Object = UnityEngine.Object;

namespace Magify
{
    public partial class MagifyManager
    {
        public static class Lto
        {
            public static event Action<LtoInfo> OnAdded;
            public static event Action<LtoInfo> OnUpdated;
            public static event Action<LtoInfo> OnRemoved;
            public static event Action<LtoInfo> OnFinished;

            private const string TextureLto = "Magify/LtoTextureView";
            private const string BundleLto = "Magify/LtoBundleView";
            private const string DefaultLtoNamePattern = "default_spot_{0}.png";

            [NotNull]
            private static readonly CompositeDisposable _disposables = new();
            [NotNull]
            private static readonly List<LtoInfo> _localOffers = new();
            [NotNull]
            private static readonly List<LtoInfo> _temp = new();
            [NotNull]
            private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Lto);

            //TODO: check if NotNull
            private static TaskScheduler _taskScheduler;
            private static uint _operationId;

            // ToDo: [NotNull]
            public static IReadOnlyCollection<LtoInfo> GetActiveLtoOffers()
            {
                ThrowIfMagifyIsNotReady($"{nameof(Lto)}.{nameof(GetActiveLtoOffers)}");
                if (MagifyPlatformAPI is IModernBridgePlatformAPI)
                {
                    return MagifyPlatformAPI.ActiveLtoOffers;
                }

                return _localOffers;
            }

            public static void CompleteActiveLto([NotNull] string offerName)
            {
                ThrowIfMagifyIsNotReady($"{nameof(Lto)}.{nameof(CompleteActiveLto)}");
                _logger.Log($"Complete active lto. {nameof(offerName)}={offerName};");
                MagifyPlatformAPI!.CompleteOffer(offerName);
            }

            internal static void Initialize([NotNull] TaskScheduler taskScheduler)
            {
                _taskScheduler = taskScheduler;
                _logger.Log("MagifyManager.LTO Initialize() called.");

                switch (MagifyPlatformAPI)
                {
                    case ILegacyBridgePlatformAPI legacy:
                        legacy.OnLtoDidUpdate += offers =>
                        {
                            var operationId = ++_operationId;
                            _logger.Log($"[ID:{operationId}] {nameof(OnLtoDidUpdate)} has been called with offers: {string.Join("; ", offers.Select(value => $"{value}"))}");
                            OnLtoDidUpdate(offers, operationId);
                        };
                        legacy.OnLtoDidFinish += offer =>
                        {
                            var operationId = ++_operationId;
                            _logger.Log($"[ID:{operationId}] {nameof(OnLtoDidFinish)} has been called with offer: {offer}");
                            OnLtoDidFinish(offer, operationId);
                        };
                        break;
                    case IModernBridgePlatformAPI modern:
                        Observable.FromEvent<LtoInfo>(t => modern.OnOfferAdded += t, t => modern.OnOfferAdded -= t)
                            .Subscribe(offer => OnLtoDidAdd(offer, ++_operationId))
                            .AddTo(_disposables);
                        Observable.FromEvent<LtoInfo>(t => modern.OnOfferUpdated += t, t => modern.OnOfferUpdated -= t)
                            .Subscribe(offer => OnLtoDidUpdate(offer, ++_operationId))
                            .AddTo(_disposables);
                        Observable.FromEvent<LtoInfo>(t => modern.OnOfferRemoved += t, t => modern.OnOfferRemoved -= t)
                            .Subscribe(offer => OnLtoDidRemove(offer, ++_operationId))
                            .AddTo(_disposables);
                        Observable.FromEvent<LtoInfo>(t => modern.OnOfferFinished += t, t => modern.OnOfferFinished -= t)
                            .Subscribe(offer => OnLtoDidFinish(offer, ++_operationId))
                            .AddTo(_disposables);
                        break;
                }
            }

            internal static void Setup()
            {
                _logger.Log("MagifyManager.LTO Setup() called.");
                _logger.Log("Getting active offers from sdk");
                var activeOffers = MagifyPlatformAPI.ActiveLtoOffers;
                _logger.Log($"Active offers amount is {activeOffers.Count}");
                if (activeOffers.Count > 0)
                {
                    var operationId = ++_operationId;
                    _logger.Log($"Emulate calling of ({nameof(OnLtoDidUpdate)})");
                    OnLtoDidUpdate(activeOffers, operationId);
                }
            }

            internal static void Reset()
            {
                _localOffers.Clear();
                _temp.Clear();
                _taskScheduler?.ClearQueue();
                _operationId = default;
            }

            private static void OnLtoDidAdd(LtoInfo offer, uint operationId)
            {
                _logger.Log($"[ID:{operationId}] SDK sent {nameof(OnLtoDidAdd)} with offer: {JsonFacade.SerializeObject(offer)}.");
                _taskScheduler.Enqueue(add, $"[{nameof(OnLtoDidAdd)}({operationId})]");

                void add()
                {
                    _logger.Log($"[ID:{operationId}] call event {nameof(OnAdded)} after {nameof(OnLtoDidAdd)} with offer: {JsonFacade.SerializeObject(offer)}.");
                    try
                    {
                        OnAdded?.Invoke(offer);
                    }
                    catch (Exception e)
                    {
                        _logger.LogError($"During {OnAdded} callback was thrown an exception: {e.Message}");
                        _logger.LogException(e);
                    }
                    _logger.Log($"[ID:{operationId}] event {nameof(OnAdded)} sent with offer: {JsonFacade.SerializeObject(offer)}.");
                }
            }

            private static void OnLtoDidUpdate(LtoInfo offer, uint operationId)
            {
                _logger.Log($"[ID:{operationId}] SDK sent {nameof(OnLtoDidUpdate)} with offer: {JsonFacade.SerializeObject(offer)}.");
                _taskScheduler.Enqueue(update, $"[{nameof(OnLtoDidUpdate)}({operationId})]");

                void update()
                {
                    _logger.Log($"[ID:{operationId}] call event {nameof(OnUpdated)} after {nameof(OnLtoDidUpdate)} with offer: {JsonFacade.SerializeObject(offer)}.");
                    try
                    {
                        OnUpdated?.Invoke(offer);
                    }
                    catch (Exception e)
                    {
                        _logger.LogError($"During {OnUpdated} callback was thrown an exception: {e.Message}");
                        _logger.LogException(e);
                    }
                    _logger.Log($"[ID:{operationId}] event {nameof(OnUpdated)} sent with offer: {JsonFacade.SerializeObject(offer)}.");
                }
            }

            private static void OnLtoDidRemove(LtoInfo offer, uint operationId)
            {
                _logger.Log($"[ID:{operationId}] SDK sent {nameof(OnLtoDidRemove)} with offer: {JsonFacade.SerializeObject(offer)}.");
                _taskScheduler.Enqueue(remove, $"[{nameof(OnLtoDidRemove)}({operationId})]");

                void remove()
                {
                    _logger.Log($"[ID:{operationId}] call event {nameof(OnRemoved)} after {nameof(OnLtoDidRemove)} with offer: {JsonFacade.SerializeObject(offer)}.");
                    try
                    {
                        OnRemoved?.Invoke(offer);
                    }
                    catch (Exception e)
                    {
                        _logger.LogError($"During {OnRemoved} callback was thrown an exception: {e.Message}");
                        _logger.LogException(e);
                    }
                    _logger.Log($"[ID:{operationId}] event {nameof(OnRemoved)} sent with offer: {JsonFacade.SerializeObject(offer)}.");
                }
            }

            private static void OnLtoDidFinish(LtoInfo offer, uint operationId)
            {
                _logger.Log($"[ID:{operationId}] SDK sent {nameof(OnLtoDidFinish)} with offer: {JsonFacade.SerializeObject(offer)}.");
                _taskScheduler.Enqueue(finish, $"[{nameof(OnLtoDidFinish)}({operationId})]");

                void finish()
                {
                    _logger.Log($"[ID:{operationId}] call event {nameof(OnFinished)} after {nameof(OnLtoDidFinish)} with offer: {JsonFacade.SerializeObject(offer)}.");
                    try
                    {
                        OnFinished?.Invoke(offer);
                    }
                    catch (Exception e)
                    {
                        _logger.LogError($"During {OnFinished} callback was thrown an exception: {e.Message}");
                        _logger.LogException(e);
                    }
                    _logger.Log($"[ID:{operationId}] event {nameof(OnFinished)} sent with offer: {JsonFacade.SerializeObject(offer)}.");
                }
            }

            private static void OnLtoDidUpdate(IReadOnlyCollection<LtoInfo> activeOffers, uint operationId)
            {
                _logger.Log($"[ID:{operationId}] SDK sent {nameof(OnLtoDidUpdate)}. ScheduleOrExecuteOnMain");
                _taskScheduler.Enqueue(update, $"[{nameof(OnLtoDidUpdate)}({operationId})]");
                _logger.Log($"[ID:{operationId}] {nameof(OnLtoDidUpdate)} finished");

                void update()
                {
                    _logger.Log($"[ID:{operationId}] {nameof(OnLtoDidUpdate)} has been called. Active offers:\n{string.Join("\n", activeOffers)}");
                    RemoveNonActiveOffers(activeOffers);

                    foreach (var rawOffer in activeOffers)
                    {
                        TryAddOffer(rawOffer, operationId);
                    }
                }
            }

            private static void TryAddOffer(LtoInfo rawOffer, uint operationId)
            {
                _logger.Log($"[ID:{operationId}] {nameof(TryAddOffer)} called for {rawOffer}");
                var existingLtoIndex = _localOffers.FindIndex(c => c.CampaignName == rawOffer.CampaignName);
                if (existingLtoIndex != -1)
                {
                    _logger.Log($"[ID:{operationId}] Offer for {rawOffer.CampaignName} already exists - take updates and call {nameof(OnUpdated)}");
                    var currentLto = _localOffers[existingLtoIndex];
                    if (currentLto.Spot != rawOffer.Spot)
                    {
                        _logger.LogError($"[ID:{operationId}] Updated offer for {rawOffer.CampaignName} has been changed relatively to existing lto. Ignore spot change");
                    }
                    _logger.Log($"[ID:{operationId}] Check is currentLto is equal to rawOffer.\n CurrentLto: {currentLto}\n RawOffer: {rawOffer}.");
                    if (!currentLto.Equals(rawOffer))
                    {
                        _logger.Log($"[ID:{operationId}] Take updates from rawOffer.");
                        currentLto.TakeUpdatesFrom(rawOffer);
                        try
                        {
                            OnUpdated?.Invoke(currentLto);
                        }
                        catch (Exception e)
                        {
                            _logger.LogError($"During {OnUpdated} callback was thrown an exception: {e.Message}");
                            _logger.LogException(e);
                        }
                    }
                    _logger.Log($"[ID:{operationId}] event {nameof(OnUpdated)} sent.");
                }
                else
                {
                    _logger.Log($"[ID:{operationId}] Add new offer {rawOffer}");
                    _localOffers.Add(rawOffer);
                    _logger.Log($"[ID:{operationId}] call event {nameof(OnAdded)} after {nameof(TryAddOffer)}. Has subscribers: {OnAdded != null}");
                    try
                    {
                        OnAdded?.Invoke(rawOffer);
                    }
                    catch (Exception e)
                    {
                        _logger.LogError($"During {OnAdded} callback was thrown an exception: {e.Message}");
                        _logger.LogException(e);
                    }
                    _logger.Log($"[ID:{operationId}] event {nameof(OnAdded)} sent.");
                }
            }

            private static void RemoveNonActiveOffers(IReadOnlyCollection<LtoInfo> activeOffers)
            {
                _temp.Clear();
                foreach (var offer in _localOffers)
                {
                    if (activeOffers.FirstOrDefault(c => c.CampaignName == offer.CampaignName) == null)
                    {
                        _temp.Add(offer);
                    }
                }

                if (_temp.Count == 0)
                {
                    return;
                }

                _logger.Log($"Found {_temp.Count} finished offers: {string.Join(", ", _temp.Select(c => c.Spot + $"({c.CampaignName})"))}");
                foreach (var offer in _temp)
                {
                    _localOffers.Remove(offer);
                    try
                    {
                        OnRemoved?.Invoke(offer);
                    }
                    catch (Exception e)
                    {
                        _logger.LogError($"During {OnRemoved} callback was thrown an exception: {e.Message}");
                        _logger.LogException(e);
                    }
                }
            }

            #region Get Lto creatives

            internal static async UniTask<LtoView> GetLto(LtoInfo offer, Transform parent, CancellationToken cancellationToken)
            {
                var resourceType = offer.BadgeCreative.Resource.Type;
                return resourceType switch
                {
                    CreativeResource.ArtifactType.Image => Get<LtoTextureView, Texture>(parent, TextureLto, await GetTexture(offer, cancellationToken)),
                    CreativeResource.ArtifactType.Bundle => Get<LtoBundleView, AssetBundle>(parent, BundleLto, await GetBundle(offer, cancellationToken)),
                    _ => throw new ArgumentOutOfRangeException(nameof(offer), $"Don't have LtoRenderer for {resourceType}")
                };
            }

            internal static LtoView GetDefaultLto(LtoInfo offer, Transform parent)
            {
                return Get<LtoTextureView, Texture>(parent, TextureLto, GetDefaultTexture(offer));
            }

            private static async UniTask<ContentHandle<Texture>> GetTexture(LtoInfo offer, CancellationToken cancellationToken)
            {
                return await Storage.LoadImage(offer.BadgeCreative.Resource.Url, Timeout.Infinite, cancellationToken);
            }

            private static async UniTask<ContentHandle<AssetBundle>> GetBundle(LtoInfo offer, CancellationToken cancellationToken)
            {
                return await Storage.LoadBundle(offer.BadgeCreative.Resource.Url, Timeout.Infinite, cancellationToken);
            }

            private static ContentHandle<Texture> GetDefaultTexture(LtoInfo offer)
            {
                return Storage.LoadImageFromStreamingAssets(string.Format(DefaultLtoNamePattern, offer.Spot));
            }

            private static LtoView Get<TView, TContent>(Transform parent, string path, ContentHandle<TContent> content)
                where TView : LtoView<TContent>
                where TContent : class
            {
                if (content.Value == null) return null;
                if (content.Code != StorageResultCode.Success)
                {
                    content.Dispose();
                    return null;
                }

                var viewPrefab = Resources.Load<TView>(path);
                if (viewPrefab == null)
                {
                    _logger.LogError($"Can't load lto view prefab of type {typeof(TView)} from path: {path}");
                    content.Dispose();
                    return null;
                }
                if (parent == null)
                {
                    _logger.Log($"Parent is null for lto view prefab of type {typeof(TView)} from path: {path}");
                    content.Dispose();
                    return null;
                }

                var view = Object.Instantiate(viewPrefab, parent);
                view.Content = content;
                view.PrepareAsset();

                return view;
            }

            #endregion
        }
    }
}