using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Xml;
using UnityEditor;
using UnityEngine;

namespace Magify
{
    internal class LinkXmlGenerator
    {
        private readonly Dictionary<Type, Type> _typeConversion = new Dictionary<Type, Type>();
        private readonly HashSet<Type> _types = new HashSet<Type>();

        public void AddType(Type type)
        {
            if (type == null) return;
            AddTypeInternal(type);
        }

        public void AddTypes(params Type[] types)
        {
            if (types == null) return;
            foreach (var t in types)
            {
                AddTypeInternal(t);
            }
        }

        public void AddTypes(IEnumerable<Type> types)
        {
            if (types == null) return;
            foreach (var t in types)
            {
                AddTypeInternal(t);
            }
        }

        private void AddTypeInternal(Type t)
        {
            if (t == null) return;
            _types.Add(_typeConversion.TryGetValue(t, out var convertedType) ? convertedType : t);
        }

        public void SetTypeConversion(Type a, Type b)
        {
            _typeConversion[a] = b;
        }

        public void AddAsset(string assetPath)
        {
            var assets = AssetDatabase.GetDependencies(assetPath);

            var types = new List<Type>();
            foreach (var asset in assets)
            {
                var type = AssetDatabase.GetMainAssetTypeAtPath(asset);
                if (type == typeof(GameObject))
                {
                    var obj = (GameObject)AssetDatabase.LoadAssetAtPath(asset, typeof(GameObject));
                    types.AddRange(obj.GetComponentsInChildren<Component>(true).Select(c => c.GetType()));
                }
                else
                {
                    types.Add(type);
                }
            }

            AddTypes(types);
        }

        public void AddAssets(IEnumerable<string> assetPaths)
        {
            foreach (var assetPath in assetPaths)
            {
                AddAsset(assetPath);
            }
        }

        public void Save(string path)
        {
            var assemblyMap = new Dictionary<Assembly, List<Type>>();
            foreach (var t in _types)
            {
                var a = t.Assembly;
                if (!assemblyMap.TryGetValue(a, out var types))
                {
                    assemblyMap.Add(a, types = new List<Type>());
                }
                types.Add(t);
            }

            var doc = new XmlDocument();
            var linker = doc.AppendChild(doc.CreateElement("linker"));
            foreach (var k in assemblyMap)
            {
                if (k.Key.FullName.Contains("UnityEditor")) continue;

                var assembly = linker.AppendChild(doc.CreateElement("assembly"));
                var attr = doc.CreateAttribute("fullname");
                attr.Value = k.Key.FullName;
                if (assembly.Attributes == null) continue;

                assembly.Attributes.Append(attr);
                foreach (var t in k.Value)
                {
                    var typeEl = assembly.AppendChild(doc.CreateElement("type"));
                    var tattr = doc.CreateAttribute("fullname");
                    tattr.Value = t.FullName;
                    if (typeEl.Attributes != null)
                    {
                        typeEl.Attributes.Append(tattr);
                        var pattr = doc.CreateAttribute("preserve");
                        pattr.Value = "all";
                        typeEl.Attributes.Append(pattr);
                    }
                }
            }

            doc.Save(path);
        }
    }
}