using System;
using System.Collections.Generic;
using System.Linq;
using UnityEditor;
using UnityEngine;

namespace Magify
{
    public class CreativeBuilderWindow : EditorWindow
    {
        [Flags]
        private enum BuildTarget : ushort
        {
            Mobile = 6, // Android + iOS
            Editor = 1,
            Android = 2,
            iOS = 4
        }

        private class CreativeData
        {
            public readonly GameObject Prefab;
            public bool IsNeedBuild;

            public CreativeData(GameObject prefab, bool isNeedBuild = false)
            {
                Prefab = prefab;
                IsNeedBuild = isNeedBuild;
            }
        }

        private readonly Dictionary<string, CreativeData> _creatives = new();
        private readonly HashSet<string> _checkedCreative = new();

        private bool _includeAllLanguagesInBundle;
        private BuildTarget _buildTarget = 0;
        private string _checkResult = string.Empty;
        private bool _wasCreativesCheck;
        private string _directoryPath = string.Empty;
        private string _filterText = string.Empty;
        private Dictionary<string, List<GameObject>> _duplicatesMap = new();

        private Vector2 _creativeScrollPosition;
        private GUIStyle _richLabelStyle;
        private GUIStyle _richHelpBoxStyle;

        private GUIStyle RichLabelStyle
        {
            get
            {
                return _richLabelStyle ??= new GUIStyle(EditorStyles.label)
                {
                    richText = true,
                    wordWrap = true
                };
            }
        }

        [MenuItem("Magify/Open Bundle Builder", priority = 1)]
        public static void ShowWindow()
        {
            GetWindow(typeof(CreativeBuilderWindow), false, "Creative Builder");
        }

        private void OnEnable()
        {
            _includeAllLanguagesInBundle = EditorPrefs.GetBool(CreativeBuilderConstants.IncludeAllLanguagesKey, false);
            _directoryPath = EditorPrefs.GetString(CreativeBuilderConstants.DirectoryPathKey, string.Empty);
            EditorApplication.projectChanged += OnProjectChange;
        }

        private void OnDisable()
        {
            EditorPrefs.SetBool(CreativeBuilderConstants.IncludeAllLanguagesKey, _includeAllLanguagesInBundle);
            EditorPrefs.SetString(CreativeBuilderConstants.DirectoryPathKey, _directoryPath);
            EditorApplication.projectChanged -= OnProjectChange;
        }

        private void OnGUI()
        {
            EditorGUILayout.LabelField("All creatives in project:", EditorStyles.boldLabel);
            DrawCreativesList();
            DrawDuplicates();
            DrawCheckCreatives();
            DrawBuildSettings();
        }

        private void OnFocus()
        {
            CollectCreatives();
        }

        private void DrawCreativesList()
        {
            _filterText = EditorGUILayout.TextField(_filterText, EditorStyles.toolbarSearchField);

            if (GUILayout.Button(string.IsNullOrEmpty(_directoryPath) ? "Select directory" : _directoryPath))
            {
                var path = EditorUtility.OpenFolderPanel("Creatives directory", _directoryPath, string.Empty);
                if (path.StartsWith(Application.dataPath))
                {
                    _directoryPath = "Assets" +  path.Substring(Application.dataPath.Length);
                    CollectCreatives();
                }
                else
                {
                    Debug.LogWarning("The selected path is not inside the project!");
                }
            }

            _creativeScrollPosition = EditorGUILayout.BeginScrollView(_creativeScrollPosition, "Box", GUILayout.ExpandHeight(false));

            foreach (var creativePair in _creatives
                .Where(t => t.Key.ToLower().Contains(_filterText.ToLower()))
                .OrderBy(t => t.Key))
            {
                var creative = creativePair.Value;
                if (!creative.Prefab)
                {
                    continue;
                }

                creative.IsNeedBuild = EditorGUILayout.ToggleLeft(creative.Prefab.name, creative.IsNeedBuild);
            }

            EditorGUILayout.EndScrollView();
            EditorGUILayout.Space();
        }

        private void DrawDuplicates()
        {
            if (_duplicatesMap.Count > 0)
            {
                EditorGUILayout.BeginVertical("Box");
                EditorGUILayout.Space();
                foreach (var pair in _duplicatesMap)
                {
                    var creativeName = pair.Key;

                    EditorGUILayout.LabelField($"The name <b>{creativeName}</b> used for more than one creative:", RichLabelStyle);
                    foreach (var creative in pair.Value)
                    {
                        EditorGUILayout.BeginHorizontal();
                        if (GUILayout.Button("Show", GUILayout.MaxWidth(45)))
                        {
                            EditorGUIUtility.PingObject(creative);
                        }
                        EditorGUILayout.LabelField(AssetDatabase.GetAssetPath(creative));
                        EditorGUILayout.EndHorizontal();
                    }

                    EditorGUILayout.Space();
                }
                EditorGUILayout.EndVertical();
                EditorGUILayout.Space();
            }
        }

        private void DrawCheckCreatives()
        {
            EditorGUI.BeginDisabledGroup(_duplicatesMap.Count > 0);
            if (GUILayout.Button("Check"))
            {
                _checkResult = CreativeBuilder.CheckCreatives(GetSelectedCreatives(), _includeAllLanguagesInBundle);
                _wasCreativesCheck = true;
            }

            if (_wasCreativesCheck)
            {
                var beginVertical = EditorGUILayout.BeginVertical("HelpBox");
                if (_checkResult.Length == 0)
                {
                    var iconContent = EditorGUIUtility.IconContent("greenLight");
                    iconContent.text = "With selected creatives all is fine!";
                    EditorGUILayout.LabelField(iconContent, GUILayout.Height(40));
                }
                else
                {
                    var iconContent = EditorGUIUtility.IconContent("redLight");
                    iconContent.text = "With selected creatives something wrong! See message below.";
                    EditorGUILayout.LabelField(iconContent, GUILayout.MinHeight(40), GUILayout.MinWidth(40));
                    EditorGUILayout.LabelField(_checkResult, RichLabelStyle, GUILayout.MaxWidth(beginVertical.width));
                }
                EditorGUILayout.EndVertical();
            }
            EditorGUI.EndDisabledGroup();
            EditorGUILayout.Space();
        }

        private void DrawBuildSettings()
        {
            _includeAllLanguagesInBundle = EditorGUILayout.Toggle("Include all languages in one bundle", _includeAllLanguagesInBundle);
            _buildTarget = (BuildTarget)EditorGUILayout.EnumFlagsField("Bundle build target: ", _buildTarget);

            EditorGUI.BeginDisabledGroup(_duplicatesMap.Count > 0);
            if (GUILayout.Button("Build"))
            {
                BuildBundle();
            }
            EditorGUI.EndDisabledGroup();
        }

        private void OnProjectChange()
        {
            UpdateDuplicateCreatives();
            _wasCreativesCheck = false;
        }

        private void CollectCreatives()
        {
            if (string.IsNullOrEmpty(_directoryPath))
            {
                return;
            }

            _duplicatesMap.Clear();
            _checkedCreative.Clear();

            var collectedCreatives = CreativeBuilder.CollectCreatives(_directoryPath);
            foreach (var creative in collectedCreatives)
            {
                _checkedCreative.Add(creative.name);
                if (_creatives.ContainsKey(creative.name))
                {
                    continue;
                }

                _creatives.Add(creative.name, new CreativeData(creative));
            }

            if (collectedCreatives.Count != _creatives.Count)
            {
                var missingCreatives = _creatives.Select(p => p.Key)
                    .Where(key => !_checkedCreative.Contains(key))
                    .ToList();
                foreach (var creativeName in missingCreatives)
                {
                    _creatives.Remove(creativeName);
                }
            }

            foreach (var creativePair in _creatives)
            {
                var creative = creativePair.Value;
                var prefabName = creative.Prefab.name;

                if (!_duplicatesMap.ContainsKey(prefabName))
                {
                    _duplicatesMap.Add(prefabName, new List<GameObject>());
                }

                _duplicatesMap[prefabName].Add(creative.Prefab);
            }

            _duplicatesMap = _duplicatesMap.Where(pair => pair.Value.Count > 1).ToDictionary(p => p.Key, p => p.Value);
        }

        private void UpdateDuplicateCreatives()
        {
            var redundantKeys = new List<string>();
            foreach (var duplicatePair in _duplicatesMap)
            {
                var duplicatePrefabs = duplicatePair.Value;
                var availablePrefabs = duplicatePrefabs.Where(p => p && duplicatePair.Key == p.name).ToList();
                if (availablePrefabs.Count > 1)
                {
                    duplicatePrefabs.Clear();
                    duplicatePrefabs.AddRange(availablePrefabs);
                }
                else
                {
                    redundantKeys.Add(duplicatePair.Key);
                }
            }

            foreach (var key in redundantKeys)
            {
                _duplicatesMap.Remove(key);
            }
        }

        private void BuildBundle()
        {
            Caching.ClearCache();

            if (_buildTarget == 0)
            {
                Debug.LogError("Choose build target!");
                return;
            }

            var creativesForBuild = GetSelectedCreatives();
            if (creativesForBuild.Count == 0)
            {
                Debug.LogError("Choose at least one creative for build!");
                return;
            }

            var buildCreatives = new HashSet<string>();
            foreach (var currentEnumFlag in _buildTarget.GetUniqueFlags())
            {
                var currentBuildTarget = (BuildTarget)currentEnumFlag;
                var unityBuildTarget = BuildTargetToUnityVersion(currentBuildTarget);
                var rootFolder = currentBuildTarget == BuildTarget.Editor ? "Editor" : unityBuildTarget.ToString();
                CreativeBuilder.BuildCreatives(unityBuildTarget, rootFolder, creativesForBuild, buildCreatives, _includeAllLanguagesInBundle);
            }
            CreativeBuilder.CreateLinkFile(buildCreatives);

            if (CreativeBuilder.HasError())
            {
                Debug.LogError(CreativeBuilder.GetErrorReport());
            }
        }

        private List<GameObject> GetSelectedCreatives()
        {
            return _creatives.Where(pair => pair.Value.IsNeedBuild)
                .Select(pair => pair.Value.Prefab)
                .ToList();
        }

        private UnityEditor.BuildTarget BuildTargetToUnityVersion(BuildTarget buildTarget)
        {
            switch (buildTarget)
            {
                case BuildTarget.Android:
                    return UnityEditor.BuildTarget.Android;
                case BuildTarget.iOS:
                    return UnityEditor.BuildTarget.iOS;
#if UNITY_EDITOR_WIN
                case BuildTarget.Editor:
                    return UnityEditor.BuildTarget.StandaloneWindows64;
#elif UNITY_EDITOR_OSX
                case BuildTarget.Editor:
                    return UnityEditor.BuildTarget.StandaloneOSX;
#elif UNITY_EDITOR_LINUX
                case BuildTarget.Editor:
                    return UnityEditor.BuildTarget.StandaloneLinux64;
#endif
                default:
                    Debug.LogError($"Platform {_buildTarget} don't supported!");
                    return UnityEditor.BuildTarget.NoTarget;
            }
        }
    }
}