﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using Newtonsoft.Json;
using UnityEditor;
using UnityEngine;
using Object = UnityEngine.Object;

#if UNITY_LOCALIZATION
using UnityEngine.Localization.Components;
using UnityEngine.Localization.Settings;
#endif

namespace Magify
{
    public static class CreativeBuilder
    {
        private const string AssetBundleRootDirectory = "Bundles";
        private static readonly StringBuilder _errorReport = new();

        public static void BuildPlatform(BuildTarget buildTarget, ISet<string> builtCreatives, bool includeAllLanguages, string searchInDirectory)
        {
            BuildPlatform(buildTarget, buildTarget.ToString(), builtCreatives, includeAllLanguages, searchInDirectory);
        }

        [MenuItem(PackageInfo.RootMenuItem + "Build Creatives Only For Editor (from Assets folder)")]
        public static void BuildAllAssetBundlesEditor()
        {
#if UNITY_EDITOR_WIN
            BuildPlatform(BuildTarget.StandaloneWindows64, "Editor", null, false, searchInDirectory: "Assets");
#elif UNITY_EDITOR_OSX
            BuildPlatform(BuildTarget.StandaloneOSX, "Editor", null, false, searchInDirectory: "Assets");
#elif UNITY_EDITOR_LINUX
            BuildPlatform(BuildTarget.StandaloneLinux64, "Editor", null, false, searchInDirectory: "Assets");
#endif
        }

        public static bool HasError() => _errorReport.Length > 0;

        public static string GetErrorReport() => _errorReport.ToString();

        public static void BuildCreatives(BuildTarget buildTarget, string rootFolder, List<GameObject> creatives, ISet<string> builtCreatives, bool includeAllLanguages)
        {
            _errorReport.Clear();

            var outputPath = Path.Combine(AssetBundleRootDirectory, rootFolder);
            if (Directory.Exists(outputPath))
            {
                Directory.Delete(outputPath, true);
            }
            Directory.CreateDirectory(outputPath);

            foreach (var creative in creatives)
            {
                var folderName = creative.name.ToLower();
                var path = Path.Combine(outputPath, folderName);
                Directory.CreateDirectory(path);

                var bundleLocalization = creative.GetComponent<BundleLocalization>();
                if (bundleLocalization != null && !includeAllLanguages)
                {
                    foreach (var locale in bundleLocalization.GetAvailableLocales())
                    {
                        buildBundle(creative, path, locale);
                    }
                }
                else
                {
                    buildBundle(creative, path, CreativeBuilderConstants.DefaultLanguageCodeKey);
                }
            }

            CleanupBundlesFolder(outputPath);

            void buildBundle(GameObject creative, string path, string cultureCode)
            {
                var assetPath = AssetDatabase.GetAssetPath(creative);
                try
                {
                    BuildCreative(creative, path, buildTarget, cultureCode, includeAllLanguages);
                    builtCreatives?.Add(assetPath);
                }
                catch (Exception e)
                {
                    _errorReport.Append(e);
                    _errorReport.AppendLine();
                }
            }
        }

        public static void BuildCreative(GameObject creativePrefab, string outputPath, BuildTarget buildTarget, string cultureCode, bool includeAllLanguages)
        {
            var tempCreativePath = CopyCreative(creativePrefab);
            var tempInformationPath = GetTempPathForMetadata();

            var bundles = new List<AssetBundleBuild>();
            var bundle = new AssetBundleBuild
            {
                assetBundleName = Guid.NewGuid() + ".bundle",
                assetNames = new[] { tempCreativePath, tempInformationPath }
            };

            try
            {
                // We need to open and disable all localization, because even after removing it, it can block text changes in TextMeshPro
                // We do this in a separate EditPrefabContentsScope, because we need to open and close the prefab
                // so that the localization is no longer affected
                using (var editingScope = new PrefabUtility.EditPrefabContentsScope(tempCreativePath))
                {
                    var bundleLocalization = editingScope.prefabContentsRoot.GetComponent<BundleLocalization>();
                    var availableLocales = bundleLocalization?.GetAvailableLocales();

                    if (bundleLocalization != null)
                    {
                        Object.DestroyImmediate(bundleLocalization, true);
                    }

                    DisableAllLocalizeScripts(editingScope.prefabContentsRoot);

                    BundleFontCustomizer.PrepareFonts(editingScope.prefabContentsRoot, cultureCode, availableLocales, includeAllLanguages);
                    RunAllCreativeProcessor(editingScope.prefabContentsRoot);
                    BundleFontCustomizer.OnPrepareFontsPostprocess(editingScope.prefabContentsRoot);

                    CreateBundleMetadata(availableLocales, tempInformationPath);
                }

                bundles.Add(bundle);
                BuildPipeline.BuildAssetBundles(outputPath, bundles.ToArray(), BuildAssetBundleOptions.AssetBundleStripUnityVersion, buildTarget);
                var oldPath = Path.Combine(outputPath, bundle.assetBundleName);
                var newPath = Path.Combine(outputPath, $"{cultureCode}.bundle");
                File.Move(oldPath, newPath);
            }
            finally
            {
                BundleFontCustomizer.RevertFontsSettings();
                AssetDatabase.DeleteAsset(tempCreativePath);
                AssetDatabase.DeleteAsset(tempInformationPath);
                AssetDatabase.SaveAssets();
            }
        }

        public static void CreateLinkFile(ISet<string> assets)
        {
            var generator = new LinkXmlGenerator();
            generator.AddAssets(assets);
            generator.AddAssets(assets.SelectMany(AssetDatabase.GetDependencies));
            generator.Save(Path.Combine(AssetBundleRootDirectory, "link.xml"));
        }

        public static string CheckCreatives(List<GameObject> creatives, bool includeAllLanguages)
        {
#if UNITY_LOCALIZATION
            LocalizationSettings.InitializationOperation.WaitForCompletion();
#endif

            var error = new StringBuilder();
            foreach (var creative in creatives)
            {
                var tempPath = string.Empty;
                try
                {
                    tempPath = CopyCreative(creative);
                    using var editingScope = new PrefabUtility.EditPrefabContentsScope(tempPath);
                    CheckFonts(editingScope.prefabContentsRoot, includeAllLanguages);
                }
                catch (Exception e)
                {
                    error.Append(e.Message);
                    error.AppendLine();
                }
                finally
                {
                    BundleFontCustomizer.RevertFontsSettings();
                    if (tempPath != string.Empty)
                    {
                        AssetDatabase.DeleteAsset(tempPath);
                        AssetDatabase.SaveAssets();
                    }
                }
            }

            return error.ToString();
        }

#if UNITY_LOCALIZATION
        private static void DisableAllLocalizeScripts(GameObject rootPrefab)
        {
            foreach (var localizeStringEvent in rootPrefab.GetComponentsInChildren<LocalizeStringEvent>())
            {
                localizeStringEvent.enabled = false;
            }
        }
#endif

        private static void BuildPlatform(BuildTarget buildTarget, string rootFolder, ISet<string> builtCreatives, bool includeAllLanguages, string searchInDirectory)
        {
            var creatives = CollectCreatives(searchInDirectory);
            var duplicates = creatives.GroupBy(x => x.name.ToLower())
                .Where(g => g.Count() > 1)
                .Select(y => y.Key)
                .ToList();

            foreach (var name in duplicates)
            {
                Debug.LogError($"The name {name} used for more than one creative:");
                foreach (var creative in creatives.Where(c => c.name == name))
                {
                    Debug.LogError(AssetDatabase.GetAssetPath(creative), creative);
                }
            }

            if (duplicates.Count > 0)
            {
                return;
            }

#if UNITY_LOCALIZATION
            LocalizationSettings.InitializationOperation.WaitForCompletion();
#endif
            BuildCreatives(buildTarget, rootFolder, creatives, builtCreatives, includeAllLanguages);

            if (HasError())
            {
                Debug.LogError(GetErrorReport());
            }
        }

        internal static List<GameObject> CollectCreatives(string directory)
        {
            Caching.ClearCache();

            var creatives = new List<GameObject>();
            foreach (var assetGuid in AssetDatabase.FindAssets("t:prefab", new[] { directory }))
            {
                var assetPath = AssetDatabase.GUIDToAssetPath(assetGuid);
                var prefab = AssetDatabase.LoadAssetAtPath<GameObject>(assetPath);
                var prefabAssetType = PrefabUtility.GetPrefabAssetType(prefab);
                if (!(prefabAssetType == PrefabAssetType.Regular || prefabAssetType == PrefabAssetType.Variant))
                {
                    continue;
                }

                var oldCreative = prefab.GetComponent<BundledCreative>();
                var creative = prefab.GetComponent<BundledPrefab>();
                if (oldCreative != null || creative != null)
                {
                    creatives.Add(prefab);
                }
            }

            return creatives;
        }

        private static void RunAllCreativeProcessor(GameObject creative)
        {
            var buildCreativeProcess = creative.GetComponents<CreativeBuildProcessor>();
            foreach (var creativeProcessor in buildCreativeProcess)
            {
                creativeProcessor.ProcessCreative();
                Object.DestroyImmediate(creativeProcessor, true);
            }
        }

        private static void CreateBundleMetadata(List<string> availableLanguages, string path)
        {
            var jsonData = JsonConvert.SerializeObject(availableLanguages);
            File.WriteAllText(path, jsonData);
            AssetDatabase.Refresh();
        }

        private static void CleanupBundlesFolder(string path)
        {
            foreach (var file in Directory.GetFiles(path, "*", SearchOption.AllDirectories))
            {
                if (Path.GetFileNameWithoutExtension(file) == Path.GetFileName(Path.GetDirectoryName(file)) ||
                    Path.GetExtension(file).ToLower() == ".manifest")
                {
                    File.Delete(file);
                }
            }
        }

        private static void CheckFonts(GameObject creative, bool includeAllLanguages)
        {
            var bundleLocalization = creative.GetComponent<BundleLocalization>();
            if (bundleLocalization != null)
            {
                var availableLocales = bundleLocalization.GetAvailableLocales();

                foreach (var locale in availableLocales)
                {
                    BundleFontCustomizer.PrepareFonts(creative, locale, availableLocales, includeAllLanguages);
                    BundleFontCustomizer.RevertFontsSettings();
                }
            }
            else
            {
                BundleFontCustomizer.PrepareFonts(creative, CreativeBuilderConstants.DefaultLanguageCodeKey, null, includeAllLanguages);
                BundleFontCustomizer.RevertFontsSettings();
            }
        }

        private static string GetTempPathForCreative(string originalPath)
        {
            return Path.Combine("Assets", Path.GetFileName(originalPath));
        }

        private static string GetTempPathForMetadata()
        {
            return Path.Combine("Assets", CreativeBuilderConstants.MetadataFileName);
        }

        private static string CopyCreative(GameObject creative)
        {
            var originalPath = AssetDatabase.GetAssetPath(creative);
            var tempPath = GetTempPathForCreative(originalPath);
            var isCopied = AssetDatabase.CopyAsset(originalPath, tempPath);
            if (!isCopied)
            {
                throw new Exception($"Can't copy creative from {originalPath} to {tempPath}");
            }
            return tempPath;
        }
    }
}