using System.Linq;
using System.Collections.Generic;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Rx;
using NUnit.Framework;

namespace Magify.Tests
{
    internal class ObserveOnMainThreadTests
    {
        [Test]
        public async Task WhenObserveOnMainThread_AndNextOnThreadPool_ThenCallbackOnMainThread()
        {
            // Arrange
            var cancellationToken = Utils.GetCancellationToken();
            var isMainThread = default(bool?);
            using var subject = new Subject<Unit>();
            using var _ = subject.ObserveOnMainThread(cancellationToken).Subscribe(_ => isMainThread = TaskScheduler.IsMainThread);

            // Act
            await UniTask.RunOnThreadPool(() => subject.OnNext(Unit.Default), cancellationToken: cancellationToken);

            // Assert
            isMainThread?.Should()!.BeTrue();
        }

        [Test]
        [TestCase(2)]
        [TestCase(10)]
        [TestCase(100)]
        public async Task WhenMultipleObserveOnMainThread_AndNextOnThreadPool_ThenCallbackOnMainThread(int observers)
        {
            // Arrange
            var cancellationToken = Utils.GetCancellationToken();
            var isMainThread = new bool?[observers];
            using var subject = new Subject<Unit>();
            using var disposables = new CompositeDisposable(observers);
            observers.EnumerateEach().ForEach(i =>
                    subject.ObserveOnMainThread(cancellationToken)
                           .Subscribe(_ => isMainThread[i] = TaskScheduler.IsMainThread)
                           .AddTo(disposables));

            // Act
            await UniTask.RunOnThreadPool(() => subject.OnNext(Unit.Default), cancellationToken: cancellationToken);

            // Assert
            isMainThread.Should().AllBeEquivalentTo(true);
            isMainThread.Count(v => v is true).Should().Be(observers);
        }

        [Test]
        public async Task WhenDirectObserve_AndObserveOnMainThread_AndNextOnThreadPool_ThenCallbackOnMainThread()
        {
            // Arrange
            var cancellationToken = Utils.GetCancellationToken();
            var firstIsOnMainThread = default(bool?);
            var secondIsOnThreadPool = default(bool?);
            using var subject = new Subject<Unit>();
            using var _1 = subject.ObserveOnMainThread(cancellationToken).Subscribe(_ => firstIsOnMainThread = TaskScheduler.IsMainThread);
            using var _2 = subject.Subscribe(_ => secondIsOnThreadPool = TaskScheduler.IsMainThread);

            // Act
            await UniTask.RunOnThreadPool(() => subject.OnNext(Unit.Default), cancellationToken: cancellationToken);

            // Assert
            firstIsOnMainThread?.Should()!.BeTrue();
            secondIsOnThreadPool?.Should()!.BeFalse();
        }

        [Test]
        [TestCase(2)]
        [TestCase(10)]
        [TestCase(100)]
        public async Task WhenObserveOnMainThread_AndManyNextOnThreadPool_ThenManyCallbacksOnMainThread(int repeats)
        {
            // Arrange
            var cancellationToken = Utils.GetCancellationToken();
            var isMainThread = new List<bool?>();
            using var subject = new Subject<Unit>();
            using var _ = subject.ObserveOnMainThread(cancellationToken).Subscribe(_ => isMainThread.Add(TaskScheduler.IsMainThread));

            // Act
            var tasks = repeats.EnumerateEach().Select(_ => UniTask.RunOnThreadPool(() => subject.OnNext(Unit.Default), cancellationToken: cancellationToken));
            await UniTask.WhenAll(tasks);
            await UniTask.SwitchToMainThread();

            // Assert
            isMainThread.Count.Should().Be(repeats);
            isMainThread.Should().AllBeEquivalentTo(true);
        }
    }
}